package com.biz.crm.mdm.business.terminal.local.service.internal;

import com.biz.crm.mdm.business.terminal.local.entity.TerminalRelaCustomerOrg;
import com.biz.crm.mdm.business.terminal.local.repository.TerminalRelaCustomerOrgRepository;
import com.biz.crm.mdm.business.terminal.local.service.TerminalRelaCustomerOrgService;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalRebindCustomerOrgDto;
import com.biz.crm.mdm.business.terminal.sdk.event.TerminalRCustomerOrgEventListener;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 终端与客户组织关联
 *
 * @author sunx
 * @since 2021-10-18 17:48:34
 */
@Service("terminalRelaCustomerOrgService")
public class TerminalRelaCustomerOrgServiceImpl implements TerminalRelaCustomerOrgService {

  @Autowired(required = false) private TerminalRelaCustomerOrgRepository terminalRelaCustomerOrgRepository;

  @Autowired(required = false)
  private List<TerminalRCustomerOrgEventListener> terminalRCustomerOrgEventListeners;

  @Override
  public List<TerminalRelaCustomerOrg> findByTerminalCodes(List<String> terminalCodeList) {
    if (CollectionUtils.isEmpty(terminalCodeList)) {
      return Lists.newLinkedList();
    }
    return terminalRelaCustomerOrgRepository.findByTerminalCodes(terminalCodeList);
  }

  @Override
  @Transactional
  public void saveBatch(List<TerminalRelaCustomerOrg> list, String terminalCode) {
    Validate.notBlank(terminalCode, "终端编码信息不能为空");
    terminalRelaCustomerOrgRepository.deleteByTerminalCodes(Lists.newArrayList(terminalCode));
    if (CollectionUtils.isEmpty(list)) {
      return;
    }

    Optional<TerminalRelaCustomerOrg> first =
        list.stream()
            .filter(
                a ->
                    StringUtils.isBlank(a.getTerminalCode())
                        || StringUtils.isBlank(a.getOrgCode())
                        || !terminalCode.equals(a.getTerminalCode()))
            .findFirst();
    Validate.isTrue(!first.isPresent(), "终端编码或客户组织编码不能为空,且必须属于同一终端");
    Map<String, List<TerminalRelaCustomerOrg>> map =
        list.stream().collect(Collectors.groupingBy(TerminalRelaCustomerOrg::getOrgCode));
    map.entrySet().forEach(a -> Validate.isTrue(a.getValue().size() <= 1, "存在重复的组织信息"));

    for (TerminalRelaCustomerOrg terminalRelaCustomerOrg : list) {
      terminalRelaCustomerOrg.setId(null);
      terminalRelaCustomerOrg.setTenantCode(TenantUtils.getTenantCode());
    }
    terminalRelaCustomerOrgRepository.saveBatch(list);
  }

  @Override
  @Transactional
  public void rebindOrg(TerminalRebindCustomerOrgDto dto) {
    Validate.notNull(dto, "参数不能为空");
    Validate.notBlank(dto.getNewOrgCode(), "更换后的组织信息不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getTerminalCodeList()), "终端信息不能为空");
    List<String> deleteOrgCodeList = Lists.newArrayList(dto.getNewOrgCode());
    if (StringUtils.isNotBlank(dto.getOldOrgCode())) {
      deleteOrgCodeList.add(dto.getOldOrgCode());
    }
    terminalRelaCustomerOrgRepository.deleteByOrgCodesAndTerminalCodes(
        deleteOrgCodeList, dto.getTerminalCodeList());

    List<TerminalRelaCustomerOrg> list = Lists.newLinkedList();
    Set<String> set = Sets.newLinkedHashSet(dto.getTerminalCodeList());
    for (String terminalCode : set) {
      TerminalRelaCustomerOrg cur = new TerminalRelaCustomerOrg();
      cur.setId(null);
      cur.setTenantCode(TenantUtils.getTenantCode());
      cur.setTerminalCode(terminalCode);
      cur.setOrgCode(dto.getNewOrgCode());
      list.add(cur);
    }
    terminalRelaCustomerOrgRepository.saveBatch(list);

    // 发送重新绑定组织事件通知
    if (CollectionUtils.isEmpty(terminalRCustomerOrgEventListeners)) {
      return;
    }
    for (TerminalRCustomerOrgEventListener terminalRCustomerOrgEventListener :
        terminalRCustomerOrgEventListeners) {
      terminalRCustomerOrgEventListener.onRebind(
          dto.getOldOrgCode(), dto.getNewOrgCode(), dto.getTerminalCodeList());
    }
  }

  /**
   * 根据终端编码删除信息
   *
   * @param terminalCodes
   */
  @Override
  @Transactional
  public void deleteByTerminalCodes(List<String> terminalCodes) {
    if (CollectionUtils.isEmpty(terminalCodes)) {
      return;
    }
    this.terminalRelaCustomerOrgRepository.deleteByTerminalCodes(terminalCodes);
  }

  @Override
  @Transactional
  public void create(String customerOrgCode, String terminalCode, String tenantCode) {
    Validate.notBlank(terminalCode, "终端编码信息不能为空");
    this.terminalRelaCustomerOrgRepository.deleteByTerminalCodes(Lists.newArrayList(terminalCode));
    if (StringUtils.isBlank(customerOrgCode)) {
      return;
    }
    TerminalRelaCustomerOrg terminalRelaCustomerOrg = new TerminalRelaCustomerOrg();
    terminalRelaCustomerOrg.setTerminalCode(terminalCode);
    terminalRelaCustomerOrg.setOrgCode(customerOrgCode);
    terminalRelaCustomerOrg.setTenantCode(tenantCode);
    this.terminalRelaCustomerOrgRepository.save(terminalRelaCustomerOrg);
  }
}
