package com.biz.crm.mdm.business.terminal.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.biz.crm.mdm.business.terminal.local.entity.Terminal;
import com.biz.crm.mdm.business.terminal.local.entity.TerminalContact;
import com.biz.crm.mdm.business.terminal.local.entity.TerminalRelaCustomerOrg;
import com.biz.crm.mdm.business.terminal.local.entity.TerminalRelaOrg;
import com.biz.crm.mdm.business.terminal.local.entity.TerminalTag;
import com.biz.crm.mdm.business.terminal.local.repository.TerminalRepository;
import com.biz.crm.mdm.business.terminal.local.service.TerminalContactService;
import com.biz.crm.mdm.business.terminal.local.service.TerminalRelaCustomerOrgService;
import com.biz.crm.mdm.business.terminal.local.service.TerminalRelaOrgService;
import com.biz.crm.mdm.business.terminal.local.service.TerminalService;
import com.biz.crm.mdm.business.terminal.local.service.TerminalTagService;
import com.biz.crm.mdm.business.terminal.sdk.constant.TerminalConstant;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalCodeSearchDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalEventDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalPaginationDto;
import com.biz.crm.mdm.business.terminal.sdk.event.TerminalEventListener;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalSupplyVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalRelaOrgVo;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 终端信息(Terminal)表服务实现类
 *
 * @author sunx
 * @since 2021-10-18 17:46:51
 */
@Service("terminalService")
public class TerminalServiceImpl implements TerminalService {

  @Autowired(required = false)
  private TerminalRepository terminalRepository;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  private TerminalRelaCustomerOrgService terminalRelaCustomerOrgService;

  @Autowired(required = false)
  private TerminalRelaOrgService terminalRelaOrgService;

  @Autowired(required = false)
  private OrgVoService orgVoService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private TerminalContactService terminalContactService;

  @Autowired(required = false)
  private TerminalSupplyVoService terminalSupplyVoService;

  @Autowired(required = false)
  private TerminalTagService terminalTagService;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Override
  public Page<Terminal> findByConditions(Pageable pageable, TerminalPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new TerminalPaginationDto());
    Page<Terminal> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<Terminal> byConditions = this.terminalRepository.findByConditions(page, dto);
    List<Terminal> records = byConditions.getRecords();
    if (CollectionUtils.isEmpty(records)) {
      return byConditions;
    }
    // 查询组织和客户组织
    List<String> orgCodes = records.stream().map(Terminal::getOrgCode).collect(Collectors.toList());
    List<OrgVo> orgVos = this.orgVoService.findByOrgCodes(orgCodes);
    List<String> terminalCodes =
        records.stream().map(Terminal::getTerminalCode).collect(Collectors.toList());
    List<TerminalRelaCustomerOrg> customerOrgList =
        this.terminalRelaCustomerOrgService.findByTerminalCodes(terminalCodes);
    List<TerminalRelaOrg> orgList = terminalRelaOrgService.findByTerminalCodes(terminalCodes);
    List<TerminalTag> terminalTagList =
        this.terminalTagService.findByTerminalCodes(Sets.newHashSet(terminalCodes));
    // 转换
    Map<String, TerminalRelaCustomerOrg> customerOrgMap = Maps.newHashMap();
    Map<String, TerminalRelaOrg> orgMap = Maps.newHashMap();
    Map<String, List<TerminalTag>> tagMap = Maps.newHashMap();
    Map<String, OrgVo> orgVoMap =
        Optional.ofNullable(orgVos).orElse(Lists.newArrayList()).stream()
            .collect(Collectors.toMap(OrgVo::getOrgCode, Function.identity()));
    if (CollectionUtils.isNotEmpty(customerOrgList)) {
      customerOrgMap =
          customerOrgList.stream()
              .filter(o -> StringUtils.isNotEmpty(o.getTerminalCode()))
              .collect(
                  Collectors.toMap(TerminalRelaCustomerOrg::getTerminalCode, v -> v, (a, b) -> a));
    }
    if (CollectionUtils.isNotEmpty(orgList)) {
      orgMap =
          orgList.stream()
              .filter(o -> StringUtils.isNotEmpty(o.getTerminalCode()))
              .collect(Collectors.toMap(TerminalRelaOrg::getTerminalCode, v -> v, (a, b) -> a));
    }
    if (CollectionUtils.isNotEmpty(terminalTagList)) {
      tagMap =
          terminalTagList.stream()
              .filter(a -> StringUtils.isNotBlank(a.getTerminalCode()))
              .collect(Collectors.groupingBy(TerminalTag::getTerminalCode));
    }
    // 查询联系人信息
    List<TerminalContact> terminalContacts =
        this.terminalContactService.findByTerminalCodes(terminalCodes);
    Map<String, List<TerminalContact>> listMap = new HashMap<>();
    if (!CollectionUtils.isEmpty(terminalContacts)) {
      listMap =
          terminalContacts.stream()
              .collect(Collectors.groupingBy(TerminalContact::getTerminalCode));
    }
    // 组装 组织和客户组织数据
    for (Terminal terminal : records) {
      OrgVo orgVo = orgVoMap.get(terminal.getOrgCode());
      if (orgVo != null) {
        terminal.setOrgName(orgVo.getOrgName());
      }
      if (customerOrgMap.containsKey(terminal.getTerminalCode())) {
        terminal.setCustomerOrg(customerOrgMap.get(terminal.getTerminalCode()));
      }
      if (orgMap.containsKey(terminal.getTerminalCode())) {
        terminal.setOrg(orgMap.get(terminal.getTerminalCode()));
      }
      terminal.setTagList(tagMap.get(terminal.getTerminalCode()));
      if (CollectionUtils.isNotEmpty(terminal.getTagList())) {
        terminal.setTagStr(
            terminal.getTagList().stream()
                .filter(a -> StringUtils.isNotBlank(a.getTagDescription()))
                .map(TerminalTag::getTagDescription)
                .collect(Collectors.joining(",")));
      }
      terminal.setContactList(listMap.get(terminal.getTerminalCode()));
    }
    return byConditions;
  }

  @Override
  public Terminal findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.terminalRepository.findById(id);
  }

  @Override
  @Transactional
  public Terminal create(Terminal terminal) {
    Validate.notNull(terminal, "终端信息缺失");
    Validate.isTrue(StringUtils.isNotBlank(terminal.getTerminalName()), "终端名称不能为空");
    // 如果userCode为空需要期初一个编码，否则执行验重逻辑
    if (StringUtils.isEmpty(terminal.getTerminalCode())) {
      terminal.setTerminalCode(
          generateCodeService.generateCode(TerminalConstant.TERMINAL_CODE, 1).get(0));
    } else {
      Integer count = this.terminalRepository.countByTerminalCode(terminal.getTerminalCode());
      Validate.isTrue(null == count || 1 > count, terminal.getTerminalCode() + "终端编码已存在");
    }
    terminal.setTenantCode(TenantUtils.getTenantCode());
    terminal.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    terminal.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.terminalRepository.saveOrUpdate(terminal);
    TerminalVo vo =
        this.nebulaToolkitService.copyObjectByBlankList(
            terminal, TerminalVo.class, HashSet.class, ArrayList.class);
    TerminalEventDto eventDto = new TerminalEventDto();
    eventDto.setOriginal(null);
    eventDto.setNewest(vo);
    SerializableBiConsumer<TerminalEventListener, TerminalEventDto> onCreate =
        TerminalEventListener::onCreate;
    this.nebulaNetEventClient.publish(eventDto, TerminalEventListener.class, onCreate);
    return terminal;
  }

  @Override
  @Transactional
  public Terminal update(Terminal terminal) {
    Validate.notNull(terminal, "终端信息缺失");
    Validate.isTrue(StringUtils.isNotBlank(terminal.getId()), "终端id不能为空");
    String currentId = terminal.getId();
    Terminal current = this.terminalRepository.findById(currentId);
    Validate.notNull(current, "修改信息不存在");
    Validate.isTrue(terminal.getTerminalCode().equals(current.getTerminalCode()), "编码不能修改");
    this.terminalRepository.saveOrUpdate(terminal);
    TerminalVo vo1 =
        this.nebulaToolkitService.copyObjectByBlankList(
            current, TerminalVo.class, HashSet.class, ArrayList.class);
    TerminalVo vo2 =
        this.nebulaToolkitService.copyObjectByBlankList(
            terminal, TerminalVo.class, HashSet.class, ArrayList.class);
    final Map<String, List<TerminalRelaOrgVo>> map =
        this.findTerminalOrgMap(Lists.newArrayList(current));
    vo2.setOrgList(map.get(terminal.getTerminalCode()));
    TerminalEventDto eventDto = new TerminalEventDto();
    eventDto.setOriginal(vo1);
    eventDto.setNewest(vo2);
    SerializableBiConsumer<TerminalEventListener, TerminalEventDto> onUpdate =
        TerminalEventListener::onUpdate;
    this.nebulaNetEventClient.publish(eventDto, TerminalEventListener.class, onUpdate);
    return terminal;
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.terminalRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    final List<Terminal> list = this.terminalRepository.findByIds(ids);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    Map<String, List<TerminalRelaOrgVo>> map = this.findTerminalOrgMap(list);
    List<TerminalVo> voList =
        (List<TerminalVo>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                list, Terminal.class, TerminalVo.class, HashSet.class, ArrayList.class);
    for (TerminalVo vo : voList) {
      TerminalEventDto eventDto = new TerminalEventDto();
      vo.setOrgList(map.get(vo.getTerminalCode()));
      eventDto.setNewest(vo);
      SerializableBiConsumer<TerminalEventListener, TerminalEventDto> onEnable =
          TerminalEventListener::onEnable;
      this.nebulaNetEventClient.publish(eventDto, TerminalEventListener.class, onEnable);
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.terminalRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    final List<Terminal> list = this.terminalRepository.findByIds(ids);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    List<TerminalVo> voList =
        (List<TerminalVo>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                list, Terminal.class, TerminalVo.class, HashSet.class, ArrayList.class);
    for (TerminalVo vo : voList) {
      TerminalEventDto eventDto = new TerminalEventDto();
      eventDto.setNewest(vo);
      SerializableBiConsumer<TerminalEventListener, TerminalEventDto> onDisable =
          TerminalEventListener::onDisable;
      this.nebulaNetEventClient.publish(eventDto, TerminalEventListener.class, onDisable);
    }
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<Terminal> terminalList = this.terminalRepository.findByIds(ids);
    if (CollectionUtils.isEmpty(terminalList)) {
      return;
    }
    this.terminalRepository.updateDelFlagByIds(ids);
    List<String> terminalCodes =
        terminalList.stream()
            .filter(terminal -> StringUtils.isNotBlank(terminal.getTerminalCode()))
            .map(Terminal::getTerminalCode)
            .collect(Collectors.toList());
    Map<String, Set<String>> map = Maps.newHashMap();
    if (CollectionUtils.isNotEmpty(terminalCodes)) {
      final List<TerminalRelaOrg> terminalOrgList =
          this.terminalRelaOrgService.findByTerminalCodes(terminalCodes);
      if (CollectionUtils.isNotEmpty(terminalOrgList)) {
        map =
            terminalOrgList.stream()
                .filter(a -> StringUtils.isNoneBlank(a.getTerminalCode(), a.getOrgCode()))
                .collect(
                    Collectors.groupingBy(
                        TerminalRelaOrg::getTerminalCode,
                        Collectors.mapping(TerminalRelaOrg::getOrgCode, Collectors.toSet())));
      }
    }
    this.deleteExtInfo(terminalCodes);
    List<TerminalVo> voList =
        (List<TerminalVo>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                terminalList, Terminal.class, TerminalVo.class, HashSet.class, ArrayList.class);
    for (TerminalVo vo : voList) {
      Set<String> curSet = map.get(vo.getTerminalCode());
      if (CollectionUtils.isNotEmpty(curSet)) {
        List<TerminalRelaOrgVo> curOrgList = Lists.newLinkedList();
        for (String orgCode : curSet) {
          final TerminalRelaOrgVo cur = new TerminalRelaOrgVo();
          cur.setOrgCode(orgCode);
          curOrgList.add(cur);
        }
        vo.setOrgList(curOrgList);
      }
      TerminalEventDto eventDto = new TerminalEventDto();
      eventDto.setOriginal(vo);
      SerializableBiConsumer<TerminalEventListener, TerminalEventDto> onDelete =
          TerminalEventListener::onDelete;
      this.nebulaNetEventClient.publish(eventDto, TerminalEventListener.class, onDelete);
    }
  }

  /**
   * 根据条件获取匹配的终端信息
   *
   * @param ids
   * @param terminalCodes
   * @return
   */
  @Override
  public List<Terminal> findDetailsByIdsOrTerminalCodes(
      List<String> ids, List<String> terminalCodes) {
    if (CollectionUtils.isEmpty(ids) && CollectionUtils.isEmpty(terminalCodes)) {
      return Lists.newLinkedList();
    }
    return this.terminalRepository.findDetailsByIdsOrTerminalCodes(ids, terminalCodes);
  }

  /**
   * 根据终端编码集合查询
   *
   * @param terminalCodes
   * @return
   */
  @Override
  public List<Terminal> findByTerminalCodes(List<String> terminalCodes) {
    if (CollectionUtils.isEmpty(terminalCodes)) {
      return new ArrayList<>(0);
    }
    return this.terminalRepository.findByTerminalCodes(terminalCodes);
  }

  /**
   * 根据客户组织编码集合查询
   *
   * @param customerOrgCodes
   * @return
   */
  @Override
  public List<Terminal> findByCustomerOrgCodes(List<String> customerOrgCodes) {
    if (CollectionUtils.isEmpty(customerOrgCodes)) {
      return new ArrayList<>(0);
    }

    return this.terminalRepository.findByCustomerOrgCodes(
        customerOrgCodes, TenantUtils.getTenantCode(), DelFlagStatusEnum.NORMAL.getCode());
  }

  @Override
  public Set<String> findByTerminalCodeSearchDto(TerminalCodeSearchDto dto) {
    dto = Optional.ofNullable(dto).orElse(new TerminalCodeSearchDto());
    if (CollectionUtils.isEmpty(dto.getOrgCodeSet())
        && CollectionUtils.isEmpty(dto.getChannelSet())
        && CollectionUtils.isEmpty(dto.getTagSet())) {
      return Sets.newHashSet();
    }
    return this.terminalRepository.findByTerminalCodeSearchDto(dto);
  }

  @Override
  public Terminal findByTerminalCode(String terminalCode) {
    if (StringUtils.isBlank(terminalCode)) {
      return null;
    }
    List<Terminal> list =
        this.terminalRepository.findByTerminalCodes(Lists.newArrayList(terminalCode));
    if (CollectionUtils.isEmpty(list)) {
      return null;
    }

    return list.get(0);
  }

  @Override
  public Terminal findByProcessNumber(String processNumber) {
    if (StringUtils.isBlank(processNumber)) {
      return null;
    }
    return this.terminalRepository.findByProcessNumber(processNumber);
  }

  @Override
  @Transactional
  public void updateByProcess(Terminal terminal) {
    Validate.notNull(terminal, "终端信息不能为空");
    Validate.notBlank(terminal.getId(), "终端id不能为空");
    Validate.notBlank(terminal.getProcessStatus(), "终端审批状态不能为空");
    this.terminalRepository.updateById(terminal);
  }

  /**
   * 删除扩展信息
   *
   * @param terminalCodes
   */
  private void deleteExtInfo(List<String> terminalCodes) {
    /*
     * 删除终端关联信息：
     * 1、删除关联组织信息
     * 2、删除关联联系人信息
     * 3、删除关联供货关系信息
     * */
    Validate.isTrue(!CollectionUtils.isEmpty(terminalCodes), "删除关联信息时缺失编码");
    terminalRelaOrgService.deleteByTerminalCodes(terminalCodes);
    terminalRelaCustomerOrgService.deleteByTerminalCodes(terminalCodes);
    terminalContactService.deleteByTerminalCodes(terminalCodes);
    terminalSupplyVoService.deleteByTerminalCodes(terminalCodes);
  }

  /**
   * 只有主表信息，不包含扩展信息
   *
   * @param ids
   * @return
   */
  private List<TerminalVo> findVoListByIds(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return Lists.newLinkedList();
    }
    List<Terminal> list = this.findDetailsByIdsOrTerminalCodes(ids, null);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newLinkedList();
    }
    return (List<TerminalVo>)
        this.nebulaToolkitService.copyCollectionByWhiteList(
            list, Terminal.class, TerminalVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 获取终端对应的组织信息
   *
   * @param list
   * @return
   */
  private Map<String, List<TerminalRelaOrgVo>> findTerminalOrgMap(List<Terminal> list) {
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    Map<String, List<TerminalRelaOrgVo>> map = Maps.newHashMap();
    Set<String> terminalCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getTerminalCode()))
            .map(Terminal::getTerminalCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isNotEmpty(terminalCodeSet)) {
      final List<TerminalRelaOrg> orgList =
          this.terminalRelaOrgService.findByTerminalCodes(Lists.newArrayList(terminalCodeSet));
      if (CollectionUtils.isNotEmpty(orgList)) {
        List<TerminalRelaOrgVo> orgVoList =
            (List<TerminalRelaOrgVo>)
                this.nebulaToolkitService.copyCollectionByBlankList(
                    orgList,
                    TerminalRelaOrg.class,
                    TerminalRelaOrgVo.class,
                    HashSet.class,
                    ArrayList.class);
        map = orgVoList.stream().collect(Collectors.groupingBy(TerminalRelaOrgVo::getTerminalCode));
      }
    }
    return map;
  }
}
