package com.biz.crm.mdm.business.terminal.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.terminal.local.entity.Terminal;
import com.biz.crm.mdm.business.terminal.local.mapper.TerminalMapper;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalCodeSearchDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalPaginationDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalQueryDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalSearchDto;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalRelaOrgVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Set;

/**
 * 终端信息(repository)
 *
 * @author sunx
 * @since 2021-10-18 17:46:50
 */
@Component
public class TerminalRepository extends ServiceImpl<TerminalMapper, Terminal> {

  private static final Integer SIZE = 1000;

  /**
   * 分页
   *
   * @param page
   * @param dto
   * @return
   */
  public Page<Terminal> findByConditions(Page<Terminal> page, TerminalPaginationDto dto) {
    dto.setTenantCode(TenantUtils.getTenantCode());
    return baseMapper.findByConditions(page, dto);
  }

  public Terminal findById(String id) {
    return this.baseMapper.selectById(id);
  }

  /**
   * 根据id集合获取详情集合
   *
   * @param ids
   * @return
   */
  public List<Terminal> findByIds(List<String> ids) {
    return this.lambdaQuery().in(Terminal::getId, ids).list();
  }

  /**
   * 根据编码集合获取详情集合
   *
   * @param terminalCodes
   * @return
   */
  public List<Terminal> findByTerminalCodes(List<String> terminalCodes) {
    return this.lambdaQuery()
        .eq(Terminal::getTenantCode, TenantUtils.getTenantCode())
        .eq(Terminal::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .in(Terminal::getTerminalCode, terminalCodes)
        .list();
  }

  /**
   * 根据id或编码集合获取详情集合
   *
   * @param terminalCodes
   * @return
   */
  public List<Terminal> findDetailsByIdsOrTerminalCodes(
      List<String> ids, List<String> terminalCodes) {
    return this.lambdaQuery()
        .eq(Terminal::getTenantCode, TenantUtils.getTenantCode())
        .and(
            i ->
                i.in(CollectionUtils.isNotEmpty(ids), Terminal::getId, ids)
                    .or()
                    .in(
                        CollectionUtils.isNotEmpty(terminalCodes),
                        Terminal::getTerminalCode,
                        terminalCodes))
        .list();
  }

  /**
   * 根据id集合 更新组织启用/禁用状态
   *
   * @param ids
   * @param enable
   */
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enable) {
    this.lambdaUpdate()
        .in(Terminal::getId, ids)
        .set(Terminal::getEnableStatus, enable.getCode())
        .update();
  }

  /**
   * 逻辑删除
   *
   * @param ids
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate()
        .in(Terminal::getId, ids)
        .set(Terminal::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

  /**
   * 根据终端编码统计数量
   *
   * @param terminalCode
   * @return
   */
  public Integer countByTerminalCode(String terminalCode) {
    return this.lambdaQuery()
        .eq(Terminal::getTenantCode, TenantUtils.getTenantCode())
        .eq(Terminal::getTerminalCode, terminalCode)
        .count();
  }

  /**
   * 根据企业组织获取对应的终端主信息
   *
   * @param orgCodeList
   * @return
   */
  public List<TerminalRelaOrgVo> findTerminalRelaOrgByOrgCodes(List<String> orgCodeList) {
    return this.baseMapper.findTerminalRelaOrgByOrgCodes(orgCodeList, TenantUtils.getTenantCode());
  }

  /**
   * 根据客户组织编码集合查询
   *
   * @param customerOrgCodes
   * @param tenantCode
   * @param delFlag
   * @return
   */
  public List<Terminal> findByCustomerOrgCodes(
      List<String> customerOrgCodes, String tenantCode, String delFlag) {
    return this.baseMapper.findByCustomerOrgCodes(customerOrgCodes, tenantCode, delFlag);
  }

  /**
   * 根据组织、渠道、标签获取匹配的终端编码
   *
   * @param dto
   * @return
   */
  public Set<String> findByTerminalCodeSearchDto(TerminalCodeSearchDto dto) {
    return this.baseMapper.findByTerminalCodeSearchDto(dto);
  }

  /**
   * 根据组织、渠道、标签获取匹配的终端
   *
   * @param dto 参数dto
   * @return 终端基本信息
   */
  public List<Terminal> findByTerminalSearchDto(TerminalSearchDto dto) {
    return this.baseMapper.findByTerminalSearchDto(dto);
  }

  /**
   * 根据高德id查询数据集合
   *
   * @param amapIds
   * @return
   */
  public List<Terminal> findByAmapIds(Set<String> amapIds) {
    return this.lambdaQuery().in(Terminal::getAmapId, amapIds).list();
  }

  /**
   * 根据流程编号获取终端信息
   *
   * @param processNumber
   * @return
   */
  public Terminal findByProcessNumber(String processNumber) {
    return this.lambdaQuery().eq(Terminal::getProcessNumber, processNumber).one();
  }

  /**
   * 根据TerminalQueryDto获取对应的终端编码集合
   *
   * @param dto
   * @return
   */
  public Set<String> findByTerminalQueryDto(TerminalQueryDto dto) {
    if (CollectionUtils.isEmpty(dto.getOrgCodeSet())) {
      return this.findItemTerminalQueryDto(dto);
    }
    Set<String> set = Sets.newHashSet();
    for (List<String> item : Lists.partition(Lists.newArrayList(dto.getOrgCodeSet()), SIZE)) {
      dto.setOrgCodeSet(Sets.newHashSet(item));
      final Set<String> cur = this.findItemTerminalQueryDto(dto);
      if (CollectionUtils.isEmpty(cur)) {
        continue;
      }
      set.addAll(cur);
    }
    return set;
  }

  private Set<String> findItemTerminalQueryDto(TerminalQueryDto dto) {
    Set<String> set = Sets.newHashSet();
    if (CollectionUtils.isEmpty(dto.getTerminalCodeSet())) {
      return this.baseMapper.findByTerminalQueryDto(dto, TenantUtils.getTenantCode());
    }
    for (List<String> item : Lists.partition(Lists.newArrayList(dto.getTerminalCodeSet()), SIZE)) {
      dto.setTerminalCodeSet(Sets.newHashSet(item));
      final Set<String> cur =
          this.baseMapper.findByTerminalQueryDto(dto, TenantUtils.getTenantCode());
      if (CollectionUtils.isNotEmpty(cur)) {
        set.addAll(cur);
      }
    }
    return set;
  }

  /**
   * 根据创建人账号，年月日期，业务系统类型查询数据
   *
   * @param createAccount
   * @param fromType
   * @param startDate
   * @param endDate
   * @return
   */
  public List<Terminal> findCountByCreateAccountAndFromTypeAndCreateTimeScope(String createAccount, String fromType, String startDate, String endDate) {
    return this.lambdaQuery().eq(Terminal::getCreateAccount, createAccount)
        .eq(Terminal::getFromType, fromType)
        .between(Terminal::getCreateTime, startDate, endDate)
        .list();
  }
}
