package com.biz.crm.mdm.business.terminal.local.service.internal;

import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.customer.user.sdk.service.CustomerUserRelateCustomerVoService;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserRelateCustomerVo;
import com.biz.crm.mdm.business.terminal.local.entity.TerminalSupply;
import com.biz.crm.mdm.business.terminal.local.entity.TerminalSupplyDetail;
import com.biz.crm.mdm.business.terminal.local.repository.TerminalSupplyRepository;
import com.biz.crm.mdm.business.terminal.local.service.TerminalSupplyDetailService;
import com.biz.crm.mdm.business.terminal.local.service.TerminalSupplyService;
import com.biz.crm.mdm.business.terminal.sdk.constant.TerminalConstant;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalBindSupplyDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalRebindSupplyDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalSupplyDetailDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalSupplyDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalUnbindSupplyDto;
import com.biz.crm.mdm.business.terminal.sdk.enums.TerminalSupplyTypeEnum;
import com.biz.crm.mdm.business.terminal.sdk.event.TerminalSupplyEventListener;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalSupplyDetailVo;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalSupplyVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 终端业务员(TerminalSupply)表服务实现类
 *
 * @author sunx
 * @since 2021-10-18 17:49:35
 */
@Service("terminalSupplyService")
public class TerminalSupplyServiceImpl implements TerminalSupplyService {

  @Autowired(required = false) 
  private TerminalSupplyRepository terminalSupplyRepository;

  @Autowired(required = false) 
  private TerminalSupplyDetailService terminalSupplyDetailService;

  @Autowired(required = false) 
  private CustomerUserRelateCustomerVoService customerUserRelateCustomerVoService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false) 
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  @Lazy
  private List<TerminalSupplyEventListener> terminalSupplyEventListeners;

  /**
   * 基于数据库执行的数据视图执行内容缓存（最多500毫秒）
   */
  private static volatile Cache<String, List<TerminalSupply>> cache = null;

  public TerminalSupplyServiceImpl(){
    if(cache == null) {
      synchronized (TerminalSupplyServiceImpl.class) {
        while(cache == null) {
          cache = CacheBuilder.newBuilder()
                  .initialCapacity(10000)
                  .expireAfterWrite(500, TimeUnit.MILLISECONDS)
                  .maximumSize(100000)
                  .build();
        }
      }
    }
  }
  @Override
  public List<TerminalSupply> findByTerminalCodes(List<String> terminalCodeList) {
    if (CollectionUtils.isEmpty(terminalCodeList)) {
      return Lists.newLinkedList();
    }
    String cacheKey = StringUtils.join(TenantUtils.getTenantCode(), terminalCodeList);
    List<TerminalSupply> records = cache.getIfPresent(cacheKey);
    if (records == null) {
      records = terminalSupplyRepository.findByTerminalCodes(terminalCodeList);
      cache.put(cacheKey, records);
    }
    return records;
  }

  @Override
  @Transactional
  public void saveBatch(List<TerminalSupplyDto> list, String terminalCode) {
    /*
     * 保存供货关系信息：
     * 1、根据终端编码获取历史的供货关系信息
     * 2、根据供货关系id集合删除下级关联的明细信息
     * 3、根据终端编码删除供货关系信息
     * 4、保存供货关系信息
     * 5、保存关联的明细信息
     * */
    Validate.notBlank(terminalCode, "终端编码信息不能为空");
    List<TerminalSupply> oldList = this.findByTerminalCodes(Lists.newArrayList(terminalCode));
    if (CollectionUtils.isNotEmpty(oldList)) {
      terminalSupplyDetailService.deleteBySupplyIds(
          oldList.stream().map(TerminalSupply::getId).collect(Collectors.toList()));
    }
    terminalSupplyRepository.deleteByTerminalCodes(Lists.newArrayList(terminalCode));
    if (CollectionUtils.isEmpty(list)) {
      return;
    }

    this.batchSavePrevValidate(list, terminalCode);

    List<String> codeList =
        generateCodeService.generateCode(TerminalConstant.TERMINAL_SUPPLY_CODE, list.size());
    List<TerminalSupplyDetailDto> detailDtoList = Lists.newLinkedList();
    // 设置supplyId
    for (int i = 0; i < list.size(); i++) {
      list.get(i).setId(codeList.get(i));
      list.get(i).setTenantCode(TenantUtils.getTenantCode());
      if (CollectionUtils.isNotEmpty(list.get(i).getDetails())) {
        for (TerminalSupplyDetailDto detail : list.get(i).getDetails()) {
          detail.setSupplyId(codeList.get(i));
          detail.setTenantCode(TenantUtils.getTenantCode());
        }
        detailDtoList.addAll(list.get(i).getDetails());
      }
    }

    terminalSupplyRepository.saveBatch(
        this.nebulaToolkitService.copyCollectionByWhiteList(
            list, TerminalSupplyDto.class, TerminalSupply.class, HashSet.class, ArrayList.class));

    // 保存明细信息
    if (CollectionUtils.isNotEmpty(detailDtoList)) {
      terminalSupplyDetailService.saveBatch(
          (List<TerminalSupplyDetail>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  detailDtoList,
                  TerminalSupplyDetailDto.class,
                  TerminalSupplyDetail.class,
                  HashSet.class,
                  ArrayList.class));
    }
  }

  @Override
  @Transactional
  public void bind(TerminalBindSupplyDto dto) {
    Validate.notNull(dto, "参数不能为空");
    Boolean f =
        StringUtils.isBlank(dto.getPositionCode())
            && StringUtils.isBlank(dto.getCustomerUserName())
            && StringUtils.isBlank(dto.getCustomerCode());
    Validate.isTrue(!f, "岗位、客户用户账号、客户编码不能同时为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getTerminalCodeList()), "终端信息不能为空");

    // 解绑
    List<String> supplyIdList =
        findNeedUnbindSupplyIdList(
            dto.getPositionCode(),
            dto.getCustomerUserName(),
            dto.getCustomerCode(),
            dto.getTerminalCodeList());
    if (CollectionUtils.isNotEmpty(supplyIdList)) {
      this.unbind(supplyIdList, false);
    }

    // 绑定
    List<TerminalSupply> list = Lists.newLinkedList();
    String supplyType = TerminalSupplyTypeEnum.DEFAULT.getValue();
    if (StringUtils.isNotBlank(dto.getCustomerUserName())) {
      supplyType = TerminalSupplyTypeEnum.CUSTOMER_USER.getValue();
    }

    for (String terminalCode : dto.getTerminalCodeList()) {
      TerminalSupply cur = new TerminalSupply();
      cur.setTerminalCode(terminalCode);
      cur.setTenantCode(TenantUtils.getTenantCode());
      cur.setSupplyType(supplyType);
      cur.setPositionCode(StringUtils.EMPTY);
      cur.setCustomerCode(StringUtils.EMPTY);
      cur.setUserName(StringUtils.EMPTY);
      if (StringUtils.isNotBlank(dto.getPositionCode())) {
        cur.setPositionCode(dto.getPositionCode());
      } else if (StringUtils.isNotBlank(dto.getCustomerUserName())) {
        cur.setUserName(dto.getCustomerUserName());
      } else {
        cur.setCustomerCode(dto.getCustomerCode());
      }
      list.add(cur);
    }
    terminalSupplyRepository.saveBatch(list);

    // 发送通知
    if (CollectionUtils.isEmpty(terminalSupplyEventListeners)) {
      return;
    }
    for (TerminalSupplyEventListener terminalSupplyEventListener : terminalSupplyEventListeners) {
      terminalSupplyEventListener.onBind(
          dto.getPositionCode(),
          dto.getCustomerUserName(),
          dto.getCustomerCode(),
          dto.getTerminalCodeList());
    }
  }

  @Override
  @Transactional
  public void unbind(TerminalUnbindSupplyDto dto) {
    Validate.notNull(dto, "参数不能为空");
    Boolean f =
        StringUtils.isBlank(dto.getPositionCode())
            && StringUtils.isBlank(dto.getCustomerUserName())
            && StringUtils.isBlank(dto.getCustomerCode());
    Validate.isTrue(!f, "岗位、客户用户账号、客户编码不能同时为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getTerminalCodeList()), "终端信息不能为空");

    List<String> supplyIdList =
        this.findNeedUnbindSupplyIdList(
            dto.getPositionCode(),
            dto.getCustomerUserName(),
            dto.getCustomerCode(),
            dto.getTerminalCodeList());
    this.unbind(supplyIdList, true);
  }

  @Override
  @Transactional
  public void unbindByCustomer(TerminalUnbindSupplyDto dto) {
    Validate.notNull(dto, "参数不能为空");
    Validate.notBlank(dto.getCustomerCode(), "客户编码不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getTerminalCodeList()), "终端信息不能为空");
    List<TerminalSupply> list =
        this.terminalSupplyRepository.findByCustomerCodeAndTerminalCodesAndSupplyType(
            dto.getCustomerCode(), dto.getTerminalCodeList(), StringUtils.EMPTY);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    // k-supplyType,v-List<TerminalSupply>
    Map<String, List<TerminalSupply>> map =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getSupplyType()))
            .collect(Collectors.groupingBy(TerminalSupply::getSupplyType));
    // 需要置空对应的客户信息
    List<TerminalSupply> list1 = map.get(TerminalSupplyTypeEnum.DEFAULT.getValue());
    // 需要解绑的id集合
    List<String> list2 =
        map
            .getOrDefault(TerminalSupplyTypeEnum.CUSTOMER_USER.getValue(), Lists.newLinkedList())
            .stream()
            .map(TerminalSupply::getId)
            .collect(Collectors.toList());
    if (CollectionUtils.isNotEmpty(list2)) {
      this.unbind(list2, true);
    }

    if (CollectionUtils.isNotEmpty(list1)) {
      for (TerminalSupply item : list1) {
        item.setCustomerCode(StringUtils.EMPTY);
      }
      this.terminalSupplyRepository.saveOrUpdateBatch(list1);
    }
  }

  @Override
  @Transactional
  public void rebind(TerminalRebindSupplyDto dto) {
    Validate.notNull(dto, "参数不能为空");
    Boolean f =
        StringUtils.isBlank(dto.getPositionCode())
            && StringUtils.isBlank(dto.getCustomerUserName())
            && StringUtils.isBlank(dto.getCustomerCode());
    Validate.isTrue(!f, "岗位、客户用户账号、客户编码不能同时为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getTerminalCodeList()), "终端信息不能为空");
    List<String> supplyIdList =
        this.findNeedUnbindSupplyIdList(
            dto.getPositionCode(),
            dto.getCustomerUserName(),
            dto.getCustomerCode(),
            dto.getTerminalCodeList());
    this.unbind(supplyIdList, true);

    TerminalBindSupplyDto dto1 = new TerminalBindSupplyDto();
    dto1.setCustomerCode(dto.getCurCustomerCode());
    dto1.setCustomerUserName(dto.getCurCustomerUserName());
    dto1.setPositionCode(dto.getCurPositionCode());
    dto1.setTerminalCodeList(dto.getTerminalCodeList());
    this.bind(dto1);
  }

  @Override
  @Transactional
  public void rebindByCustomer(TerminalRebindSupplyDto dto) {
    Validate.notNull(dto, "参数不能为空");
    Validate.notBlank(dto.getCustomerCode(), "客户编码不能为空");
    Validate.notBlank(dto.getCurCustomerCode(), "替换后客户编码不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getTerminalCodeList()), "终端信息不能为空");
    List<TerminalSupply> list =
        this.terminalSupplyRepository.findByCustomerCodeAndTerminalCodesAndSupplyType(
            dto.getCustomerCode(), dto.getTerminalCodeList(), StringUtils.EMPTY);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    // k-supplyType,v-List<TerminalSupply>
    Map<String, List<TerminalSupply>> map =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getSupplyType()))
            .collect(Collectors.groupingBy(TerminalSupply::getSupplyType));
    // 需要更新客户信息供货信息
    List<TerminalSupply> list1 = map.get(TerminalSupplyTypeEnum.DEFAULT.getValue());
    // 需要解绑的id集合
    List<TerminalSupply> list2 = map.get(TerminalSupplyTypeEnum.CUSTOMER_USER.getValue());
    if (CollectionUtils.isNotEmpty(list2)) {
      List<String> unbindSupplyIds =
          list2.stream().map(TerminalSupply::getId).collect(Collectors.toList());
      this.unbind(unbindSupplyIds, true);

      // 获取客户对应的客户用户信息
      List<CustomerUserRelateCustomerVo> userRelateCustomerVoList =
          this.customerUserRelateCustomerVoService.findByCustomerCodes(
              Lists.newArrayList(dto.getCurCustomerCode()));
      Optional<CustomerUserRelateCustomerVo> optional =
          userRelateCustomerVoList.stream().findFirst();
      if (optional.isPresent()) {
        TerminalBindSupplyDto dto1 = new TerminalBindSupplyDto();
        dto1.setCustomerCode(dto.getCurCustomerCode());
        dto1.setCustomerUserName(optional.get().getUserName());
        dto1.setTerminalCodeList(
            list2.stream()
                .filter(a -> StringUtils.isNotBlank(a.getTerminalCode()))
                .map(TerminalSupply::getTerminalCode)
                .distinct()
                .collect(Collectors.toList()));
        this.bind(dto1);
      }
    }

    if (CollectionUtils.isNotEmpty(list1)) {
      for (TerminalSupply item : list1) {
        item.setCustomerCode(dto.getCurCustomerCode());
      }
      this.terminalSupplyRepository.saveOrUpdateBatch(list1);
    }
  }

  /**
   * 根据终端信息删除
   *
   * @param terminalCodes
   */
  @Override
  public void deleteByTerminalCodes(List<String> terminalCodes) {
    if (CollectionUtils.isEmpty(terminalCodes)) {
      return;
    }
    this.terminalSupplyRepository.deleteByTerminalCodes(terminalCodes);
  }

  /**
   * 根据用户名查询 供货关系类型为客户用户
   *
   * @param userNameList
   */
  @Override
  public List<TerminalSupply> findByUserNames(List<String> userNameList) {
    if (CollectionUtils.isEmpty(userNameList)) {
      return new ArrayList<>(0);
    }
    String cacheKey = StringUtils.join(TenantUtils.getTenantCode(), userNameList);
    List<TerminalSupply> records = cache.getIfPresent(cacheKey);
    if (records == null) {
      records = this.terminalSupplyRepository.findByUserNames(
              userNameList, TerminalSupplyTypeEnum.CUSTOMER_USER.getValue());
      cache.put(cacheKey, records);
    }
    return records;
  }

  @Override
  public List<TerminalSupply> findByCustomerCodes(Set<String> customerCodeSet) {
    if(CollectionUtils.isEmpty(customerCodeSet)){
      return Lists.newLinkedList();
    }
    String cacheKey = StringUtils.join(TenantUtils.getTenantCode(), customerCodeSet);
    List<TerminalSupply> records = cache.getIfPresent(cacheKey);
    if (records == null) {
      records = this.terminalSupplyRepository.findByCustomerCodes(customerCodeSet);
      cache.put(cacheKey, records);
    }
    return records;
  }

  @Override
  public List<TerminalSupply> findByPositionCodes(Set<String> positionCodeSet) {
    if(CollectionUtils.isEmpty(positionCodeSet)){
      return Lists.newLinkedList();
    }
    String cacheKey = StringUtils.join(TenantUtils.getTenantCode(), positionCodeSet);
    List<TerminalSupply> records = cache.getIfPresent(cacheKey);
    if (records == null) {
      records = this.terminalSupplyRepository.findByPositionCodes(positionCodeSet);
      cache.put(cacheKey, records);
    }
    return records;
  }

  /**
   * 解绑
   *
   * @param supplyIdList 供货关系id集合
   * @param needNotifier true需要发通知，false不需要发通知
   */
  @Transactional
  public void unbind(List<String> supplyIdList, Boolean needNotifier) {
    Validate.isTrue(CollectionUtils.isNotEmpty(supplyIdList), "供货关系id集合不能为空");
    List<TerminalSupply> supplyList = terminalSupplyRepository.findByIds(supplyIdList);
    List<TerminalSupplyDetail> supplyDetailList =
        terminalSupplyDetailService.findBySupplyIds(supplyIdList);

    terminalSupplyDetailService.deleteBySupplyIds(supplyIdList);
    terminalSupplyRepository.removeByIds(supplyIdList);

    if (Boolean.FALSE.equals(needNotifier)) {
      return;
    }
    // 发送通知
    if (CollectionUtils.isEmpty(terminalSupplyEventListeners)) {
      return;
    }
    List<TerminalSupplyVo> supplyVoList = Lists.newLinkedList();
    List<TerminalSupplyDetailVo> supplyDetailVoList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(supplyList)) {
      supplyVoList =
          (List<TerminalSupplyVo>)
              this.nebulaToolkitService.copyCollectionByBlankList(
                  supplyList,
                  TerminalSupply.class,
                  TerminalSupplyVo.class,
                  HashSet.class,
                  ArrayList.class);
    }
    if (CollectionUtils.isNotEmpty(supplyDetailList)) {
      supplyDetailVoList =
          (List<TerminalSupplyDetailVo>)
              this.nebulaToolkitService.copyCollectionByBlankList(
                  supplyDetailList,
                  TerminalSupplyDetail.class,
                  TerminalSupplyDetailVo.class,
                  HashSet.class,
                  ArrayList.class);
    }
    for (TerminalSupplyEventListener terminalSupplyEventListener : terminalSupplyEventListeners) {
      terminalSupplyEventListener.onUnbind(supplyVoList, supplyDetailVoList);
    }
  }

  /**
   * 根据绑定供货关系参数获取需要解绑的历史记录id集合
   *
   * @param positionCode
   * @param customerUserName
   * @param customerCode
   * @param terminalCodeList
   * @return
   */
  private List<String> findNeedUnbindSupplyIdList(
      String positionCode,
      String customerUserName,
      String customerCode,
      List<String> terminalCodeList) {
    if (StringUtils.isNotBlank(positionCode)) {
      return this.findNeedUnbindByPositionCode(positionCode, terminalCodeList);
    }
    if (StringUtils.isNotBlank(customerUserName)) {
      return this.findNeedUnbindByCustomerUserName(customerUserName, terminalCodeList);
    }
    if (StringUtils.isNotBlank(customerCode)) {
      return this.findNeedUnbindByCustomerCode(customerCode, terminalCodeList);
    }
    return Lists.newLinkedList();
  }

  /**
   * 获取企业用户岗位绑定的终端供货记录id集合
   *
   * @param positionCode
   * @param terminalCodeList
   * @return
   */
  private List<String> findNeedUnbindByPositionCode(
      String positionCode, List<String> terminalCodeList) {
    List<String> re = Lists.newLinkedList();
    if (StringUtils.isBlank(positionCode)) {
      return re;
    }
    List<TerminalSupply> list =
        terminalSupplyRepository.findByPositionCodesAndTerminalCodes(
            Lists.newArrayList(positionCode), terminalCodeList);
    if (CollectionUtils.isEmpty(list)) {
      return re;
    }
    return list.stream().map(TerminalSupply::getId).collect(Collectors.toList());
  }

  /**
   * 获取客户用户绑定的终端供货记录id集合
   *
   * @param customerUserName
   * @param terminalCodeList
   * @return
   */
  private List<String> findNeedUnbindByCustomerUserName(
      String customerUserName, List<String> terminalCodeList) {
    List<String> re = Lists.newLinkedList();
    if (StringUtils.isBlank(customerUserName)) {
      return re;
    }
    List<TerminalSupply> list =
        terminalSupplyRepository.findByCustomerUserNameAndTerminalCodes(
            customerUserName, terminalCodeList);
    if (CollectionUtils.isEmpty(list)) {
      return re;
    }
    return list.stream().map(TerminalSupply::getId).collect(Collectors.toList());
  }

  /**
   * 获取企业绑定的终端供货记录id集合
   *
   * @param customerCode
   * @param terminalCodeList
   * @return
   */
  private List<String> findNeedUnbindByCustomerCode(
      String customerCode, List<String> terminalCodeList) {
    List<String> re = Lists.newLinkedList();
    if (StringUtils.isBlank(customerCode)) {
      return re;
    }
    List<TerminalSupply> list =
        terminalSupplyRepository.findByCustomerCodeAndTerminalCodesAndSupplyType(
            customerCode, terminalCodeList, TerminalSupplyTypeEnum.DEFAULT.getValue());
    if (CollectionUtils.isEmpty(list)) {
      return re;
    }
    return list.stream().map(TerminalSupply::getId).collect(Collectors.toList());
  }

  /**
   * 批量保存供货关系前置数据验证
   *
   * @param list
   * @param terminalCode
   */
  private void batchSavePrevValidate(List<TerminalSupplyDto> list, String terminalCode) {
    Optional<TerminalSupplyDto> first =
        list.stream()
            .filter(
                a ->
                    StringUtils.isBlank(a.getTerminalCode())
                        || StringUtils.isBlank(a.getSupplyType())
                        || !terminalCode.equals(a.getTerminalCode()))
            .findFirst();
    Validate.isTrue(!first.isPresent(), "终端编码、供货类型不能为空,且必须属于同一终端");

    list.forEach(
        a -> {
          Validate.isTrue(TerminalSupplyTypeEnum.exists(a.getSupplyType()), "供货类型参数异常");
          if (TerminalSupplyTypeEnum.DEFAULT.getValue().equals(a.getSupplyType())) {
            Validate.notBlank(a.getPositionCode(), "存在未选择负责信息的记录");
          } else {
            Validate.notBlank(a.getUserName(), "存在未选择客户用户信息的记录");
          }
        });

    Set<String> set = Sets.newHashSet();
    list.forEach(
        a ->
            Validate.isTrue(
                set.add(
                    StringUtils.join(
                        Lists.newArrayList(
                            a.getSupplyType(),
                            Optional.ofNullable(a.getPositionCode()).orElse(""),
                            Optional.ofNullable(a.getUserName()).orElse("")),
                        '-')),
                "存在重复的供货关系记录信息"));
  }
}
