package com.biz.crm.mdm.business.terminal.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.customer.org.sdk.service.CustomerOrgVoSdkService;
import com.biz.crm.mdm.business.customer.org.sdk.vo.CustomerOrgVo;
import com.biz.crm.mdm.business.org.sdk.dto.OrgQueryDto;
import com.biz.crm.mdm.business.org.sdk.dto.RelateOrgCodeQueryDto;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.biz.crm.mdm.business.terminal.local.entity.Terminal;
import com.biz.crm.mdm.business.terminal.local.entity.TerminalContact;
import com.biz.crm.mdm.business.terminal.local.entity.TerminalRelaCustomerOrg;
import com.biz.crm.mdm.business.terminal.local.entity.TerminalRelaOrg;
import com.biz.crm.mdm.business.terminal.local.repository.TerminalRelaOrgRepository;
import com.biz.crm.mdm.business.terminal.local.repository.TerminalRepository;
import com.biz.crm.mdm.business.terminal.local.service.TerminalContactService;
import com.biz.crm.mdm.business.terminal.local.service.TerminalRelaCustomerOrgService;
import com.biz.crm.mdm.business.terminal.local.service.TerminalRelaOrgService;
import com.biz.crm.mdm.business.terminal.local.service.TerminalService;
import com.biz.crm.mdm.business.terminal.sdk.constant.TerminalConstant;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalCodeSearchDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalContactDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalPaginationDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalQueryDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalRelaCustomerOrgDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalRelaOrgDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalSearchDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalSupplyDto;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalSupplyVoService;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalContactVo;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalRelaCustomerOrgVo;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalRelaOrgVo;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalSupplyVo;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 终端信息Vo接口 实现
 *
 * @author sunx
 * @date 2021/10/19
 */
@Slf4j
@Service
public class TerminalVoServiceImpl implements TerminalVoService {

  @Autowired(required = false)
  private TerminalService terminalService;

  @Autowired(required = false)
  private TerminalRelaOrgService terminalRelaOrgService;

  @Autowired(required = false)
  private TerminalRelaCustomerOrgService terminalRelaCustomerOrgService;

  @Autowired(required = false)
  private TerminalContactService terminalContactService;

  @Autowired(required = false)
  private TerminalSupplyVoService terminalSupplyVoService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private OrgVoService orgVoService;

  @Autowired(required = false)
  private CustomerOrgVoSdkService customerOrgVoSdkService;

  @Autowired(required = false)
  private TerminalRepository terminalRepository;

  @Autowired(required = false)
  private TerminalRelaOrgRepository terminalRelaOrgRepository;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  public static final String TERMINAL_PROCESS_NAME= "终端审批流程";

  /**
   * 基于数据库执行的数据视图执行内容缓存（最多500毫秒）
   */
  private static volatile Cache<String, List<TerminalVo>> cache = null;

  @Override
  public List<TerminalVo> findDetailsByIdsOrTerminalCodes(
      List<String> ids, List<String> terminalCodes) {
    List<TerminalVo> re = Lists.newLinkedList();
    if (CollectionUtils.isEmpty(ids) && CollectionUtils.isEmpty(terminalCodes)) {
      return re;
    }
    List<Terminal> terminalList =
        terminalService.findDetailsByIdsOrTerminalCodes(ids, terminalCodes);
    if (CollectionUtils.isEmpty(terminalList)) {
      return re;
    }
    re = this.convertEntityToVo(terminalList);
    return re;
  }

  @Override
  @Transactional
  public TerminalVo create(TerminalDto dto) {
    Validate.notNull(dto, "终端信息缺失");
    this.validateContacts(dto);
    // 如果userCode为空需要期初一个编码，否则执行验重逻辑
    if (StringUtils.isEmpty(dto.getTerminalCode())) {
      dto.setTerminalCode(
          generateCodeService.generateCode(TerminalConstant.TERMINAL_CODE, 1).get(0));
    } else {
      Integer count = this.terminalRepository.countByTerminalCode(dto.getTerminalCode());
      Validate.isTrue(null == count || 1 > count, dto.getTerminalCode() + "终端编码已存在");
    }
    Terminal terminal =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, Terminal.class, HashSet.class, ArrayList.class);
    terminal.setProcessStatus(ProcessStatusEnum.PREPARE.getDictCode());
    dto.setTerminalCode(terminal.getTerminalCode());
    this.bindExtInfo(dto);
    this.terminalService.create(terminal);
    return this.buildByDtoAndTerminal(dto, terminal);
  }

  @Override
  @Transactional
  public TerminalVo createAndSubmit(TerminalDto dto) {
    Validate.notNull(dto, "终端信息缺失");
    this.validateContacts(dto);
    // 如果userCode为空需要期初一个编码，否则执行验重逻辑
    if (StringUtils.isEmpty(dto.getTerminalCode())) {
      dto.setTerminalCode(
          generateCodeService.generateCode(TerminalConstant.TERMINAL_CODE, 1).get(0));
    } else {
      Integer count = this.terminalRepository.countByTerminalCode(dto.getTerminalCode());
      Validate.isTrue(null == count || 1 > count, dto.getTerminalCode() + "终端编码已存在");
    }
    Terminal terminal =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, Terminal.class, HashSet.class, ArrayList.class);
    dto.setTerminalCode(terminal.getTerminalCode());
    this.bindExtInfo(dto);
    this.terminalService.create(terminal);
    TerminalVo re = this.buildByDtoAndTerminal(dto, terminal);
    // 接入工作流,审批状态是否与通知等事务有关请处理
    dto.setId(terminal.getId());
    terminal.setProcessNumber(this.commitProcess(dto));
    terminal.setProcessStatus(ProcessStatusEnum.COMMIT.getDictCode());
    this.terminalRepository.updateById(terminal);
    return re;
  }

  @Override
  @Transactional
  public TerminalVo update(TerminalDto dto) {
    Validate.notNull(dto, "终端信息缺失");
    this.validateContacts(dto);
    Terminal terminal =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, Terminal.class, HashSet.class, ArrayList.class);
    dto.setTerminalCode(terminal.getTerminalCode());
    this.bindExtInfo(dto);
    this.terminalService.update(terminal);
    return this.buildByDtoAndTerminal(dto, terminal);
  }

  @Override
  public List<TerminalVo> findSelectByKeyword(String keyword) {
    Pageable pageable = PageRequest.of(0, 20);
    TerminalPaginationDto dto = new TerminalPaginationDto();
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    dto.setKeyword(keyword);
    Page<Terminal> page = this.terminalService.findByConditions(pageable, dto);
    if (Objects.isNull(page) || CollectionUtils.isEmpty(page.getRecords())) {
      return Lists.newLinkedList();
    }
    List<TerminalVo> list = Lists.newArrayList();
    for (Terminal item : page.getRecords()) {
      TerminalVo cur = new TerminalVo();
      cur.setTerminalCode(item.getTerminalCode());
      cur.setTerminalName(item.getTerminalName());
      list.add(cur);
    }
    return list;
  }

  @Override
  public List<TerminalVo> findMainDetailsByTerminalCodesUsePost(List<String> terminalCodeList) {
    return this.findMainDetailsByTerminalCodes(terminalCodeList);
  }

  @Override
  public List<TerminalVo> findTerminalAndContactByTerminalCodes(List<String> terminalCodeList) {
    if (CollectionUtils.isEmpty(terminalCodeList)) {
      return Lists.newLinkedList();
    }
    List<Terminal> terminalList = this.terminalService.findDetailsByIdsOrTerminalCodes(null, terminalCodeList);
    if (CollectionUtils.isEmpty(terminalList)) {
      return Lists.newLinkedList();
    }
    // 联系人
    List<TerminalContact> terminalContactList = this.terminalContactService.findByTerminalCodes(terminalCodeList);
    List<TerminalContactVo> terminalContactVos = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(terminalContactVos)) {
      terminalContactVos = (List<TerminalContactVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
              terminalContactList, TerminalContact.class, TerminalContactVo.class, HashSet.class, ArrayList.class);
    }
    Map<String, List<TerminalContactVo>> terminalContactVoMap = terminalContactVos.stream()
            .collect(Collectors.groupingBy(TerminalContactVo::getTerminalCode));
    List<TerminalVo> terminalVos = (List<TerminalVo>) this.nebulaToolkitService.copyCollectionByBlankList(
            terminalList, Terminal.class, TerminalVo.class, HashSet.class, ArrayList.class);
    terminalVos.forEach(terminalVo -> terminalVo.setContacts(terminalContactVoMap.get(terminalVo.getTerminalCode())));
    return terminalVos;
  }

  @Override
  public List<TerminalVo> findMainDetailsByTerminalCodes(List<String> terminalCodeList) {
    if (CollectionUtils.isEmpty(terminalCodeList)) {
      return Lists.newLinkedList();
    }
    List<Terminal> terminalList =
        this.terminalService.findDetailsByIdsOrTerminalCodes(null, terminalCodeList);
    if (CollectionUtils.isEmpty(terminalList)) {
      return Lists.newLinkedList();
    }
    List<TerminalRelaOrg> terminalRelaOrgList =
        this.terminalRelaOrgService.findByTerminalCodes(terminalCodeList);
    // k-terminalCode,v-orgCodeList
    Map<String, List<String>> map =
        terminalRelaOrgList.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getTerminalCode(), a.getOrgCode()))
            .collect(
                Collectors.groupingBy(
                    TerminalRelaOrg::getTerminalCode,
                    Collectors.mapping(TerminalRelaOrg::getOrgCode, Collectors.toList())));
    Set<String> orgCodeSet =
        terminalRelaOrgList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getOrgCode()))
            .map(TerminalRelaOrg::getOrgCode)
            .collect(Collectors.toSet());
    List<OrgVo> orgVoList = this.orgVoService.findByOrgCodes(Lists.newArrayList(orgCodeSet));
    // k-orgCode,v-orgName
    Map<String, String> mapOrg = Maps.newHashMap();
    if (CollectionUtils.isNotEmpty(orgVoList)) {
      mapOrg =
          orgVoList.stream()
              .filter(a -> StringUtils.isNoneBlank(a.getOrgCode(), a.getOrgName()))
              .collect(Collectors.toMap(OrgVo::getOrgCode, OrgVo::getOrgName, (a, b) -> a));
    }
    // 查看联系人
    List<TerminalContact> terminalContactList =
        this.terminalContactService.findByTerminalCodes(terminalCodeList);
    Map<String, List<TerminalContactVo>> terminalContactVoMap = new HashMap<>();
    if (!CollectionUtils.isEmpty(terminalContactList)) {
      List<TerminalContactVo> terminalContactVos =
          (List<TerminalContactVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  terminalContactList,
                  TerminalContact.class,
                  TerminalContactVo.class,
                  HashSet.class,
                  ArrayList.class);
      terminalContactVoMap =
          terminalContactVos.stream()
              .collect(Collectors.groupingBy(TerminalContactVo::getTerminalCode));
    }
    // 查看客户组织
    List<TerminalRelaCustomerOrg> terminalRelaCustomerOrgs =
        this.terminalRelaCustomerOrgService.findByTerminalCodes(terminalCodeList);
    Map<String, List<TerminalRelaCustomerOrgVo>> terminalRelaCustomerOrgVoMap = new HashMap<>();
    // k-orgCode, v-orgName
    Map<String, String> customerOrgVoMap = new HashMap<>();
    if (!CollectionUtils.isEmpty(terminalRelaCustomerOrgs)) {
      List<TerminalRelaCustomerOrgVo> terminalRelaCustomerOrgVos =
          (List<TerminalRelaCustomerOrgVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  terminalRelaCustomerOrgs,
                  TerminalRelaCustomerOrg.class,
                  TerminalRelaCustomerOrgVo.class,
                  HashSet.class,
                  ArrayList.class);
      terminalRelaCustomerOrgVoMap =
          terminalRelaCustomerOrgVos.stream()
              .collect(Collectors.groupingBy(TerminalRelaCustomerOrgVo::getTerminalCode));
      List<CustomerOrgVo> customerOrgVos =
          this.customerOrgVoSdkService.findListByCodes(
              terminalRelaCustomerOrgVos.stream()
                  .map(TerminalRelaCustomerOrgVo::getOrgCode)
                  .collect(Collectors.toList()));
      if (!CollectionUtils.isEmpty(customerOrgVos)) {
        customerOrgVoMap =
            customerOrgVos.stream()
                .filter(
                    a -> StringUtils.isNoneBlank(a.getCustomerOrgCode(), a.getCustomerOrgName()))
                .collect(
                    Collectors.toMap(
                        CustomerOrgVo::getCustomerOrgCode, CustomerOrgVo::getCustomerOrgName));
      }
    }

    List<TerminalVo> re =
        (List<TerminalVo>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                terminalList, Terminal.class, TerminalVo.class, HashSet.class, ArrayList.class);
    for (TerminalVo item : re) {
      List<String> curOrgCodeList = map.get(item.getTerminalCode());
      if (!CollectionUtils.isEmpty(curOrgCodeList)) {
        List<TerminalRelaOrgVo> curList = Lists.newLinkedList();
        List<String> orgNameList = Lists.newLinkedList();
        for (String orgCode : curOrgCodeList) {
          TerminalRelaOrgVo cur = new TerminalRelaOrgVo();
          cur.setTerminalCode(item.getTerminalCode());
          cur.setOrgCode(orgCode);
          cur.setOrgName(mapOrg.get(orgCode));
          curList.add(cur);
          orgNameList.add(cur.getOrgName());
        }
        item.setOrgList(curList);
        item.setOrgNameStr(StringUtils.join(orgNameList, ","));
      }
      // 赋值联系人
      List<TerminalContactVo> terminalContactVos = terminalContactVoMap.get(item.getTerminalCode());
      if (!CollectionUtils.isEmpty(terminalContactVos)) {
        item.setContacts(terminalContactVos);
      }
      // 赋值客户组织
      List<TerminalRelaCustomerOrgVo> terminalRelaCustomerOrgVos =
          terminalRelaCustomerOrgVoMap.get(item.getTerminalCode());
      if (!CollectionUtils.isEmpty(terminalRelaCustomerOrgVos)) {
        for (TerminalRelaCustomerOrgVo terminalRelaCustomerOrgVo : terminalRelaCustomerOrgVos) {
          terminalRelaCustomerOrgVo.setOrgName(
              customerOrgVoMap.get(terminalRelaCustomerOrgVo.getOrgCode()));
        }
        item.setCustomerOrgList(terminalRelaCustomerOrgVos);
      }
    }
    return re;
  }

  @Override
  public Page<TerminalVo> findByTerminalPaginationDto(
      Pageable pageable, TerminalPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new TerminalPaginationDto());
    Page<Terminal> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<Terminal> entityPage = this.terminalRepository.findByConditions(page, dto);
    Page<TerminalVo> pageResult =
        new Page<>(entityPage.getCurrent(), entityPage.getSize(), entityPage.getTotal());
    if (CollectionUtils.isEmpty(entityPage.getRecords())) {
      return pageResult;
    }
    pageResult.setRecords(this.convertEntityToVo(entityPage.getRecords()));
    return pageResult;
  }

  @Override
  public Set<String> findByTerminalCodeSearchDto(TerminalCodeSearchDto dto) {
    dto = Optional.ofNullable(dto).orElse(new TerminalCodeSearchDto());
    return this.terminalService.findByTerminalCodeSearchDto(dto);
  }

  @Override
  public List<TerminalVo> findByAmapIds(Set<String> amapIds) {
    String cacheKey = StringUtils.join(amapIds);
    if(cache == null) {
      synchronized (TerminalVoServiceImpl.class) {
        while(cache == null) {
          cache = CacheBuilder.newBuilder()
                  .initialCapacity(10000)
                  .expireAfterWrite(300, TimeUnit.MILLISECONDS)
                  .maximumSize(100000)
                  .build();
        }
      }
    }
    List<TerminalVo> graph = cache.getIfPresent(cacheKey);
    if (graph == null) {
      graph = findByAmapIdsFromDB(amapIds);
      cache.put(cacheKey, graph);
    }
    return graph;
  }

  private List<TerminalVo> findByAmapIdsFromDB(Set<String> amapIds) {
    log.warn(" db 查询 findByAmapIds ");
    if (CollectionUtils.isEmpty(amapIds)) {
      return Lists.newArrayList();
    }
    List<Terminal> list = this.terminalRepository.findByAmapIds(amapIds);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newArrayList();
    }
    return (List<TerminalVo>)
        this.nebulaToolkitService.copyCollectionByWhiteList(
            list, Terminal.class, TerminalVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public void submitAudit(TerminalDto dto) {
    Validate.notNull(dto, "参数不能为空");
    Validate.notBlank(dto.getId(), "终端id不能为空");
    final Terminal terminal = this.terminalRepository.findById(dto.getId());
    Validate.notNull(terminal, "终端信息不存在");
    Set<String> set =
        Sets.newHashSet(
            ProcessStatusEnum.PREPARE.getDictCode(),
            ProcessStatusEnum.REJECT.getDictCode(),
            ProcessStatusEnum.RECOVER.getDictCode());
    Validate.isTrue(set.contains(terminal.getProcessStatus()), "终端状态不能执行提交审核");
    dto.setTerminalCode(terminal.getTerminalCode());
    // todo 接入工作流,审批状态是否与通知等事务有关请处理
    terminal.setProcessNumber(this.commitProcess(dto));
    terminal.setProcessStatus(ProcessStatusEnum.COMMIT.getDictCode());
    ProcessBusinessDto processBusiness = dto.getProcessBusiness();
    terminal.setProcessRemark(processBusiness.getRemark());
    terminal.setProcessKey(processBusiness.getProcessKey());
    this.terminalRepository.updateById(terminal);
  }

  @Override
  public Set<String> findByTerminalQueryDto(TerminalQueryDto dto) {
    if (Objects.isNull(dto)
        || StringUtils.isAllBlank(
            dto.getTerminalCode(),
            dto.getTerminalName(),
            dto.getChannel(),
            dto.getOrgName(),
            dto.getDelFlag(),
            dto.getProcessStatus())) {
      return Sets.newHashSet();
    }

    if (StringUtils.isNotBlank(dto.getOrgName())) {
      final OrgQueryDto queryDto = new OrgQueryDto();
      queryDto.setOrgName(dto.getOrgName());
      final Set<String> orgCodeSet = this.orgVoService.findByOrgQueryDto(queryDto);
      if (CollectionUtils.isEmpty(orgCodeSet)) {
        return Sets.newHashSet();
      }
      dto.setOrgCodeSet(orgCodeSet);
    }
    return this.terminalRepository.findByTerminalQueryDto(dto);
  }

  @Override
  public Map<String, Set<String>> findAllowSaleTerminalByOrgCodes(Set<String> orgCodes) {
    if (org.springframework.util.CollectionUtils.isEmpty(orgCodes)) {
      return Maps.newHashMap();
    }
    final RelateOrgCodeQueryDto queryDto = new RelateOrgCodeQueryDto();
    queryDto.setOrgCodeSet(orgCodes);
    queryDto.setSearchType(-1);
    final Map<String, String> orgRuleMap = this.orgVoService.findByRelateOrgCodeQueryDto(queryDto);
    if (org.springframework.util.CollectionUtils.isEmpty(orgRuleMap)) {
      return Maps.newHashMap();
    }
    List<TerminalRelaOrg> list =
        this.terminalRelaOrgRepository.findAllowSaleTerminalByOrgCodes(orgRuleMap.keySet());
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    Map<String, Set<String>> map =
        list.stream()
            .filter(
                a ->
                    StringUtils.isNoneBlank(a.getTerminalCode(), a.getOrgCode())
                        && orgRuleMap.keySet().contains(a.getOrgCode()))
            .collect(
                Collectors.groupingBy(
                    TerminalRelaOrg::getTerminalCode,
                    Collectors.mapping(TerminalRelaOrg::getOrgCode, Collectors.toSet())));
    Map<String, Set<String>> re = Maps.newHashMap();
    for (Entry<String, Set<String>> item : map.entrySet()) {
      Set<String> rule = Sets.newHashSet();
      for (String orgCode : item.getValue()) {
        final String s = orgRuleMap.get(orgCode);
        if (StringUtils.isBlank(s)) {
          continue;
        }
        rule.add(s);
      }
      if (CollectionUtils.isEmpty(rule)) {
        continue;
      }
      re.put(item.getKey(), rule);
    }
    return re;
  }


  @Override
  public List<TerminalVo> findByTerminalSearchDto(TerminalSearchDto dto) {
    dto = Optional.ofNullable(dto).orElse(new TerminalSearchDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(dto.getOrgCodeSet())
        && CollectionUtils.isEmpty(dto.getChannelSet())
        && CollectionUtils.isEmpty(dto.getTagSet())
        && CollectionUtils.isEmpty(dto.getTerminalCodeSet())) {
      return Lists.newArrayList();
    }
    List<Terminal> terminalList = this.terminalRepository.findByTerminalSearchDto(dto);
    if (CollectionUtils.isEmpty(terminalList)) {
      return Lists.newArrayList();
    }
    return (List<TerminalVo>) this.nebulaToolkitService.copyCollectionByBlankList(
            terminalList, Terminal.class, TerminalVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 保存扩展信息
   *
   * @param dto
   */
  private void bindExtInfo(TerminalDto dto) {
    /*
     * 保存终端关联信息：
     * 1、保存关联组织信息
     * 2、保存关联联系人信息
     * 3、保存关联供货关系信息
     * */
    Validate.notNull(dto, "终端信息缺失");
    List<TerminalRelaOrg> orgList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getOrgList())) {
      for (TerminalRelaOrgDto item : dto.getOrgList()) {
        item.setTerminalCode(dto.getTerminalCode());
      }
      orgList =
          (List<TerminalRelaOrg>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getOrgList(),
                  TerminalRelaOrgDto.class,
                  TerminalRelaOrg.class,
                  HashSet.class,
                  ArrayList.class);
      orgList.forEach(a -> a.setTerminalCode(dto.getTerminalCode()));
    }
    terminalRelaOrgService.saveBatch(orgList, dto.getTerminalCode());

    List<TerminalRelaCustomerOrg> customerOrgList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getCustomerOrgList())) {
      for (TerminalRelaCustomerOrgDto item : dto.getCustomerOrgList()) {
        item.setTerminalCode(dto.getTerminalCode());
      }
      customerOrgList =
          (List<TerminalRelaCustomerOrg>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getCustomerOrgList(),
                  TerminalRelaCustomerOrgDto.class,
                  TerminalRelaCustomerOrg.class,
                  HashSet.class,
                  ArrayList.class);
      customerOrgList.forEach(a -> a.setTerminalCode(dto.getTerminalCode()));
    }
    terminalRelaCustomerOrgService.saveBatch(customerOrgList, dto.getTerminalCode());

    List<TerminalContact> contactList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getContacts())) {
      for (TerminalContactDto item : dto.getContacts()) {
        item.setTerminalCode(dto.getTerminalCode());
      }
      contactList =
          (List<TerminalContact>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getContacts(),
                  TerminalContactDto.class,
                  TerminalContact.class,
                  HashSet.class,
                  ArrayList.class);
      contactList.forEach(a -> a.setTerminalCode(dto.getTerminalCode()));
    }
    terminalContactService.saveBatch(contactList, dto.getTerminalCode());

    if (CollectionUtils.isNotEmpty(dto.getSupplys())) {
      for (TerminalSupplyDto item : dto.getSupplys()) {
        item.setTerminalCode(dto.getTerminalCode());
      }
    }
    terminalSupplyVoService.saveBatch(dto.getSupplys(), dto.getTerminalCode());
  }

  /**
   * 构建返回信息
   *
   * @param dto
   * @param terminal
   * @return
   */
  private TerminalVo buildByDtoAndTerminal(TerminalDto dto, Terminal terminal) {
    TerminalVo vo =
        this.nebulaToolkitService.copyObjectByWhiteList(
            terminal, TerminalVo.class, HashSet.class, ArrayList.class);
    if (CollectionUtils.isNotEmpty(dto.getOrgList())) {
      vo.setOrgList(
          (List<TerminalRelaOrgVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getOrgList(),
                  TerminalRelaOrgDto.class,
                  TerminalRelaOrgVo.class,
                  HashSet.class,
                  ArrayList.class));
    }
    if (CollectionUtils.isNotEmpty(dto.getCustomerOrgList())) {
      vo.setCustomerOrgList(
          (List<TerminalRelaCustomerOrgVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getCustomerOrgList(),
                  TerminalRelaCustomerOrgDto.class,
                  TerminalRelaCustomerOrgVo.class,
                  HashSet.class,
                  ArrayList.class));
    }
    if (CollectionUtils.isNotEmpty(dto.getContacts())) {
      vo.setContacts(
          (List<TerminalContactVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getContacts(),
                  TerminalContactDto.class,
                  TerminalContactVo.class,
                  HashSet.class,
                  ArrayList.class));
    }
    if (CollectionUtils.isNotEmpty(dto.getSupplys())) {
      vo.setSupplys(
          (List<TerminalSupplyVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getSupplys(),
                  TerminalSupplyDto.class,
                  TerminalSupplyVo.class,
                  HashSet.class,
                  ArrayList.class));
    }
    return vo;
  }

  private List<TerminalVo> buildTerminalVoList(
      List<Terminal> terminalList,
      List<TerminalRelaOrg> terminalRelaOrgList,
      List<TerminalRelaCustomerOrg> terminalRelaCustomerOrgList,
      List<TerminalContact> terminalContactList,
      List<TerminalSupplyVo> terminalSupplyVoList) {
    List<TerminalVo> re = Lists.newLinkedList();
    if (CollectionUtils.isEmpty(terminalList)) {
      return re;
    }
    re =
        (List<TerminalVo>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                terminalList, Terminal.class, TerminalVo.class, HashSet.class, ArrayList.class);

    Map<String, List<TerminalRelaOrgVo>> mapOrg = Maps.newHashMap();
    Map<String, List<TerminalRelaCustomerOrgVo>> mapCustomerOrg = Maps.newHashMap();
    Map<String, List<TerminalContactVo>> mapContact = Maps.newHashMap();
    Map<String, List<TerminalSupplyVo>> mapSupply = Maps.newHashMap();

    if (CollectionUtils.isNotEmpty(terminalRelaOrgList)) {
      List<TerminalRelaOrgVo> list =
          (List<TerminalRelaOrgVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  terminalRelaOrgList,
                  TerminalRelaOrg.class,
                  TerminalRelaOrgVo.class,
                  HashSet.class,
                  ArrayList.class);
      mapOrg = list.stream().collect(Collectors.groupingBy(TerminalRelaOrgVo::getTerminalCode));
    }

    if (CollectionUtils.isNotEmpty(terminalRelaCustomerOrgList)) {
      List<TerminalRelaCustomerOrgVo> list =
          (List<TerminalRelaCustomerOrgVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  terminalRelaCustomerOrgList,
                  TerminalRelaCustomerOrg.class,
                  TerminalRelaCustomerOrgVo.class,
                  HashSet.class,
                  ArrayList.class);
      mapCustomerOrg =
          list.stream().collect(Collectors.groupingBy(TerminalRelaCustomerOrgVo::getTerminalCode));
    }

    if (CollectionUtils.isNotEmpty(terminalContactList)) {
      List<TerminalContactVo> list =
          (List<TerminalContactVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  terminalContactList,
                  TerminalContact.class,
                  TerminalContactVo.class,
                  HashSet.class,
                  ArrayList.class);
      mapContact = list.stream().collect(Collectors.groupingBy(TerminalContactVo::getTerminalCode));
    }

    if (CollectionUtils.isNotEmpty(terminalSupplyVoList)) {
      mapSupply =
          terminalSupplyVoList.stream()
              .collect(Collectors.groupingBy(TerminalSupplyVo::getTerminalCode));
    }
    for (TerminalVo vo : re) {
      vo.setOrgList(mapOrg.get(vo.getTerminalCode()));
      vo.setCustomerOrgList(mapCustomerOrg.get(vo.getTerminalCode()));
      vo.setContacts(mapContact.get(vo.getTerminalCode()));
      vo.setSupplys(mapSupply.get(vo.getTerminalCode()));
    }
    return re;
  }

  /**
   * 校验联系人信息
   *
   * @param dto
   */
  private void validateContacts(TerminalDto dto) {
    Validate.notNull(dto, "终端信息缺失");
    Validate.notEmpty(dto.getTerminalCode(), "终端编码不能为空");
    List<TerminalContactDto> contacts = dto.getContacts();
    if (CollectionUtils.isEmpty(contacts)) {
      return;
    }
    // 校验字段
    for (TerminalContactDto contact : contacts) {
      if (StringUtils.isNotBlank(contact.getContactName())
          || StringUtils.isNotBlank(contact.getContactPhone())
          || !Objects.isNull(contact.getContactMain())) {
        Validate.notBlank(contact.getContactPhone(), "缺失联系电话");
        Validate.notNull(contact.getContactMain(), "请选择是否是主联系人");
        Validate.notBlank(contact.getContactName(), "缺失姓名");
      }
    }
    // 校验是否存在主联系人
    List<TerminalContactDto> contanctsByMain =
        contacts.stream().filter(o -> Boolean.TRUE.equals(o.getContactMain())).collect(Collectors.toList());
    Validate.isTrue(contanctsByMain.size() == 1, "必须有且仅有一个主联系人");
  }

  /**
   * 转换终端实体为终端信息Vo,并完善关联信息
   *
   * @param terminalList 终端实体列表
   * @return 终端信息Vo列表
   */
  private List<TerminalVo> convertEntityToVo(List<Terminal> terminalList) {
    List<TerminalVo> re = Lists.newArrayList();
    List<String> terminalCodeList =
        terminalList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getTerminalCode()))
            .map(Terminal::getTerminalCode)
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(terminalCodeList)) {
      return re;
    }

    List<TerminalRelaOrg> terminalRelaOrgList =
        terminalRelaOrgService.findByTerminalCodes(terminalCodeList);
    List<TerminalRelaCustomerOrg> terminalRelaCustomerOrgList =
        terminalRelaCustomerOrgService.findByTerminalCodes(terminalCodeList);

    Set<String> orgCodeSet =
        terminalRelaOrgList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getOrgCode()))
            .map(TerminalRelaOrg::getOrgCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isNotEmpty(orgCodeSet)) {
      List<OrgVo> orgVoList = this.orgVoService.findByOrgCodes(Lists.newArrayList(orgCodeSet));
      orgVoList = CollectionUtils.isEmpty(orgVoList) ? Lists.newLinkedList() : orgVoList;
      Map<String, String> map =
          orgVoList.stream()
              .filter(a -> StringUtils.isNoneBlank(a.getOrgCode(), a.getOrgName()))
              .collect(Collectors.toMap(OrgVo::getOrgCode, OrgVo::getOrgName, (a, b) -> a));
      terminalRelaOrgList.forEach(a -> a.setOrgName(map.get(a.getOrgCode())));
    }

    Set<String> customerOrgCodeSet =
        terminalRelaCustomerOrgList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getOrgCode()))
            .map(TerminalRelaCustomerOrg::getOrgCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isNotEmpty(customerOrgCodeSet)) {
      List<CustomerOrgVo> customerOrgVoList =
          this.customerOrgVoSdkService.findListByCodes(Lists.newArrayList(customerOrgCodeSet));
      customerOrgVoList =
          CollectionUtils.isEmpty(customerOrgVoList) ? Lists.newLinkedList() : customerOrgVoList;
      Map<String, String> map =
          customerOrgVoList.stream()
              .filter(a -> StringUtils.isNoneBlank(a.getCustomerOrgCode(), a.getCustomerOrgName()))
              .collect(
                  Collectors.toMap(
                      CustomerOrgVo::getCustomerOrgCode,
                      CustomerOrgVo::getCustomerOrgName,
                      (a, b) -> a));
      terminalRelaCustomerOrgList.forEach(a -> a.setOrgName(map.get(a.getOrgCode())));
    }

    List<TerminalContact> terminalContactList =
        terminalContactService.findByTerminalCodes(terminalCodeList);
    List<TerminalSupplyVo> terminalSupplyVoList =
        terminalSupplyVoService.findByTerminalCodes(terminalCodeList);
    re =
        this.buildTerminalVoList(
            terminalList,
            terminalRelaOrgList,
            terminalRelaCustomerOrgList,
            terminalContactList,
            terminalSupplyVoList);
    return re;
  }

  @Autowired(required = false)
  private ProcessBusinessService processBusinessService;

  @Value("${crm.business.terminal.process-key:}")
  private String defaultProcessKey;

  /**
   * 终端新增提交工作流进行审批，提交成功返回流程实例ID，提交失败则抛出异常
   *
   * @param dto 终端请求DTO
   */
  private String commitProcess(TerminalDto dto) {
    ProcessBusinessDto processBusiness = dto.getProcessBusiness();
    String processKey = processBusiness.getProcessKey();
    if (StringUtils.isBlank(processKey)) {
      processBusiness.setProcessKey(defaultProcessKey);
    }
    processBusiness.setProcessTitle(TERMINAL_PROCESS_NAME);
    processBusiness.setBusinessNo(dto.getTerminalCode());
    processBusiness.setBusinessFormJson(JsonUtils.obj2JsonString(dto));
    processBusiness.setBusinessCode(TerminalConstant.TERMINAL_PROCESS_NAME);
    ProcessBusinessVo processBusinessVo = processBusinessService.processStart(processBusiness);
    return processBusinessVo.getProcessNo();
  }
}
