package com.biz.crm.mdm.business.terminal.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.terminal.local.entity.Terminal;
import com.biz.crm.mdm.business.terminal.local.service.TerminalService;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalPageDto;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalPaginationDto;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalAddressVo;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalPageVo;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * 终端信息(Terminal)表控制层
 *
 * @author sunx
 * @since 2021-10-18 17:46:52
 */
@Slf4j
@Api(tags = "终端管理: Terminal: 终端信息")
@RestController
@RequestMapping(value = {"/v1/terminal/terminal"})
public class TerminalController {

  @Autowired(required = false)
  private TerminalService terminalService;

  @ApiOperation(value = "查询分页列表")
  @GetMapping(value = {"/findByConditions"})
  public Result<Page<Terminal>> findByConditions(
      @PageableDefault(50) Pageable pageable, TerminalPaginationDto paginationDto) {
    try {
      paginationDto = Optional.ofNullable(paginationDto).orElse(new TerminalPaginationDto());
      paginationDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      Page<Terminal> result = this.terminalService.findByConditions(pageable, paginationDto);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "启用")
  @PatchMapping("/enable")
  public Result<?> enable(@RequestBody List<String> ids) {
    try {
      this.terminalService.enableBatch(ids);
      return Result.ok("启用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "禁用")
  @PatchMapping("/disable")
  public Result<?> disable(@RequestBody List<String> ids) {
    try {
      this.terminalService.disableBatch(ids);
      return Result.ok("禁用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "逻辑删除")
  @DeleteMapping("/delete")
  public Result<?> delete(@RequestParam("ids") List<String> ids) {
    try {
      this.terminalService.updateDelFlagByIds(ids);
      return Result.ok("删除成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据创建人账号，年月日期，业务系统类型统计数据
   *
   * @param dateType 日期类型：month,week,day
   * @return {"count":12}
   */
  @ApiOperation("根据人员账号获取本月的已完成拜访数")
  @GetMapping("findCountByCreateAccountAndFromTypeAndDateType")
  public Result<?> findCountByCreateAccountAndDateTypeAndFromType(@ApiParam(name = "createAccount", value = "人员账号") @RequestParam(value = "createAccount", required = false) String createAccount,
                                                                  @ApiParam(name = "fromType", value = "业务来源系统") @RequestParam(value = "fromType") String fromType,
                                                                  @ApiParam(name = "dateType", value = "日期类型：month,week,day") @RequestParam(value = "dateType") String dateType) {
    try {
      Map<String, Integer> map = this.terminalService.findCountByCreateAccountAndDateTypeAndFromType(createAccount, dateType, fromType);
      return Result.ok(map);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  @ApiOperation(value = "终端收货地址")
  @GetMapping("/terminalAddressDetail")
  public Result<List<TerminalAddressVo>> terminalAddressDetail(@ApiParam(value="终端id（兆信）")@RequestParam(name="terminalId") String terminalId) {
    try {
      List<TerminalAddressVo> terminalAddressVoList=this.terminalService.terminalAddressDetail(terminalId);
      return Result.ok(terminalAddressVoList);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 营销活动管理查询终端信息分页接口
   */
  @ApiOperation(value = "终端信息分页查询")
  @PostMapping("/findByTerminalPageDto")
  public Result<Page<TerminalPageVo>> findByTerminalPageDto(@PageableDefault(50) Pageable pageable,
                                                      @RequestBody TerminalPageDto dto){
    try {
      Page<TerminalPageVo> page =this.terminalService.findByTerminalPageDto(pageable,dto);
      return Result.ok(page);
    } catch (RuntimeException e) {
      log.error(e.getMessage(),e);
      return Result.error(e.getMessage());
    }
  }
}
