package com.biz.crm.mdm.business.terminal.local.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.customer.sdk.service.SupplyRelationshipSdkService;
import com.biz.crm.mdm.business.customer.sdk.vo.SupplyRelationshipVo;
import com.biz.crm.mdm.business.terminal.sdk.dto.*;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.google.common.collect.Lists;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.*;

/**
 * 终端信息(Terminal)vo控制层
 *
 * @author sunx
 * @date 2021/10/19
 */
@Slf4j
@Api(tags = "终端管理: TerminalVo: 终端信息")
@RestController
@RequestMapping(value = {"/v1/terminal/terminal"})
public class TerminalVoController {

  @Autowired(required = false)
  private TerminalVoService terminalVoService;

  @Autowired(required = false)
  private SupplyRelationshipSdkService relationshipSdkService;

  
  /**
   * 详情
   *
   * @param id
   * @return
   */
  @ApiOperation(value = "详情")
  @GetMapping(value = {"/findDetailById"})
  public Result<TerminalVo> findDetailById(@RequestParam("id") String id) {
    try {
      TerminalVo vo = null;
      List<TerminalVo> list =
          this.terminalVoService.findDetailsByIdsOrTerminalCodes(Lists.newArrayList(id), null);
      Optional<TerminalVo> optional = list.stream().findFirst();
      if (optional.isPresent()) {
        vo = optional.get();
        List<SupplyRelationshipVo> relationshipVos = relationshipSdkService.findByBuyerId(vo.getId());
        vo.setRelationshipList(relationshipVos);
      }
      return Result.ok(vo);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据终端编码获取详情
   *
   * @param terminalCode
   * @return
   */
  @ApiOperation(value = "根据终端编码获取详情")
  @GetMapping(value = {"/findDetailByTerminalCode"})
  public Result<TerminalVo> findDetailByTerminalCode(
      @RequestParam("terminalCode") String terminalCode) {
    try {
      TerminalVo vo = null;
      List<TerminalVo> list =
          this.terminalVoService.findDetailsByIdsOrTerminalCodes(
              null, Lists.newArrayList(terminalCode));
      Optional<TerminalVo> optional = list.stream().findFirst();
      if (optional.isPresent()) {
        vo = optional.get();
      }
      return Result.ok(vo);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据id或编码集合获取终端详情
   *
   * @param ids
   * @param codes
   * @return
   */
  @ApiOperation(value = "根据id或编码集合获取终端详情")
  @GetMapping(value = {"/findDetailByIdsOrCodes"})
  public Result<List<TerminalVo>> findDetailByIdsOrCodes(
      @RequestParam("ids") List<String> ids, @RequestParam("codes") List<String> codes) {
    try {
      List<TerminalVo> list = this.terminalVoService.findDetailsByIdsOrTerminalCodes(ids, codes);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 创建
   *
   * @param dto
   * @return
   */
  @ApiOperation(value = "创建")
  @PostMapping(value = "")
  public Result<TerminalVo> create(@RequestBody TerminalDto dto) {
    try {
      TerminalVo current = this.terminalVoService.create(dto);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 保存并提交
   *
   * @param dto
   * @return
   */
  @ApiOperation(value = "保存并提交")
  @PostMapping(value = "/handlerCreateAndSubmit")
  public Result<TerminalVo> handlerCreateAndSubmit(@RequestBody TerminalDto dto) {
    try {
      TerminalVo current = this.terminalVoService.createAndSubmit(dto);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 更新
   *
   * @param dto
   * @return
   */
  @ApiOperation(value = "更新")
  @PatchMapping(value = "")
  public Result<TerminalVo> update(@RequestBody TerminalDto dto) {
    try {
      TerminalVo current = this.terminalVoService.update(dto);
      return Result.ok(current);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 小程序更新
   * 网关没有配置patch转post
   *
   * @param dto
   * @return
   */
  @ApiOperation(value = "更新")
  @PostMapping(value = "updateTurnPatch")
  public Result<TerminalVo> updateTurnPatch(@RequestBody TerminalDto dto) {
    return this.update(dto);
  }

  /**
   * 根据编码获取终端主信息+组织信息+联系人
   *
   * @param terminalCodeList
   * @return
   */
  @ApiOperation(value = "根据编码获取终端主信息+组织信息+联系人")
  @GetMapping("/findMainDetailsByTerminalCodes")
  public Result<List<TerminalVo>> findMainDetailsByTerminalCodes(
      @RequestParam("terminalCodeList") List<String> terminalCodeList) {
    try {
      List<TerminalVo> list =
          this.terminalVoService.findMainDetailsByTerminalCodes(terminalCodeList);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 将上面方法改写成POST
   * 根据编码获取终端主信息+组织信息+联系人
   *
   * @param terminalCodeList
   * @return
   */
  @ApiOperation(value = "根据编码获取终端主信息+组织信息+联系人")
  @PostMapping("/findMainDetailsByTerminalCodesUsePost")
  public Result<List<TerminalVo>> findMainDetailsByTerminalCodesUsePost(
      @RequestBody List<String> terminalCodeList) {
    try {
      List<TerminalVo> list =
          this.terminalVoService.findMainDetailsByTerminalCodes(terminalCodeList);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据终端编码集合获取终端主信息+联系人
   *
   * @param terminalCodeList
   * @return
   */
  @ApiOperation(value = "根据终端编码集合获取终端主信息+联系人")
  @PostMapping("/findTerminalAndContactByTerminalCodes")
  public Result<List<TerminalVo>> findTerminalAndContactByTerminalCodes(@RequestBody List<String> terminalCodeList) {
    try {
      List<TerminalVo> list = this.terminalVoService.findTerminalAndContactByTerminalCodes(terminalCodeList);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 分页条件查询
   *
   * @param pageable 分页参数
   * @param dto      参数dto
   * @return 终端分页信息
   */
  @ApiOperation(value = "分页条件查询")
  @GetMapping(value = {"/findByTerminalPaginationDto"})
  public Result<Page<TerminalVo>> findByTerminalPaginationDto(
      @PageableDefault(50) Pageable pageable, TerminalPaginationDto dto) {
    try {
      dto = Optional.ofNullable(dto).orElse(new TerminalPaginationDto());
      dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      Page<TerminalVo> result = this.terminalVoService.findByTerminalPaginationDto(pageable, dto);
      return Result.ok(result);
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据组织、渠道、标签获取匹配的终端编码
   *
   * @param dto
   * @return
   */
  @ApiOperation(value = "根据组织、渠道、标签获取匹配的终端编码")
  @PostMapping("/findByTerminalCodeSearchDto")
  public Result<Set<String>> findByTerminalCodeSearchDto(@RequestBody TerminalCodeSearchDto dto) {
    try {
      return Result.ok(this.terminalVoService.findByTerminalCodeSearchDto(dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据组织、渠道、标签获取匹配的终端基本信息
   *
   * @param dto 参数dto
   * @return 终端基本信息
   */
  @ApiOperation(value = "根据组织、渠道、标签获取匹配的终端基本信息")
  @GetMapping("/findByTerminalSearchDto")
  public Result<List<TerminalVo>> findByTerminalSearchDto(@ApiParam(name = "TerminalSearchDto", value = "查询Dto") TerminalSearchDto dto) {
    try {
      return Result.ok(this.terminalVoService.findByTerminalSearchDto(dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 提交审核
   *
   * @param dto
   * @return
   */
  @ApiOperation(value = "提交审核")
  @PostMapping(value = "/handlerSubmitAudit")
  public Result<?> handlerSubmitAudit(@RequestBody TerminalDto dto) {
    try {
      this.terminalVoService.submitAudit(dto);
      return Result.ok("提交审核成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据TerminalQueryDto获取对应的终端编码集合
   *
   * @param dto
   * @return
   */
  @ApiOperation(value = "根据TerminalQueryDto获取对应的终端编码集合")
  @PostMapping("/findByTerminalQueryDto")
  public Result<Set<String>> findByTerminalQueryDto(@RequestBody TerminalQueryDto dto) {
    try {
      return Result.ok(this.terminalVoService.findByTerminalQueryDto(dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据组织获取组织及下级所有的审核通过且未删除的终端信息
   *
   * @param orgCodes
   * @return
   */
  @ApiOperation(value = "根据组织获取组织及下级所有的审核通过且未删除的终端信息")
  @GetMapping("/findAllowSaleTerminalByOrgCodes")
  public Result<Map<String, Set<String>>> findAllowSaleTerminalByOrgCodes(
      @RequestParam("orgCodes") Set<String> orgCodes) {
    try {
      return Result.ok(this.terminalVoService.findAllowSaleTerminalByOrgCodes(orgCodes));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }


  /**
   * 根据高德id集合获取对应的终端信息
   *
   * @param amapIds
   * @return 客户信息集合
   */
  @ApiOperation(value = "根据高德id集合获取对应的终端信息")
  @GetMapping("/findByAmapIds")
  public Result<List<TerminalVo>> findByAmapIds(@RequestParam("amapIds") Set<String> amapIds) {
    try {
      return Result.ok(terminalVoService.findByAmapIds(amapIds));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }


  /**
   * 批量设置是否分利
   *
   * @param dto 请求参数dto
   *            其中  terminalCodeList 和  shareBenefits 必填数据
   */
  @ApiOperation(value = "批量设置是否分利")
  @PostMapping("/modifyShareBenefits")
  public Result<?> modifyShareBenefits(@RequestBody TerminalDto dto) {
    try {
      this.terminalVoService.modifyShareBenefits(dto);
      return Result.ok("修改成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据经纬度查询当前城市范围内的终端门店
   * @param dto
   * @return
   */
  @ApiOperation(value = "根据经纬度查询当前城市范围内的终端门店")
  @GetMapping("/findByLngAndLat")
  public Result<List<TerminalVo>> findByLngAndLat(@ApiParam(name = "dto", value = "门店经纬度查询dto") TerminalLugAndLatQueryDto dto){
    try {
      return Result.ok(terminalVoService.findByLngAndLat(dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 根据id集合获取终端信息
   */
  @ApiOperation(value = "根据id集合获取终端信息")
  @PostMapping("/findTerminalList")
  public Result<List<TerminalVo>> findTerminalList(@RequestBody TerminalNewSearchDto dto){
    try {
      List<TerminalVo> list = this.terminalVoService.findTerminalList(dto);
      return Result.ok(list);
    } catch (RuntimeException e) {
      log.error(e.getMessage(),e);
      return Result.error(e.getMessage());
    }
  }


}
