package com.biz.crm.mdm.business.terminal.local.service.internal;

import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.customer.user.sdk.service.CustomerUserVoService;
import com.biz.crm.mdm.business.customer.user.sdk.vo.CustomerUserVo;
import com.biz.crm.mdm.business.product.sdk.service.ProductVoService;
import com.biz.crm.mdm.business.product.sdk.vo.MaterialProductVo;
import com.biz.crm.mdm.business.terminal.local.entity.TerminalSupply;
import com.biz.crm.mdm.business.terminal.local.entity.TerminalSupplyDetail;
import com.biz.crm.mdm.business.terminal.local.repository.TerminalSupplyRepository;
import com.biz.crm.mdm.business.terminal.local.service.TerminalSupplyDetailService;
import com.biz.crm.mdm.business.terminal.local.service.TerminalSupplyService;
import com.biz.crm.mdm.business.terminal.sdk.dto.TerminalSupplyDto;
import com.biz.crm.mdm.business.terminal.sdk.event.TerminalSupplyPositionRegister;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalSupplyVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.SupplyPositionPageVo;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalSupplyDetailVo;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalSupplyVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 供货关系
 *
 * @author sunx
 * @date 2021/10/19
 */
@Service
public class TerminalSupplyVoServiceImpl implements TerminalSupplyVoService {

  @Autowired(required = false)
  private TerminalSupplyService terminalSupplyService;

  @Autowired(required = false)
  private TerminalSupplyDetailService terminalSupplyDetailService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private TerminalSupplyRepository terminalSupplyRepository;

  @Autowired(required = false)
  private ProductVoService productVoService;

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  @Autowired(required = false)
  private CustomerUserVoService customerUserVoService;

  @Autowired(required = false)
  private TerminalSupplyPositionRegister terminalSupplyPositionRegister;

  @Override
  public List<TerminalSupplyVo> findByTerminalCodes(List<String> terminalCodeList) {
    List<TerminalSupplyVo> re = Lists.newLinkedList();
    if (CollectionUtils.isEmpty(terminalCodeList)) {
      return re;
    }
    List<TerminalSupply> terminalSupplyList =
        terminalSupplyService.findByTerminalCodes(terminalCodeList);
    if (CollectionUtils.isEmpty(terminalSupplyList)) {
      return re;
    }
    List<String> supplyIdList =
        terminalSupplyList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getId()))
            .map(TerminalSupply::getId)
            .collect(Collectors.toList());
    List<TerminalSupplyDetail> detailList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(supplyIdList)) {
      detailList = terminalSupplyDetailService.findBySupplyIds(supplyIdList);
    }
    re = buildTerminalSupplyVoList(terminalSupplyList, detailList);
    return re;
  }

  @Override
  public List<TerminalSupplyVo> findByCustomerCodes(Set<String> customerCodeSet){
    List<TerminalSupplyVo> re = Lists.newLinkedList();
    if (CollectionUtils.isEmpty(customerCodeSet)) {
      return re;
    }
    List<TerminalSupply> terminalSupplyList =
      terminalSupplyService.findByCustomerCodes(customerCodeSet);
    if (CollectionUtils.isEmpty(terminalSupplyList)) {
      return re;
    }
    List<String> supplyIdList =
      terminalSupplyList.stream()
        .filter(a -> StringUtils.isNotBlank(a.getId()))
        .map(TerminalSupply::getId)
        .collect(Collectors.toList());
    List<TerminalSupplyDetail> detailList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(supplyIdList)) {
      detailList = terminalSupplyDetailService.findBySupplyIds(supplyIdList);
    }
    re = buildTerminalSupplyVoList(terminalSupplyList, detailList);
    return re;
  }

  @Override
  @Transactional
  public void saveBatch(List<TerminalSupplyDto> list, String terminalCode) {
    terminalSupplyService.saveBatch(list, terminalCode);
  }

  /**
   * 根据终端编码删除信息
   *
   * @param terminalCodes
   */
  @Override
  public void deleteByTerminalCodes(List<String> terminalCodes) {
    this.terminalSupplyService.deleteByTerminalCodes(terminalCodes);
  }

  @Override
  public Set<String> findTerminalSupplyCustomerCodeSet(String terminalCode, String materialCode) {
    if (StringUtils.isAnyBlank(terminalCode, materialCode)) {
      return Sets.newHashSet();
    }
    MaterialProductVo materialProductVo =
        this.productVoService.findMaterialProductVoByMaterialCode(materialCode);
    if (Objects.isNull(materialProductVo)
        || (CollectionUtils.isEmpty(materialProductVo.getProductCodeSet())
        && CollectionUtils.isEmpty(materialProductVo.getProductLevelCodeSet()))) {
      return Sets.newHashSet();
    }
    return this.terminalSupplyRepository.findTerminalSupplyCustomerCodeSet(
        terminalCode,
        materialProductVo.getProductCodeSet(),
        materialProductVo.getProductLevelCodeSet());
  }

  /**
   * 组件供货关系信息
   *
   * @param terminalSupplyList
   * @param detailList
   * @return
   */
  private List<TerminalSupplyVo> buildTerminalSupplyVoList(
      List<TerminalSupply> terminalSupplyList, List<TerminalSupplyDetail> detailList) {
    List<TerminalSupplyVo> re = Lists.newLinkedList();
    if (CollectionUtils.isEmpty(terminalSupplyList)) {
      return re;
    }
    re =
        (List<TerminalSupplyVo>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                terminalSupplyList,
                TerminalSupply.class,
                TerminalSupplyVo.class,
                HashSet.class,
                ArrayList.class);

    initSupplyUser(re);
    initSupplyPosition(re);
    initSupplyCustomer(re);

    Map<String, List<TerminalSupplyDetailVo>> map = Maps.newHashMap();
    if (CollectionUtils.isNotEmpty(detailList)) {
      List<TerminalSupplyDetailVo> list =
          (List<TerminalSupplyDetailVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  detailList,
                  TerminalSupplyDetail.class,
                  TerminalSupplyDetailVo.class,
                  HashSet.class,
                  ArrayList.class);
      map = list.stream().collect(Collectors.groupingBy(TerminalSupplyDetailVo::getSupplyId));
    }
    for (TerminalSupplyVo vo : re) {
      vo.setDetails(map.get(vo.getId()));
    }
    return re;
  }

  /**
   * 设置供货关系客户信息
   *
   * @param list
   */
  private void initSupplyCustomer(List<TerminalSupplyVo> list) {
    Set<String> customerCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getCustomerCode()))
            .map(TerminalSupplyVo::getCustomerCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(customerCodeSet)) {
      return;
    }
    List<CustomerVo> customerVoList =
        this.customerVoService.findByCustomerCodes(Lists.newArrayList(customerCodeSet));
    if (CollectionUtils.isEmpty(customerVoList)) {
      return;
    }
    // k-customerCode,v-customerName
    Map<String, String> map =
        customerVoList.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getCustomerCode(), a.getCustomerName()))
            .collect(
                Collectors.toMap(
                    CustomerVo::getCustomerCode, CustomerVo::getCustomerName, (a, b) -> a));
    for (TerminalSupplyVo item : list) {
      item.setCustomerName(map.get(item.getCustomerCode()));
    }
  }

  /**
   * 设置供货关系岗位信息
   *
   * @param list
   */
  private void initSupplyPosition(List<TerminalSupplyVo> list) {
    Set<String> positionCodeSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getPositionCode()))
            .map(TerminalSupplyVo::getPositionCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(positionCodeSet)
        || Objects.isNull(terminalSupplyPositionRegister)) {
      return;
    }
    List<SupplyPositionPageVo> vos =
        this.terminalSupplyPositionRegister.onRequestByPositionCodes(positionCodeSet);
    if (CollectionUtils.isEmpty(vos)) {
      return;
    }
    // k-positionCode,v-SupplyPositionPageVo
    Map<String, SupplyPositionPageVo> map =
        vos.stream()
            .filter(a -> StringUtils.isNotBlank(a.getPositionCode()))
            .collect(
                Collectors.toMap(
                    SupplyPositionPageVo::getPositionCode, Function.identity(), (a, b) -> a));
    for (TerminalSupplyVo item : list) {
      if (StringUtils.isBlank(item.getPositionCode()) || !map.containsKey(item.getPositionCode())) {
        continue;
      }
      SupplyPositionPageVo cur = map.get(item.getPositionCode());
      if (Objects.isNull(cur)) {
        continue;
      }
      item.setUnionName(cur.getUnionName());
      item.setFullName(cur.getFullName());
      item.setPositionName(cur.getPositionName());
      item.setOrgName(cur.getOrgName());
    }
  }

  /**
   * 设置供货关系客户用户信息
   *
   * @param list
   */
  private void initSupplyUser(List<TerminalSupplyVo> list) {
    Set<String> userNameSet =
        list.stream()
            .filter(a -> StringUtils.isNotBlank(a.getUserName()))
            .map(TerminalSupplyVo::getUserName)
            .collect(Collectors.toSet());
    if (CollectionUtils.isEmpty(userNameSet)) {
      return;
    }
    List<CustomerUserVo> userVoList = this.customerUserVoService.findByUserNames(userNameSet);
    if (CollectionUtils.isEmpty(userVoList)) {
      return;
    }
    // k-userName,v-fullName
    Map<String, String> map =
        userVoList.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getUserName(), a.getFullName()))
            .collect(
                Collectors.toMap(
                    CustomerUserVo::getUserName, CustomerUserVo::getFullName, (a, b) -> a));
    for (TerminalSupplyVo item : list) {
      item.setFullName(map.get(item.getUserName()));
    }
  }
}
