package com.biz.crm.mdm.business.terminal.local.service.internal;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.utils.DistanceUtil;
import com.biz.crm.mdm.business.customer.org.sdk.service.CustomerOrgVoSdkService;
import com.biz.crm.mdm.business.customer.org.sdk.vo.CustomerOrgVo;
import com.biz.crm.mdm.business.customer.sdk.dto.SupplyRelationshipDto;
import com.biz.crm.mdm.business.customer.sdk.enums.BuyerTypeEnum;
import com.biz.crm.mdm.business.customer.sdk.service.CustomerVoService;
import com.biz.crm.mdm.business.customer.sdk.service.SupplyRelationshipSdkService;
import com.biz.crm.mdm.business.customer.sdk.vo.CustomerVo;
import com.biz.crm.mdm.business.org.sdk.dto.OrgQueryDto;
import com.biz.crm.mdm.business.org.sdk.dto.RelateOrgCodeQueryDto;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.biz.crm.mdm.business.terminal.local.entity.*;
import com.biz.crm.mdm.business.terminal.local.repository.TerminalAddressRepository;
import com.biz.crm.mdm.business.terminal.local.repository.TerminalRelaOrgRepository;
import com.biz.crm.mdm.business.terminal.local.repository.TerminalRepository;
import com.biz.crm.mdm.business.terminal.local.service.*;
import com.biz.crm.mdm.business.terminal.sdk.constant.TerminalConstant;
import com.biz.crm.mdm.business.terminal.sdk.dto.*;
import com.biz.crm.mdm.business.terminal.sdk.enums.TerminalSupplyTypeEnum;
import com.biz.crm.mdm.business.terminal.sdk.enums.TerminalVoExtendEnum;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalSupplyVoService;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.*;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.Comparator;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 终端信息Vo接口 实现
 *
 * @author sunx
 * @date 2021/10/19
 */
@Slf4j
@Service
public class TerminalVoServiceImpl implements TerminalVoService {

  @Autowired(required = false)
  private TerminalService terminalService;

  @Autowired(required = false)
  private TerminalRelaOrgService terminalRelaOrgService;

  @Autowired(required = false)
  private TerminalRelaCustomerOrgService terminalRelaCustomerOrgService;

  @Autowired(required = false)
  private TerminalContactService terminalContactService;

  @Autowired(required = false)
  private TerminalSupplyVoService terminalSupplyVoService;

  @Autowired(required = false)
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private OrgVoService orgVoService;

  @Autowired(required = false)
  private CustomerOrgVoSdkService customerOrgVoSdkService;

  @Autowired(required = false)
  private CustomerVoService customerVoService;

  @Autowired(required = false)
  private TerminalPersonVoService personVoService;

  @Autowired(required = false)
  private TerminalRepository terminalRepository;

  @Autowired(required = false)
  private TerminalRelaOrgRepository terminalRelaOrgRepository;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  private TerminalTagService terminalTagService;

  @Autowired
  private SupplyRelationshipSdkService supplyRelationshipSdkService;

  @Autowired(required = false)
  private TerminalAddressRepository addressRepository;

  public static final String TERMINAL_PROCESS_NAME= "终端审批流程";

  /**
   * 基于数据库执行的数据视图执行内容缓存（最多500毫秒）
   */
  private static volatile Cache<String, List<TerminalVo>> cache = null;

  @Override
  public List<TerminalVo> findDetailsByIdsOrTerminalCodes(
      List<String> ids, List<String> terminalCodes) {
    List<TerminalVo> re = Lists.newLinkedList();
    if (CollectionUtils.isEmpty(ids) && CollectionUtils.isEmpty(terminalCodes)) {
      return re;
    }
    List<Terminal> terminalList =
        terminalService.findDetailsByIdsOrTerminalCodes(ids, terminalCodes);
    if (CollectionUtils.isEmpty(terminalList)) {
      return re;
    }
    re = this.convertEntityToVo(terminalList);
    return re;
  }

  @Override
  @Transactional
  public TerminalVo create(TerminalDto dto) {
    Validate.notNull(dto, "终端信息缺失");
    this.validateContacts(dto);
    // 如果userCode为空需要期初一个编码，否则执行验重逻辑
    if (StringUtils.isEmpty(dto.getTerminalCode())) {
      dto.setTerminalCode(
          generateCodeService.generateCode(TerminalConstant.TERMINAL_CODE, 1).get(0));
    } else {
      Integer count = this.terminalRepository.countByTerminalCode(dto.getTerminalCode());
      Validate.isTrue(null == count || 1 > count, dto.getTerminalCode() + "终端编码已存在");
    }
    Terminal terminal =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, Terminal.class, HashSet.class, ArrayList.class);
    terminal.setProcessStatus(ProcessStatusEnum.PREPARE.getDictCode());
    dto.setTerminalCode(terminal.getTerminalCode());

    //新增租户编号
    terminal.setTenantCode(TenantUtils.getTenantCode());
    this.terminalService.create(terminal);
    dto.setId(terminal.getId());
    this.bindExtInfo(dto);
    return this.buildByDtoAndTerminal(dto, terminal);
  }

  @Override
  @Transactional
  public TerminalVo createAndSubmit(TerminalDto dto) {
    Validate.notNull(dto, "终端信息缺失");
    this.validateContacts(dto);
    // 如果userCode为空需要期初一个编码，否则执行验重逻辑
    if (StringUtils.isEmpty(dto.getTerminalCode())) {
      dto.setTerminalCode(
          generateCodeService.generateCode(TerminalConstant.TERMINAL_CODE, 1).get(0));
    } else {
      Integer count = this.terminalRepository.countByTerminalCode(dto.getTerminalCode());
      Validate.isTrue(null == count || 1 > count, dto.getTerminalCode() + "终端编码已存在");
    }
    Terminal terminal =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, Terminal.class, HashSet.class, ArrayList.class);
    dto.setTerminalCode(terminal.getTerminalCode());
    this.bindExtInfo(dto);
    //新增租户编号
    terminal.setTenantCode(TenantUtils.getTenantCode());
    this.terminalService.create(terminal);
    TerminalVo re = this.buildByDtoAndTerminal(dto, terminal);
    // 接入工作流,审批状态是否与通知等事务有关请处理
    dto.setId(terminal.getId());
    terminal.setProcessNumber(this.commitProcess(dto));
    terminal.setProcessStatus(ProcessStatusEnum.COMMIT.getDictCode());
    //重构修改方法
    this.terminalRepository.updateByIdAndTenantCode(terminal,TenantUtils.getTenantCode());
    return re;
  }

  @Override
  @Transactional
  public TerminalVo update(TerminalDto dto) {
    Validate.notNull(dto, "终端信息缺失");
    this.validateContacts(dto);
    Terminal terminal =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, Terminal.class, HashSet.class, ArrayList.class);
    dto.setTerminalCode(terminal.getTerminalCode());
    this.bindExtInfo(dto);
    this.terminalService.update(terminal);
    return this.buildByDtoAndTerminal(dto, terminal);
  }

  @Override
  public List<TerminalVo> findSelectByKeyword(String keyword) {
    Pageable pageable = PageRequest.of(0, 20);
    TerminalPaginationDto dto = new TerminalPaginationDto();
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    dto.setKeyword(keyword);
    Page<Terminal> page = this.terminalService.findByConditions(pageable, dto);
    if (Objects.isNull(page) || CollectionUtils.isEmpty(page.getRecords())) {
      return Lists.newLinkedList();
    }
    List<TerminalVo> list = Lists.newArrayList();
    for (Terminal item : page.getRecords()) {
      TerminalVo cur = new TerminalVo();
      cur.setTerminalCode(item.getTerminalCode());
      cur.setTerminalName(item.getTerminalName());
      list.add(cur);
    }
    return list;
  }

  @Override
  public List<TerminalVo> findMainDetailsByTerminalCodesUsePost(List<String> terminalCodeList) {
    return this.findMainDetailsByTerminalCodes(terminalCodeList);
  }

  @Override
  public List<TerminalVo> findTerminalAndContactByTerminalCodes(List<String> terminalCodeList) {
    if (CollectionUtils.isEmpty(terminalCodeList)) {
      return Lists.newLinkedList();
    }
    List<Terminal> terminalList = this.terminalService.findDetailsByIdsOrTerminalCodes(null, terminalCodeList);
    if (CollectionUtils.isEmpty(terminalList)) {
      return Lists.newLinkedList();
    }
    // 联系人
    List<TerminalContact> terminalContactList = this.terminalContactService.findByTerminalCodes(terminalCodeList);
    List<TerminalContactVo> terminalContactVos = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(terminalContactVos)) {
      terminalContactVos = (List<TerminalContactVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
              terminalContactList, TerminalContact.class, TerminalContactVo.class, HashSet.class, ArrayList.class);
    }
    Map<String, List<TerminalContactVo>> terminalContactVoMap = terminalContactVos.stream()
            .collect(Collectors.groupingBy(TerminalContactVo::getTerminalCode));
    List<TerminalVo> terminalVos = (List<TerminalVo>) this.nebulaToolkitService.copyCollectionByBlankList(
            terminalList, Terminal.class, TerminalVo.class, HashSet.class, ArrayList.class);
    terminalVos.forEach(terminalVo -> terminalVo.setContacts(terminalContactVoMap.get(terminalVo.getTerminalCode())));
    return terminalVos;
  }

  @Override
  public List<TerminalVo> findMainDetailsByTerminalCodes(List<String> terminalCodeList) {
    if (CollectionUtils.isEmpty(terminalCodeList)) {
      return Lists.newLinkedList();
    }
    List<Terminal> terminalList =
        this.terminalService.findDetailsByIdsOrTerminalCodes(null, terminalCodeList);
    if (CollectionUtils.isEmpty(terminalList)) {
      return Lists.newLinkedList();
    }
    List<TerminalRelaOrg> terminalRelaOrgList =
        this.terminalRelaOrgService.findByTerminalCodes(terminalCodeList);
    // k-terminalCode,v-orgCodeList
    Map<String, List<String>> map =
        terminalRelaOrgList.stream()
            .filter(a -> StringUtils.isNoneBlank(a.getTerminalCode(), a.getOrgCode()))
            .collect(
                Collectors.groupingBy(
                    TerminalRelaOrg::getTerminalCode,
                    Collectors.mapping(TerminalRelaOrg::getOrgCode, Collectors.toList())));
    Set<String> orgCodeSet =
        terminalRelaOrgList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getOrgCode()))
            .map(TerminalRelaOrg::getOrgCode)
            .collect(Collectors.toSet());
    List<OrgVo> orgVoList = this.orgVoService.findByOrgCodes(Lists.newArrayList(orgCodeSet));
    // k-orgCode,v-orgName
    Map<String, String> mapOrg = Maps.newHashMap();
    if (CollectionUtils.isNotEmpty(orgVoList)) {
      mapOrg =
          orgVoList.stream()
              .filter(a -> StringUtils.isNoneBlank(a.getOrgCode(), a.getOrgName()))
              .collect(Collectors.toMap(OrgVo::getOrgCode, OrgVo::getOrgName, (a, b) -> a));
    }
    // 查看联系人
    List<TerminalContact> terminalContactList =
        this.terminalContactService.findByTerminalCodes(terminalCodeList);
    Map<String, List<TerminalContactVo>> terminalContactVoMap = new HashMap<>();
    if (!CollectionUtils.isEmpty(terminalContactList)) {
      List<TerminalContactVo> terminalContactVos =
          (List<TerminalContactVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  terminalContactList,
                  TerminalContact.class,
                  TerminalContactVo.class,
                  HashSet.class,
                  ArrayList.class);
      terminalContactVoMap =
          terminalContactVos.stream()
              .collect(Collectors.groupingBy(TerminalContactVo::getTerminalCode));
    }
    // 查看客户组织
    List<TerminalRelaCustomerOrg> terminalRelaCustomerOrgs =
        this.terminalRelaCustomerOrgService.findByTerminalCodes(terminalCodeList);
    Map<String, List<TerminalRelaCustomerOrgVo>> terminalRelaCustomerOrgVoMap = new HashMap<>();
    // k-orgCode, v-orgName
    Map<String, String> customerOrgVoMap = new HashMap<>();
    if (!CollectionUtils.isEmpty(terminalRelaCustomerOrgs)) {
      List<TerminalRelaCustomerOrgVo> terminalRelaCustomerOrgVos =
          (List<TerminalRelaCustomerOrgVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  terminalRelaCustomerOrgs,
                  TerminalRelaCustomerOrg.class,
                  TerminalRelaCustomerOrgVo.class,
                  HashSet.class,
                  ArrayList.class);
      terminalRelaCustomerOrgVoMap =
          terminalRelaCustomerOrgVos.stream()
              .collect(Collectors.groupingBy(TerminalRelaCustomerOrgVo::getTerminalCode));
      List<CustomerOrgVo> customerOrgVos =
          this.customerOrgVoSdkService.findListByCodes(
              terminalRelaCustomerOrgVos.stream()
                  .map(TerminalRelaCustomerOrgVo::getOrgCode)
                  .collect(Collectors.toList()));
      if (!CollectionUtils.isEmpty(customerOrgVos)) {
        customerOrgVoMap =
            customerOrgVos.stream()
                .filter(
                    a -> StringUtils.isNoneBlank(a.getCustomerOrgCode(), a.getCustomerOrgName()))
                .collect(
                    Collectors.toMap(
                        CustomerOrgVo::getCustomerOrgCode, CustomerOrgVo::getCustomerOrgName));
      }
    }

    List<TerminalVo> re =
        (List<TerminalVo>)
            this.nebulaToolkitService.copyCollectionByBlankList(
                terminalList, Terminal.class, TerminalVo.class, HashSet.class, ArrayList.class);
    for (TerminalVo item : re) {
      List<String> curOrgCodeList = map.get(item.getTerminalCode());
      if (!CollectionUtils.isEmpty(curOrgCodeList)) {
        List<TerminalRelaOrgVo> curList = Lists.newLinkedList();
        List<String> orgNameList = Lists.newLinkedList();
        for (String orgCode : curOrgCodeList) {
          TerminalRelaOrgVo cur = new TerminalRelaOrgVo();
          cur.setTerminalCode(item.getTerminalCode());
          cur.setOrgCode(orgCode);
          cur.setOrgName(mapOrg.get(orgCode));
          curList.add(cur);
          orgNameList.add(cur.getOrgName());
        }
        item.setOrgList(curList);
        item.setOrgNameStr(StringUtils.join(orgNameList, ","));
      }
      // 赋值联系人
      List<TerminalContactVo> terminalContactVos = terminalContactVoMap.get(item.getTerminalCode());
      if (!CollectionUtils.isEmpty(terminalContactVos)) {
        item.setContacts(terminalContactVos);
      }
      // 赋值客户组织
      List<TerminalRelaCustomerOrgVo> terminalRelaCustomerOrgVos =
          terminalRelaCustomerOrgVoMap.get(item.getTerminalCode());
      if (!CollectionUtils.isEmpty(terminalRelaCustomerOrgVos)) {
        for (TerminalRelaCustomerOrgVo terminalRelaCustomerOrgVo : terminalRelaCustomerOrgVos) {
          terminalRelaCustomerOrgVo.setOrgName(
              customerOrgVoMap.get(terminalRelaCustomerOrgVo.getOrgCode()));
        }
        item.setCustomerOrgList(terminalRelaCustomerOrgVos);
      }
    }
    return re;
  }

  @Override
  public Page<TerminalVo> findByTerminalPaginationDto(
      Pageable pageable, TerminalPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new TerminalPaginationDto());
    Page<Terminal> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<Terminal> entityPage = this.terminalRepository.findByConditions(page, dto);
    Page<TerminalVo> pageResult =
        new Page<>(entityPage.getCurrent(), entityPage.getSize(), entityPage.getTotal());
    if (CollectionUtils.isEmpty(entityPage.getRecords())) {
      return pageResult;
    }
    pageResult.setRecords(this.convertEntityToVo(entityPage.getRecords()));
    return pageResult;
  }

  @Override
  public Set<String> findByTerminalCodeSearchDto(TerminalCodeSearchDto dto) {
    dto = Optional.ofNullable(dto).orElse(new TerminalCodeSearchDto());
    return this.terminalService.findByTerminalCodeSearchDto(dto);
  }

  @Override
  public List<TerminalVo> findByAmapIds(Set<String> amapIds) {
    String cacheKey = StringUtils.join(amapIds);
    if(cache == null) {
      synchronized (TerminalVoServiceImpl.class) {
        while(cache == null) {
          cache = CacheBuilder.newBuilder()
                  .initialCapacity(10000)
                  .expireAfterWrite(300, TimeUnit.MILLISECONDS)
                  .maximumSize(100000)
                  .build();
        }
      }
    }
    List<TerminalVo> graph = cache.getIfPresent(cacheKey);
    if (graph == null) {
      graph = findByAmapIdsFromDB(amapIds);
      cache.put(cacheKey, graph);
    }
    return graph;
  }

  private List<TerminalVo> findByAmapIdsFromDB(Set<String> amapIds) {
    log.warn(" db 查询 findByAmapIds ");
    if (CollectionUtils.isEmpty(amapIds)) {
      return Lists.newArrayList();
    }
    List<Terminal> list = this.terminalRepository.findByAmapIds(amapIds);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newArrayList();
    }
    return (List<TerminalVo>)
        this.nebulaToolkitService.copyCollectionByWhiteList(
            list, Terminal.class, TerminalVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public void submitAudit(TerminalDto dto) {
    Validate.notNull(dto, "参数不能为空");
    Validate.notBlank(dto.getId(), "终端id不能为空");
    final Terminal terminal = this.terminalRepository.findById(dto.getId());
    Validate.notNull(terminal, "终端信息不存在");
    Set<String> set =
        Sets.newHashSet(
            ProcessStatusEnum.PREPARE.getDictCode(),
            ProcessStatusEnum.REJECT.getDictCode(),
            ProcessStatusEnum.RECOVER.getDictCode());
    Validate.isTrue(set.contains(terminal.getProcessStatus()), "终端状态不能执行提交审核");
    dto.setTerminalCode(terminal.getTerminalCode());
    // todo 接入工作流,审批状态是否与通知等事务有关请处理
    terminal.setProcessNumber(this.commitProcess(dto));
    terminal.setProcessStatus(ProcessStatusEnum.COMMIT.getDictCode());
    ProcessBusinessDto processBusiness = dto.getProcessBusiness();
    terminal.setProcessRemark(processBusiness.getRemark());
    terminal.setProcessKey(processBusiness.getProcessKey());
    this.terminalRepository.updateByIdAndTenantCode(terminal,TenantUtils.getTenantCode());
  }

  @Override
  public Set<String> findByTerminalQueryDto(TerminalQueryDto dto) {
    if (Objects.isNull(dto)
        || StringUtils.isAllBlank(
            dto.getTerminalCode(),
            dto.getTerminalName(),
            dto.getChannel(),
            dto.getOrgName(),
            dto.getDelFlag(),
            dto.getProcessStatus())) {
      return Sets.newHashSet();
    }

    if (StringUtils.isNotBlank(dto.getOrgName())) {
      final OrgQueryDto queryDto = new OrgQueryDto();
      queryDto.setOrgName(dto.getOrgName());
      final Set<String> orgCodeSet = this.orgVoService.findByOrgQueryDto(queryDto);
      if (CollectionUtils.isEmpty(orgCodeSet)) {
        return Sets.newHashSet();
      }
      dto.setOrgCodeSet(orgCodeSet);
    }
    return this.terminalRepository.findByTerminalQueryDto(dto);
  }

  @Override
  public Map<String, Set<String>> findAllowSaleTerminalByOrgCodes(Set<String> orgCodes) {
    if (org.springframework.util.CollectionUtils.isEmpty(orgCodes)) {
      return Maps.newHashMap();
    }
    final RelateOrgCodeQueryDto queryDto = new RelateOrgCodeQueryDto();
    queryDto.setOrgCodeSet(orgCodes);
    queryDto.setSearchType(-1);
    final Map<String, String> orgRuleMap = this.orgVoService.findByRelateOrgCodeQueryDto(queryDto);
    if (org.springframework.util.CollectionUtils.isEmpty(orgRuleMap)) {
      return Maps.newHashMap();
    }
    List<TerminalRelaOrg> list =
        this.terminalRelaOrgRepository.findAllowSaleTerminalByOrgCodes(orgRuleMap.keySet());
    if (CollectionUtils.isEmpty(list)) {
      return Maps.newHashMap();
    }
    Map<String, Set<String>> map =
        list.stream()
            .filter(
                a ->
                    StringUtils.isNoneBlank(a.getTerminalCode(), a.getOrgCode())
                        && orgRuleMap.keySet().contains(a.getOrgCode()))
            .collect(
                Collectors.groupingBy(
                    TerminalRelaOrg::getTerminalCode,
                    Collectors.mapping(TerminalRelaOrg::getOrgCode, Collectors.toSet())));
    Map<String, Set<String>> re = Maps.newHashMap();
    for (Entry<String, Set<String>> item : map.entrySet()) {
      Set<String> rule = Sets.newHashSet();
      for (String orgCode : item.getValue()) {
        final String s = orgRuleMap.get(orgCode);
        if (StringUtils.isBlank(s)) {
          continue;
        }
        rule.add(s);
      }
      if (CollectionUtils.isEmpty(rule)) {
        continue;
      }
      re.put(item.getKey(), rule);
    }
    return re;
  }


  @Override
  public List<TerminalVo> findByTerminalSearchDto(TerminalSearchDto dto) {
    dto = Optional.ofNullable(dto).orElse(new TerminalSearchDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(dto.getOrgCodeSet())
        && CollectionUtils.isEmpty(dto.getChannelSet())
        && CollectionUtils.isEmpty(dto.getTagSet())
        && CollectionUtils.isEmpty(dto.getTerminalCodeSet())) {
      return Lists.newArrayList();
    }
    List<Terminal> terminalList = this.terminalRepository.findByTerminalSearchDto(dto);
    if (CollectionUtils.isEmpty(terminalList)) {
      return Lists.newArrayList();
    }
    return (List<TerminalVo>) this.nebulaToolkitService.copyCollectionByBlankList(
            terminalList, Terminal.class, TerminalVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 1.设置终端分利状态
   * 2.查询所关联的经销商信息
   * 3.终端设置为分利终端时
   *  3.1 判断关联的供货经销商分利判断，为空 或 false 均为非分利经销商
   * 4.终端设置为非分利终端时
   *  4.1 查询每个经销商所关联的终端；
   *  4.2 判断当前经销商关联的终端是否与本次传入的终端完全一致，如果是则修改经销商状态
   *  4.3 获得true状态的返利终端，再反推出不做修改的经销商信息，剩余的经销商即为需要修改状态
   * 5.更新经销商状态
   *
   * @param dto 参数dto
   */
  @Override
  @Transactional
  public void modifyShareBenefits(TerminalDto dto) {
    // 1
    if (ObjectUtil.isNotEmpty(dto) && CollUtil.isNotEmpty(dto.getTerminalCodeList()) && ObjectUtil.isNotEmpty(dto.getShareBenefits())) {
      terminalRepository.modifyShareBenefits(dto.getTerminalCodeList(), dto.getShareBenefits());
    }
    // 2
    List<TerminalSupplyVo> terminalSupplyVoList = terminalSupplyVoService.findByTerminalCodes(dto.getTerminalCodeList());
    if (CollUtil.isEmpty(terminalSupplyVoList)){
      // 没有关联经销商等，无需执行以下逻辑
      return;
    }
    Map<String, List<TerminalSupplyVo>> mapKeyCustomerCode = Maps.newHashMap();
    List<String> list = Lists.newArrayList();
    if (CollectionUtils.isNotEmpty(terminalSupplyVoList)) {
      mapKeyCustomerCode = terminalSupplyVoList.stream().collect(Collectors.groupingBy(TerminalSupplyVo::getCustomerCode));
    }
    Set<String> keySet = mapKeyCustomerCode.keySet();
    List<CustomerVo> customerVoList = customerVoService.findByCustomerCodes(Lists.newArrayList(keySet));
    if (dto.getShareBenefits()) { // 3
      // 3.1
      if (CollUtil.isNotEmpty(customerVoList)) {
        List<CustomerVo> collect = customerVoList.stream()
          .filter(item -> (ObjectUtil.isEmpty(item.getShareBenefits()) || !item.getShareBenefits()))
          .collect(Collectors.toList());
        if (CollUtil.isNotEmpty(collect)) {
          // 取出需更新状态的经销商编码集合
          list = collect.stream().map(CustomerVo::getCustomerCode).collect(Collectors.toList());
        }
      }
    } else { // 4
      // 4.1
      List<TerminalSupplyVo> terminalSupplyVos = terminalSupplyVoService.findByCustomerCodes(keySet);
      // 4.2
      Set<String> stringList = terminalSupplyVos.stream().map(TerminalSupplyVo::getTerminalCode).collect(Collectors.toSet());
      if (dto.getTerminalCodeList().size() == stringList.size()
        && dto.getTerminalCodeList().containsAll(stringList)) {
        list.addAll(keySet);
      } else {
        // 4.3
        List<Terminal> terminalList = new ArrayList<>();
        if (CollUtil.isNotEmpty(terminalSupplyVos)) {
          List<String> str = terminalSupplyVos.stream().map(TerminalSupplyVo::getTerminalCode).collect(Collectors.toList());
          terminalList = terminalService.findByTerminalCodes(str);
        }
        if (CollUtil.isNotEmpty(terminalList)) {
          // 取出分利状态为true的终端，这些终端相关的经销商不做状态修改
          List<Terminal> collect = terminalList.stream()
            .filter(item -> (ObjectUtil.isNotEmpty(item.getShareBenefits()) && item.getShareBenefits()))
            .collect(Collectors.toList());
          Map<String, List<TerminalSupplyVo>> listMap = terminalSupplyVos.stream().collect(Collectors.groupingBy(TerminalSupplyVo::getTerminalCode));
          Set<String> set = listMap.keySet();
          if (CollUtil.isNotEmpty(collect)) {
            List<String> terminalCodes = collect.stream().map(Terminal::getTerminalCode).collect(Collectors.toList());
            terminalCodes.forEach(set::remove);
          }
          for (String key : set) {
            list.addAll(listMap.get(key).stream().map(TerminalSupplyVo::getCustomerCode).collect(Collectors.toSet()));
          }
        }
      }
    }
    // 5.
    if (CollUtil.isNotEmpty(list)) {
      customerVoService.modifyShareBenefits(list, dto.getShareBenefits());
    }
  }

  /**
   * 保存扩展信息
   *
   * @param dto
   */
  private void bindExtInfo(TerminalDto dto) {
    /*
     * 保存终端关联信息：
     * 1、保存关联组织信息
     * 2、保存关联联系人信息
     * 3、保存关联供货关系信息
     * */
    Validate.notNull(dto, "终端信息缺失");
    List<TerminalRelaOrg> orgList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getOrgList())) {
      for (TerminalRelaOrgDto item : dto.getOrgList()) {
        item.setTerminalCode(dto.getTerminalCode());
      }
      orgList =
          (List<TerminalRelaOrg>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getOrgList(),
                  TerminalRelaOrgDto.class,
                  TerminalRelaOrg.class,
                  HashSet.class,
                  ArrayList.class);
      orgList.forEach(a -> {
        a.setTerminalCode(dto.getTerminalCode());
        a.setTenantCode(TenantUtils.getTenantCode());
        a.setTerminalId(dto.getId());
        a.setSourcesId(dto.getSourcesId());
        a.setTerminalCode(dto.getTerminalName());
      });

    }
    terminalRelaOrgService.saveBatch(orgList, dto.getTerminalCode());

    List<TerminalRelaCustomerOrg> customerOrgList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getCustomerOrgList())) {
      for (TerminalRelaCustomerOrgDto item : dto.getCustomerOrgList()) {
        item.setTerminalCode(dto.getTerminalCode());
      }
      customerOrgList =
          (List<TerminalRelaCustomerOrg>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getCustomerOrgList(),
                  TerminalRelaCustomerOrgDto.class,
                  TerminalRelaCustomerOrg.class,
                  HashSet.class,
                  ArrayList.class);
      customerOrgList.forEach(a -> a.setTerminalCode(dto.getTerminalCode()));
      //新增租户编号
      customerOrgList.forEach(a -> a.setTenantCode(TenantUtils.getTenantCode()));
    }
    terminalRelaCustomerOrgService.saveBatch(customerOrgList, dto.getTerminalCode());

    List<TerminalContact> contactList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getContacts())) {
      for (TerminalContactDto item : dto.getContacts()) {
        item.setTerminalCode(dto.getTerminalCode());
      }
      contactList =
          (List<TerminalContact>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getContacts(),
                  TerminalContactDto.class,
                  TerminalContact.class,
                  HashSet.class,
                  ArrayList.class);
      contactList.forEach(a -> a.setTerminalCode(dto.getTerminalCode()));
      //新增租户编号
      contactList.forEach(a -> a.setTenantCode(TenantUtils.getTenantCode()));
    }
    terminalContactService.saveBatch(contactList, dto.getTerminalCode());

    if (CollectionUtils.isNotEmpty(dto.getSupplys())) {
      for (TerminalSupplyDto item : dto.getSupplys()) {
        item.setTerminalCode(dto.getTerminalCode());
        //新增租户编号
        item.setTenantCode(TenantUtils.getTenantCode());
      }
    }
    terminalSupplyVoService.saveBatch(dto.getSupplys(), dto.getTerminalCode());

    List<TerminalPersonVo> personVos = dto.getPersonVos();
    if (CollectionUtils.isNotEmpty(personVos)) {
      for (TerminalPersonVo item : personVos) {
        item.setTerminalId(dto.getTerminalId());
        item.setSourcesId(dto.getSourcesId());
        item.setTerminalName(dto.getTerminalName());
        item.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        item.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        //新增租户编号
        item.setTenantCode(TenantUtils.getTenantCode());
      }
    }

    personVoService.saveBatch(personVos, dto.getTerminalId());

    List<SupplyRelationshipDto> relationshipList = dto.getRelationshipList();
    if (CollectionUtils.isNotEmpty(relationshipList)) {
      for (SupplyRelationshipDto item : relationshipList) {
        item.setBuyerId(dto.getId());
        item.setBuyerSourceId(dto.getSourcesId());
        item.setBuyerCode(dto.getTerminalCode());
        item.setBuyerName(dto.getTerminalName());
        item.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        item.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        item.setBuyerType(BuyerTypeEnum.TERMINAL.getCode());
        item.setBuyerSmallType(dto.getChannel());
        item.setBuyerClassify(dto.getTerminalClassifyId());
        //新增租户编号
        item.setTenantCode(TenantUtils.getTenantCode());
      }
      supplyRelationshipSdkService.savaOrUpdate(relationshipList,dto.getId());
    }
  }

  /**
   * 构建返回信息
   *
   * @param dto
   * @param terminal
   * @return
   */
  private TerminalVo buildByDtoAndTerminal(TerminalDto dto, Terminal terminal) {
    TerminalVo vo =
        this.nebulaToolkitService.copyObjectByWhiteList(
            terminal, TerminalVo.class, HashSet.class, ArrayList.class);
    if (CollectionUtils.isNotEmpty(dto.getOrgList())) {
      vo.setOrgList(
          (List<TerminalRelaOrgVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getOrgList(),
                  TerminalRelaOrgDto.class,
                  TerminalRelaOrgVo.class,
                  HashSet.class,
                  ArrayList.class));
    }
    if (CollectionUtils.isNotEmpty(dto.getCustomerOrgList())) {
      vo.setCustomerOrgList(
          (List<TerminalRelaCustomerOrgVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getCustomerOrgList(),
                  TerminalRelaCustomerOrgDto.class,
                  TerminalRelaCustomerOrgVo.class,
                  HashSet.class,
                  ArrayList.class));
    }
    if (CollectionUtils.isNotEmpty(dto.getContacts())) {
      vo.setContacts(
          (List<TerminalContactVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getContacts(),
                  TerminalContactDto.class,
                  TerminalContactVo.class,
                  HashSet.class,
                  ArrayList.class));
    }
    if (CollectionUtils.isNotEmpty(dto.getSupplys())) {
      vo.setSupplys(
          (List<TerminalSupplyVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getSupplys(),
                  TerminalSupplyDto.class,
                  TerminalSupplyVo.class,
                  HashSet.class,
                  ArrayList.class));
    }
    return vo;
  }

  private List<TerminalVo> buildTerminalVoList(
      List<Terminal> terminalList,
      List<TerminalRelaOrg> terminalRelaOrgList,
      List<TerminalRelaCustomerOrg> terminalRelaCustomerOrgList,
      List<TerminalContact> terminalContactList,
      List<TerminalSupplyVo> terminalSupplyVoList,
      List<TerminalTag> terminalTags,
      List<TerminalPersonEntity> terminalPersonList) {
    List<TerminalVo> re = Lists.newLinkedList();
    if (CollectionUtils.isEmpty(terminalList)) {
      return re;
    }
    re =
        (List<TerminalVo>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                terminalList, Terminal.class, TerminalVo.class, HashSet.class, ArrayList.class);

    Map<String, List<TerminalRelaOrgVo>> mapOrg = Maps.newHashMap();
    Map<String, List<TerminalRelaCustomerOrgVo>> mapCustomerOrg = Maps.newHashMap();
    Map<String, List<TerminalContactVo>> mapContact = Maps.newHashMap();
    Map<String, List<TerminalSupplyVo>> mapSupply = Maps.newHashMap();
    Map<String, List<TerminalTagVo>> tagVos = Maps.newHashMap();
    Map<String, List<TerminalPersonVo>> personMap = Maps.newHashMap();

    if (CollectionUtils.isNotEmpty(terminalRelaOrgList)) {
      List<TerminalRelaOrgVo> list =
          (List<TerminalRelaOrgVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  terminalRelaOrgList,
                  TerminalRelaOrg.class,
                  TerminalRelaOrgVo.class,
                  HashSet.class,
                  ArrayList.class);
      mapOrg = list.stream().collect(Collectors.groupingBy(TerminalRelaOrgVo::getTerminalCode));
    }

    if (CollectionUtils.isNotEmpty(terminalRelaCustomerOrgList)) {
      List<TerminalRelaCustomerOrgVo> list =
          (List<TerminalRelaCustomerOrgVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  terminalRelaCustomerOrgList,
                  TerminalRelaCustomerOrg.class,
                  TerminalRelaCustomerOrgVo.class,
                  HashSet.class,
                  ArrayList.class);
      mapCustomerOrg =
          list.stream().collect(Collectors.groupingBy(TerminalRelaCustomerOrgVo::getTerminalCode));
    }

    if (CollectionUtils.isNotEmpty(terminalContactList)) {
      List<TerminalContactVo> list =
          (List<TerminalContactVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  terminalContactList,
                  TerminalContact.class,
                  TerminalContactVo.class,
                  HashSet.class,
                  ArrayList.class);
      mapContact = list.stream().collect(Collectors.groupingBy(TerminalContactVo::getTerminalCode));
    }

    // 标签
    if (CollectionUtils.isNotEmpty(terminalTags)) {
      List<TerminalTagVo> list =
        (List<TerminalTagVo>)
          this.nebulaToolkitService.copyCollectionByWhiteList(
            terminalTags,
            TerminalTag.class,
            TerminalTagVo.class,
            HashSet.class,
            ArrayList.class);
      tagVos = list.stream().collect(Collectors.groupingBy(TerminalTagVo::getTerminalCode));
    }

    if (CollectionUtils.isNotEmpty(terminalSupplyVoList)) {
      mapSupply =
          terminalSupplyVoList.stream()
              .collect(Collectors.groupingBy(TerminalSupplyVo::getTerminalCode));
    }
    if(CollectionUtils.isNotEmpty(terminalPersonList)){
      List<TerminalPersonVo> list =
          (List<TerminalPersonVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  terminalPersonList,
                  TerminalPersonEntity.class,
                  TerminalPersonVo.class,
                  HashSet.class,
                  ArrayList.class);
      personMap = list.stream().collect(Collectors.groupingBy(TerminalPersonVo::getTerminalId));
    }
    for (TerminalVo vo : re) {
      vo.setOrgList(mapOrg.get(vo.getTerminalCode()));
      vo.setCustomerOrgList(mapCustomerOrg.get(vo.getTerminalCode()));
      vo.setContacts(mapContact.get(vo.getTerminalCode()));
      vo.setSupplys(mapSupply.get(vo.getTerminalCode()));
      vo.setTagVos(tagVos.get(vo.getTerminalCode()));
      vo.setPersonVos(personMap.get(vo.getTerminalId()));
    }
    return re;
  }

  /**
   * 校验联系人信息
   *
   * @param dto
   */
  private void validateContacts(TerminalDto dto) {
    Validate.notNull(dto, "终端信息缺失");
    Validate.notEmpty(dto.getTerminalCode(), "终端编码不能为空");
    List<TerminalContactDto> contacts = dto.getContacts();
    if (CollectionUtils.isEmpty(contacts)) {
      return;
    }
    // 校验字段
    for (TerminalContactDto contact : contacts) {
      if (StringUtils.isNotBlank(contact.getContactName())
          || StringUtils.isNotBlank(contact.getContactPhone())
          || !Objects.isNull(contact.getContactMain())) {
        Validate.notBlank(contact.getContactPhone(), "缺失联系电话");
        Validate.notNull(contact.getContactMain(), "请选择是否是主联系人");
        Validate.notBlank(contact.getContactName(), "缺失姓名");
      }
    }
    // 校验是否存在主联系人
    List<TerminalContactDto> contanctsByMain =
        contacts.stream().filter(o -> Boolean.TRUE.equals(o.getContactMain())).collect(Collectors.toList());
    Validate.isTrue(contanctsByMain.size() == 1, "必须有且仅有一个主联系人");
  }

  /**
   * 转换终端实体为终端信息Vo,并完善关联信息
   *
   * @param terminalList 终端实体列表
   * @return 终端信息Vo列表
   */
  private List<TerminalVo> convertEntityToVo(List<Terminal> terminalList) {
    List<TerminalVo> re = Lists.newArrayList();
    List<String> terminalCodeList =
        terminalList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getTerminalCode()))
            .map(Terminal::getTerminalCode)
            .collect(Collectors.toList());

    List<String> terminalIdList =
        terminalList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getTerminalId()))
            .map(Terminal::getTerminalCode)
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(terminalCodeList)) {
      return re;
    }

    List<TerminalRelaOrg> terminalRelaOrgList =
        terminalRelaOrgService.findByTerminalCodes(terminalCodeList);
    List<TerminalRelaCustomerOrg> terminalRelaCustomerOrgList =
        terminalRelaCustomerOrgService.findByTerminalCodes(terminalCodeList);

    Set<String> orgCodeSet =
        terminalRelaOrgList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getOrgCode()))
            .map(TerminalRelaOrg::getOrgCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isNotEmpty(orgCodeSet)) {
      List<OrgVo> orgVoList = this.orgVoService.findByOrgCodes(Lists.newArrayList(orgCodeSet));
      orgVoList = CollectionUtils.isEmpty(orgVoList) ? Lists.newLinkedList() : orgVoList;
      Map<String, String> map =
          orgVoList.stream()
              .filter(a -> StringUtils.isNoneBlank(a.getOrgCode(), a.getOrgName()))
              .collect(Collectors.toMap(OrgVo::getOrgCode, OrgVo::getOrgName, (a, b) -> a));
      terminalRelaOrgList.forEach(a -> a.setOrgName(map.get(a.getOrgCode())));
    }

    Set<String> customerOrgCodeSet =
        terminalRelaCustomerOrgList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getOrgCode()))
            .map(TerminalRelaCustomerOrg::getOrgCode)
            .collect(Collectors.toSet());
    if (CollectionUtils.isNotEmpty(customerOrgCodeSet)) {
      List<CustomerOrgVo> customerOrgVoList =
          this.customerOrgVoSdkService.findListByCodes(Lists.newArrayList(customerOrgCodeSet));
      customerOrgVoList =
          CollectionUtils.isEmpty(customerOrgVoList) ? Lists.newLinkedList() : customerOrgVoList;
      Map<String, String> map =
          customerOrgVoList.stream()
              .filter(a -> StringUtils.isNoneBlank(a.getCustomerOrgCode(), a.getCustomerOrgName()))
              .collect(
                  Collectors.toMap(
                      CustomerOrgVo::getCustomerOrgCode,
                      CustomerOrgVo::getCustomerOrgName,
                      (a, b) -> a));
      terminalRelaCustomerOrgList.forEach(a -> a.setOrgName(map.get(a.getOrgCode())));
    }

    List<TerminalPersonEntity> terminalPersonList= personVoService.findTerminalPerson(terminalIdList);

    List<TerminalContact> terminalContactList =
        terminalContactService.findByTerminalCodes(terminalCodeList);
    List<TerminalSupplyVo> terminalSupplyVoList =
        terminalSupplyVoService.findByTerminalCodes(terminalCodeList);
    List<TerminalTag> terminalTags =
      terminalTagService.findByTerminalCodes(new HashSet<>(terminalCodeList));
    re =
        this.buildTerminalVoList(
            terminalList,
            terminalRelaOrgList,
            terminalRelaCustomerOrgList,
            terminalContactList,
            terminalSupplyVoList,
            terminalTags,
            terminalPersonList);
    return re;
  }

  @Autowired(required = false)
  private ProcessBusinessService processBusinessService;

  @Value("${crm.business.terminal.process-key:}")
  private String defaultProcessKey;

  /**
   * 终端新增提交工作流进行审批，提交成功返回流程实例ID，提交失败则抛出异常
   *
   * @param dto 终端请求DTO
   */
  private String commitProcess(TerminalDto dto) {
    ProcessBusinessDto processBusiness = dto.getProcessBusiness();
    String processKey = processBusiness.getProcessKey();
    if (StringUtils.isBlank(processKey)) {
      processBusiness.setProcessKey(defaultProcessKey);
    } else {
      processBusiness.setProcessKey(processKey);
    }
    processBusiness.setProcessTitle(TERMINAL_PROCESS_NAME);
    processBusiness.setBusinessNo(dto.getTerminalCode());
    processBusiness.setBusinessFormJson(JsonUtils.obj2JsonString(dto));
    processBusiness.setBusinessCode(TerminalConstant.TERMINAL_PROCESS_NAME);
    ProcessBusinessVo processBusinessVo = processBusinessService.processStart(processBusiness);
    return processBusinessVo.getProcessNo();
  }

  /**
   * 根据经度和纬度查找当前城市范围内的数据
   * @param dto 距离(单位km)
   * @return
   */
  @Override
  public List<TerminalVo> findByLngAndLat(TerminalLugAndLatQueryDto dto) {
    Validate.notNull(dto, "查询终端时，参数不可为空！");
    Validate.notBlank(dto.getCurrentLng(), "查询终端时，当前经度不可为空!");
    Validate.notBlank(dto.getCurrentLat(), "查询终端时，当前纬度不可为空!");
    Validate.notBlank(dto.getDistance(), "查询终端时，查询距离不可为空!");
    Validate.notBlank(dto.getCityCode(), "查询终端时，当前城市编码不可为空!");
    final double DISTANCE = Double.parseDouble(dto.getDistance());
    double lng = Double.parseDouble(dto.getCurrentLng());
    double lat = Double.parseDouble(dto.getCurrentLat());
    //计算出当前位置范围内的经纬度
    Map<String, Double> map = this.findNeighDrugstore(lng, lat, DISTANCE);
    Double minLat = map.get("minlat");//纬度底
    Double maxLat = map.get("maxlat");//纬度顶
    Double minLng = map.get("minlng");//经度底
    Double maxLng = map.get("maxlng");//经度顶
    dto.setMinLat(minLat);
    dto.setMaxLat(maxLat);
    dto.setMinLng(minLng);
    dto.setMaxLng(maxLng);
    //按条件查询数据库
    List<TerminalVo> terminalVos = Lists.newArrayList();

    List<Terminal> terminals = this.terminalService
        .findByLngAndLat(dto, TenantUtils.getTenantCode());
    if (CollectionUtils.isEmpty(terminals)) {
      return terminalVos;
    }
    //再次筛选满足条件的数据
    List<Terminal> terminalList = terminals.stream().filter(terminal ->
        DistanceUtil.calculatePointDistance(lat, lng, terminal.getLatitude().doubleValue(),
            terminal.getLongitude().doubleValue()) <= (DISTANCE * 1000)
    ).collect(Collectors.toList());

    if (CollectionUtils.isEmpty(terminalList)) {
      return terminalVos;
    }

    terminalVos = (List<TerminalVo>) this.nebulaToolkitService
        .copyCollectionByWhiteList(terminalList, Terminal.class, TerminalVo.class, HashSet.class,
            ArrayList.class);
    //计算每个店铺距当前 所在经纬度的距离
    terminalVos.stream().forEach(terminalVo -> {
      double distance = DistanceUtil.calculatePointDistance(lat, lng,
          terminalVo.getLatitude().doubleValue(), terminalVo.getLongitude().doubleValue());
      //四舍五入保存距离
      terminalVo.setDistanceBetween(Math.round(distance));
    });
    //距离从远到近排序
    terminalVos = terminalVos.stream()
        .sorted(Comparator.comparing(TerminalVo::getDistanceBetween)).collect(
            Collectors.toList());
    return terminalVos;
  }

  @Override
  public Page<TerminalVo> findNearbyTerminals(Pageable pageable, NearbyTerminalsDto dto) {
    Page<TerminalVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    if (Objects.isNull(dto)){
      return page;
    }
    Page<TerminalVo> resultPage = this.terminalRepository.findNearbyTerminals(page, dto);
    List<TerminalVo> records = resultPage.getRecords();
    if (CollectionUtils.isNotEmpty(records)){
      List<String> ids = records.stream().map(o -> o.getId()).collect(Collectors.toList());
      List<TerminalAddressEntity> addressEntities = this.addressRepository.findByTerminalIds(ids);
      Map<String, TerminalAddressEntity> addressEntityMap = addressEntities.stream().collect(Collectors.toMap(TerminalAddressEntity::getTerminalId, Function.identity(), (v1, v2) -> v2));
      for (TerminalVo terminalVo : records) {
        TerminalAddressEntity entity = addressEntityMap.get(terminalVo.getId());
        if (Objects.nonNull(entity)){
          terminalVo.setTerminalAddress(entity.getAddressDetail());
        }
      }
    }
    return resultPage;
  }

  @Override
  public List<TerminalVo> findTerminalList(TerminalNewSearchDto dto) {
    List<TerminalVo> list = Lists.newLinkedList();
    List<String> ids = dto.getIds();
    if (CollectionUtils.isEmpty(ids)){
      return list;
    }
    List<Terminal> terminalVoList =this.terminalRepository.lambdaQuery()
            .in(Terminal::getId,ids)
            .in(Terminal::getEnableStatus,EnableStatusEnum.ENABLE.getCode())
            .in(Terminal::getTenantCode,TenantUtils.getTenantCode())
            .list();
    if (CollectionUtils.isEmpty(terminalVoList)){
      return list;
    }
    return this.convertEntityToVo(terminalVoList,dto);
  }

  /**
   * 附近选择
   * @param longitude 经度
   * @param latitude 纬度
   * @param distance 搜索范围
   * @return 经纬度范围
   */
  private Map<String, Double> findNeighDrugstore(double longitude,double latitude, double distance) {

    double minlat = 0;
    double maxlat = 0;
    double minlng = 0;
    double maxlng = 0;

    // 先计算查询点的经纬度范围
    double r = 6371;// 地球半径千米
    double dis = distance;// 距离(单位:千米)
    double dlng = 2 * Math.asin(Math.sin(dis / (2 * r))
        / Math.cos(longitude * Math.PI / 180));
    dlng = dlng * 180 / Math.PI;// 角度转为弧度
    double dlat = dis / r;
    dlat = dlat * 180 / Math.PI;
    if (dlng < 0) {
      minlng = longitude + dlng;
      maxlng = longitude - dlng;
    } else {
      minlng = longitude - dlng;
      maxlng = longitude + dlng;
    }
    if (dlat < 0) {
      minlat = latitude + dlat;
      maxlat = latitude - dlat;
    } else {
      minlat = latitude - dlat;
      maxlat = latitude + dlat;
    }
    Map<String,Double> params=new HashMap<>();
    params.put("minlat",minlat);//纬度底
    params.put("maxlat",maxlat);//纬度顶
    params.put("minlng",minlng);//经度底
    params.put("maxlng",maxlng);//经度顶
    return params;
  }
  private List<TerminalVo> convertEntityToVo(List<Terminal> terminalList,TerminalNewSearchDto dto) {
    Set<String> extendSet = dto.getExtendSet();
    List<TerminalVo> re = Lists.newArrayList();
    List<TerminalVo> result =(List<TerminalVo>) this.nebulaToolkitService.copyCollectionByWhiteList(terminalList,Terminal.class,TerminalVo.class,HashSet.class, ArrayList.class);
    List<String> terminalCodeList =
            terminalList.stream()
                    .filter(a -> StringUtils.isNotBlank(a.getTerminalCode()))
                    .map(Terminal::getTerminalCode)
                    .collect(Collectors.toList());

    List<String> terminalIdList =
            terminalList.stream()
                    .filter(a -> StringUtils.isNotBlank(a.getTerminalId()))
                    .map(Terminal::getTerminalCode)
                    .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(terminalCodeList)) {
      return re;
    }
    //终端关联org
    if(extendSet.contains(TerminalVoExtendEnum.ORG.getCode())){
      List<TerminalRelaOrg> terminalRelaOrgs = this.terminalRelaOrgRepository.findByTerminalCodes(terminalCodeList);
      if (CollectionUtils.isNotEmpty(terminalRelaOrgs)){
        List<TerminalRelaOrgVo> terminalRelaOrgVoList =(List<TerminalRelaOrgVo>)
                this.nebulaToolkitService.copyCollectionByWhiteList(terminalRelaOrgs,TerminalRelaOrg.class,TerminalRelaOrgVo.class, HashSet.class, ArrayList.class);
        Map<String, List<TerminalRelaOrgVo>> map = terminalRelaOrgVoList.stream().collect(Collectors.groupingBy(TerminalRelaOrgVo::getTerminalId));
        result.forEach(o->{
          o.setOrgList(map.get(o.getTerminalId()));
        });
      }
    }
    //终端关联采供关系
    if (extendSet.contains(TerminalVoExtendEnum.SUPPLY.getCode())){
      List<TerminalSupplyVo> supplyVos = this.terminalSupplyVoService.findByTerminalCodes(terminalCodeList);
      if (CollectionUtils.isNotEmpty(supplyVos)){
        result.forEach(o->{
          o.setSupplys(supplyVos);
        });
      }
    }
    //终端关联终端地址
    if (extendSet.contains(TerminalVoExtendEnum.ADDRESS.getCode())){
      List<TerminalAddressEntity> addressEntityList = this.addressRepository.findByTerminalCode(terminalCodeList);
      if (CollectionUtils.isNotEmpty(addressEntityList)){
        Map<String, String> map = addressEntityList.stream().collect(HashMap::new, (k,v) ->k.put(v.getTerminalId(),v.getAddressDetail()),HashMap::putAll);
        result.forEach(o->o.setTerminalAddress(map.get(o.getTerminalId())));
      }
    }
    //终端关联终端业务员
    if (extendSet.contains(TerminalVoExtendEnum.PERSON.getCode())){
      List<TerminalPersonEntity> terminalPerson = this.personVoService.findTerminalPersonByCode(terminalCodeList);
      if (CollectionUtils.isNotEmpty(terminalPerson)){
        List<TerminalPersonVo> personVos =(List<TerminalPersonVo>) this.nebulaToolkitService.copyCollectionByWhiteList(terminalPerson,TerminalPersonEntity.class,TerminalPersonVo.class, HashSet.class, ArrayList.class);
        Map<String, List<TerminalPersonVo>> map = personVos.stream().collect(Collectors.groupingBy(TerminalPersonVo::getTerminalId));
        result.forEach(o->o.setPersonVos(map.get(o.getTerminalId())));
      }
    }
    return result;
  }
}
