package com.biz.crm.mdm.business.terminal.user.local.service.internal;

import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.terminal.sdk.service.TerminalVoService;
import com.biz.crm.mdm.business.terminal.sdk.vo.TerminalVo;
import com.biz.crm.mdm.business.terminal.user.local.entity.TerminalUser;
import com.biz.crm.mdm.business.terminal.user.local.entity.TerminalUserRelaTerminal;
import com.biz.crm.mdm.business.terminal.user.local.repository.TerminalUserRepository;
import com.biz.crm.mdm.business.terminal.user.local.service.TerminalUserRelaTerminalService;
import com.biz.crm.mdm.business.terminal.user.local.service.TerminalUserService;
import com.biz.crm.mdm.business.terminal.user.sdk.dto.TerminalUserDto;
import com.biz.crm.mdm.business.terminal.user.sdk.dto.TerminalUserRelaTerminalDto;
import com.biz.crm.mdm.business.terminal.user.sdk.event.TerminalUserEventListener;
import com.biz.crm.mdm.business.terminal.user.sdk.service.TerminalUserVoService;
import com.biz.crm.mdm.business.terminal.user.sdk.vo.TerminalUserRelaTerminalVo;
import com.biz.crm.mdm.business.terminal.user.sdk.vo.TerminalUserVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 终端用户(TerminalUser)表服务实现类
 *
 * @author sunx
 * @since 2021-10-20 16:35:05
 */
@Service
public class TerminalUserVoServiceImpl implements TerminalUserVoService {

  @Autowired(required = false) private TerminalUserService terminalUserService;

  @Autowired(required = false) private TerminalUserRelaTerminalService terminalUserRelaTerminalService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  @Lazy
  private List<TerminalUserEventListener> eventListeners;

  @Autowired(required = false) private TerminalUserRepository terminalUserRepository;

  @Autowired(required = false) private TerminalVoService terminalVoService;

  @Override
  public List<TerminalUserVo> findDetailsByIdsOrUserCodes(
      List<String> ids, List<String> userCodes) {
    List<TerminalUserVo> re = Lists.newLinkedList();
    if (CollectionUtils.isEmpty(ids) && CollectionUtils.isEmpty(userCodes)) {
      return re;
    }
    List<TerminalUser> terminalUserList =
        terminalUserService.findDetailsByIdsOrUserCodes(ids, userCodes);
    if (CollectionUtils.isEmpty(terminalUserList)) {
      return re;
    }

    List<String> userCodeList =
        terminalUserList.stream()
            .filter(a -> StringUtils.isNotBlank(a.getUserCode()))
            .map(TerminalUser::getUserCode)
            .collect(Collectors.toList());
    if (CollectionUtils.isEmpty(userCodeList)) {
      return re;
    }

    List<TerminalUserRelaTerminal> terminalList =
        terminalUserRelaTerminalService.findByUserCodes(userCodeList);

    re = this.buildTerminalUserVoList(terminalUserList, terminalList);
    return re;
  }

  @Override
  @Transactional
  public TerminalUserVo create(TerminalUserDto dto) {
    Validate.notNull(dto, "终端用户信息缺失");
    TerminalUser terminalUser =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, TerminalUser.class, HashSet.class, ArrayList.class);
    dto.setLockState(
        Optional.ofNullable(dto.getLockState()).orElse(EnableStatusEnum.ENABLE.getCode()));
    //新增租户编号
    terminalUser.setTenantCode(TenantUtils.getTenantCode());
    this.terminalUserService.create(terminalUser);
    dto.setUserCode(terminalUser.getUserCode());
    this.bindExtInfo(dto);
    TerminalUserVo re = this.buildByDtoAndTerminalUser(dto, terminalUser);
    // 发送通知
    if (CollectionUtils.isEmpty(eventListeners)) {
      return re;
    }
    for (TerminalUserEventListener eventListener : eventListeners) {
      eventListener.onCreate(re);
    }
    return re;
  }

  @Override
  @Transactional
  public TerminalUserVo update(TerminalUserDto dto) {
    Validate.notNull(dto, "终端用户信息缺失");
    dto.setLockState(
        Optional.ofNullable(dto.getLockState()).orElse(EnableStatusEnum.ENABLE.getCode()));
    Boolean flag = CollectionUtils.isNotEmpty(eventListeners);
    TerminalUserVo oldVo = null;
    if (Boolean.TRUE.equals(flag)) {
      List<TerminalUserVo> list =
          this.findDetailsByIdsOrUserCodes(Lists.newArrayList(dto.getId()), null);
      if (CollectionUtils.isNotEmpty(list)) {
        oldVo = list.get(0);
      }
    }
    TerminalUser terminalUser =
        this.nebulaToolkitService.copyObjectByWhiteList(
            dto, TerminalUser.class, HashSet.class, ArrayList.class);
    //新增租户编号
    terminalUser.setTenantCode(TenantUtils.getTenantCode());
    this.terminalUserService.update(terminalUser);
    dto.setUserCode(terminalUser.getUserCode());
    this.bindExtInfo(dto);
    TerminalUserVo re = this.buildByDtoAndTerminalUser(dto, terminalUser);
    // 发送通知
    if (Boolean.FALSE.equals(flag)) {
      return re;
    }
    for (TerminalUserEventListener eventListener : eventListeners) {
      eventListener.onUpdate(oldVo, re);
    }
    return re;
  }

  @Override
  public TerminalUserVo findByUserPhone(String userPhone) {
    Validate.notBlank(userPhone, "手机号缺失");
    TerminalUser terminalUser = this.terminalUserRepository.findUserByPhone(userPhone);
    if (Objects.isNull(terminalUser)) {
      return null;
    }
    List<TerminalUserRelaTerminal> terminalList =
        terminalUserRelaTerminalService.findByUserCodes(
            Lists.newArrayList(terminalUser.getUserCode()));
    List<TerminalUserVo> list =
        this.buildTerminalUserVoList(Lists.newArrayList(terminalUser), terminalList);
    return list.get(0);
  }

  @Override
  public TerminalUserVo findByUserName(String userName) {
    Validate.notBlank(userName, "账号缺失");
    TerminalUser terminalUser = this.terminalUserRepository.findUserByUserName(userName);
    if (Objects.isNull(terminalUser)) {
      return null;
    }
    List<TerminalUserRelaTerminal> terminalList =
      terminalUserRelaTerminalService.findByUserCodes(
        Lists.newArrayList(terminalUser.getUserCode()));
    List<TerminalUserVo> list =
      this.buildTerminalUserVoList(Lists.newArrayList(terminalUser), terminalList);
    return list.get(0);
  }

  /**
   * 保存关联终端信息
   *
   * @param dto
   */
  @Transactional
  public void bindExtInfo(TerminalUserDto dto) {
    Validate.notNull(dto, "终端用户信息缺失");
    List<TerminalUserRelaTerminal> terminalList = Lists.newLinkedList();
    if (CollectionUtils.isNotEmpty(dto.getTerminalList())) {
      for (TerminalUserRelaTerminalDto item : dto.getTerminalList()) {
        item.setUserCode(dto.getUserCode());
      }
      terminalList =
          (List<TerminalUserRelaTerminal>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getTerminalList(),
                  TerminalUserRelaTerminalDto.class,
                  TerminalUserRelaTerminal.class,
                  HashSet.class,
                  ArrayList.class);
      terminalList.forEach(a -> a.setUserCode(dto.getUserCode()));
      //新增租户编号
      terminalList.forEach(a -> a.setTenantCode(TenantUtils.getTenantCode()));
    }
    terminalUserRelaTerminalService.saveBatch(terminalList, dto.getUserCode());
  }

  /**
   * 构建返回vo
   *
   * @param terminalUserList
   * @param terminalList
   * @return
   */
  private List<TerminalUserVo> buildTerminalUserVoList(
      List<TerminalUser> terminalUserList, List<TerminalUserRelaTerminal> terminalList) {
    List<TerminalUserVo> re = Lists.newLinkedList();
    Map<String, List<TerminalUserRelaTerminalVo>> map = Maps.newHashMap();
    if (CollectionUtils.isEmpty(terminalUserList)) {
      return re;
    }
    re =
        (List<TerminalUserVo>)
            this.nebulaToolkitService.copyCollectionByWhiteList(
                terminalUserList,
                TerminalUser.class,
                TerminalUserVo.class,
                HashSet.class,
                ArrayList.class);
    if (CollectionUtils.isNotEmpty(terminalList)) {
      Set<String> terminalCodeSet =
          terminalList.stream()
              .filter(a -> StringUtils.isNotBlank(a.getTerminalCode()))
              .map(TerminalUserRelaTerminal::getTerminalCode)
              .collect(Collectors.toSet());
      // k-terminalCode,v-terminalName
      Map<String, String> mapTerminal = Maps.newHashMap();
      if (CollectionUtils.isNotEmpty(terminalCodeSet)) {
        List<TerminalVo> terminalVos =
            this.terminalVoService.findMainDetailsByTerminalCodes(
                Lists.newArrayList(terminalCodeSet));
        if (CollectionUtils.isNotEmpty(terminalVos)) {
          mapTerminal =
              terminalVos.stream()
                  .filter(a -> StringUtils.isNoneBlank(a.getTerminalCode(), a.getTerminalName()))
                  .collect(
                      Collectors.toMap(
                          TerminalVo::getTerminalCode, TerminalVo::getTerminalName, (a, b) -> a));
        }
      }
      for (TerminalUserRelaTerminal item : terminalList) {
        item.setTerminalName(mapTerminal.get(item.getTerminalCode()));
      }

      List<TerminalUserRelaTerminalVo> voList =
          (List<TerminalUserRelaTerminalVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  terminalList,
                  TerminalUserRelaTerminal.class,
                  TerminalUserRelaTerminalVo.class,
                  HashSet.class,
                  ArrayList.class);
      map = voList.stream().collect(Collectors.groupingBy(TerminalUserRelaTerminalVo::getUserCode));
    }
    for (TerminalUserVo item : re) {
      item.setTerminalList(map.get(item.getUserCode()));
    }
    return re;
  }

  /**
   * 构建响应vo信息
   *
   * @param dto
   * @param terminalUser
   * @return
   */
  private TerminalUserVo buildByDtoAndTerminalUser(TerminalUserDto dto, TerminalUser terminalUser) {
    if (Objects.isNull(terminalUser) || Objects.isNull(dto)) {
      return null;
    }
    TerminalUserVo vo =
        this.nebulaToolkitService.copyObjectByWhiteList(
            terminalUser, TerminalUserVo.class, HashSet.class, ArrayList.class);
    if (CollectionUtils.isNotEmpty(dto.getTerminalList())) {
      vo.setTerminalList(
          (List<TerminalUserRelaTerminalVo>)
              this.nebulaToolkitService.copyCollectionByWhiteList(
                  dto.getTerminalList(),
                  TerminalUserRelaTerminalDto.class,
                  TerminalUserRelaTerminalVo.class,
                  HashSet.class,
                  ArrayList.class));
    }
    return vo;
  }
}
