package com.biz.crm.mdm.business.terminaluser.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.mdm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.terminaluser.local.entity.TerminalUser;
import com.biz.crm.mdm.business.terminaluser.local.repository.TerminalUserRepository;
import com.biz.crm.mdm.business.terminaluser.local.service.TerminalUserService;
import com.biz.crm.mdm.business.terminaluser.sdk.constant.TerminalUserConstant;
import com.biz.crm.mdm.business.terminaluser.sdk.dto.TerminalUserPaginationDto;
import com.biz.crm.mdm.business.terminaluser.sdk.dto.TerminalUserResetPasswordDto;
import com.biz.crm.mdm.business.terminaluser.sdk.event.TerminalUserEventListener;
import com.biz.crm.mdm.business.terminaluser.sdk.vo.TerminalUserVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 终端用户(TerminalUser)表服务实现类
 *
 * @author sunx
 * @since 2021-10-20 16:35:05
 */
@Slf4j
@Service("terminalUserService")
public class TerminalUserServiceImpl implements TerminalUserService {

  @Autowired
  private TerminalUserRepository terminalUserRepository;

  @Autowired(required = false)
  private List<TerminalUserEventListener> eventListeners;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<TerminalUser> findByConditions(Pageable pageable, TerminalUserPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new TerminalUserPaginationDto());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Page<TerminalUser> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return terminalUserRepository.findByConditions(page, dto);
  }

  @Override
  public TerminalUser findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return terminalUserRepository.findById(id);
  }

  @Override
  @Transactional
  public TerminalUser create(TerminalUser terminalUser) {
    Validate.notNull(terminalUser, "终端用户信息缺失");
    // 如果userCode为空需要期初一个编码，否则执行验重逻辑
    if (StringUtils.isEmpty(terminalUser.getUserCode())) {
      terminalUser
          .setUserCode(
              generateCodeService.generateCode(TerminalUserConstant.TERMINAL_USER_CODE, 1).get(0));
    } else {
      Integer count = terminalUserRepository.countByUserCode(terminalUser.getUserCode());
      Validate.isTrue(null == count || 1 > count, terminalUser.getUserCode() + "编码已存在");
    }
    terminalUser.setTenantCode(TenantUtils.getTenantCode());
    terminalUser.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    terminalUser.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    terminalUserRepository.saveOrUpdate(terminalUser);
    // 发送通知
    if (CollectionUtils.isEmpty(eventListeners)) {
      return terminalUser;
    }
    TerminalUserVo vo = this.nebulaToolkitService
        .copyObjectByWhiteList(terminalUser, TerminalUserVo.class, HashSet.class, ArrayList.class);
    for (TerminalUserEventListener eventListener : eventListeners) {
      eventListener.onCreate(vo);
    }
    return terminalUser;
  }

  @Override
  @Transactional
  public TerminalUser update(TerminalUser terminalUser) {
    Validate.notNull(terminalUser, "终端用户信息缺失");
    Validate.isTrue(StringUtils.isNotBlank(terminalUser.getId()), "id不能为空");
    String currentId = terminalUser.getId();
    TerminalUserVo oldVo = null;
    TerminalUser current = terminalUserRepository.findById(currentId);
    current = Validate.notNull(current, "修改信息不存在");
    Validate.isTrue(terminalUser.getUserCode().equals(current.getUserCode()), "编码不能修改");
    if (CollectionUtils.isNotEmpty(eventListeners)) {
      oldVo = this.nebulaToolkitService
          .copyObjectByBlankList(current, TerminalUserVo.class, HashSet.class, ArrayList.class);
    }
    terminalUserRepository.saveOrUpdate(terminalUser);
    // 发送修改通知
    if (CollectionUtils.isEmpty(eventListeners)) {
      return terminalUser;
    }
    TerminalUserVo newVo = this.nebulaToolkitService
        .copyObjectByWhiteList(terminalUser, TerminalUserVo.class, HashSet.class, ArrayList.class);
    for (TerminalUserEventListener eventListener : eventListeners) {
      eventListener.onUpdate(oldVo, newVo);
    }
    return terminalUser;
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    terminalUserRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    onEnableOrDisable(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    terminalUserRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    onEnableOrDisable(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    terminalUserRepository.updateDelFlagByIds(ids);
    if (CollectionUtils.isEmpty(eventListeners)) {
      return;
    }
    List<TerminalUserVo> voList = findVoListByIds(ids);
    if (CollectionUtils.isEmpty(voList)) {
      return;
    }
    for (TerminalUserEventListener eventListener : eventListeners) {
      eventListener.onDelete(voList);
    }
  }

  @Override
  public List<TerminalUser> findDetailsByIdsOrUserCodes(List<String> ids, List<String> userCodes) {
    if (CollectionUtils.isEmpty(ids) && CollectionUtils.isEmpty(userCodes)) {
      return Lists.newArrayList();
    }
    if (CollectionUtils.isNotEmpty(ids)) {
      return terminalUserRepository.findByIds(ids);
    }
    return terminalUserRepository.findByUserCodes(userCodes);
  }

  @Override
  @Transactional
  public void resetPassword(TerminalUserResetPasswordDto dto) {
    Validate.notNull(dto, "参数不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getIds()), "终端用户id集合不能为空");
    terminalUserRepository.resetPassword(dto);
  }

  /**
   * 发送启用禁用变更通知
   *
   * @param ids
   * @param enableStatusEnum
   */
  private void onEnableOrDisable(List<String> ids, EnableStatusEnum enableStatusEnum) {
    if (CollectionUtils.isEmpty(eventListeners)) {
      return;
    }
    List<TerminalUserVo> voList = findVoListByIds(ids);
    if (CollectionUtils.isEmpty(voList)) {
      return;
    }
    for (TerminalUserEventListener event : eventListeners) {
      if (enableStatusEnum.equals(EnableStatusEnum.ENABLE)) {
        event.onEnable(voList);
      } else if (enableStatusEnum.equals(EnableStatusEnum.DISABLE)) {
        event.onDisable(voList);
      }
    }
  }

  /**
   * 只有主表信息，不包含扩展信息
   *
   * @param ids
   * @return
   */
  private List<TerminalUserVo> findVoListByIds(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return Lists.newArrayList();
    }
    List<TerminalUser> list = this.findDetailsByIdsOrUserCodes(ids, null);
    if (CollectionUtils.isEmpty(list)) {
      return Lists.newArrayList();
    }
    return (List<TerminalUserVo>) this.nebulaToolkitService
        .copyCollectionByWhiteList(list, TerminalUser.class, TerminalUserVo.class, HashSet.class,
            ArrayList.class);
  }
}
