package com.biz.crm.mdm.business.terminaluser.local.service.internal;

import com.biz.crm.mdm.business.terminaluser.local.entity.TerminalUser;
import com.biz.crm.mdm.business.terminaluser.local.entity.TerminalUserRTerminal;
import com.biz.crm.mdm.business.terminaluser.local.service.TerminalUserRTerminalService;
import com.biz.crm.mdm.business.terminaluser.local.service.TerminalUserService;
import com.biz.crm.mdm.business.terminaluser.sdk.dto.TerminalUserDto;
import com.biz.crm.mdm.business.terminaluser.sdk.dto.TerminalUserRTerminalDto;
import com.biz.crm.mdm.business.terminaluser.sdk.service.TerminalUserVoService;
import com.biz.crm.mdm.business.terminaluser.sdk.vo.TerminalUserRTerminalVo;
import com.biz.crm.mdm.business.terminaluser.sdk.vo.TerminalUserVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 终端用户(TerminalUser)表服务实现类
 *
 * @author sunx
 * @since 2021-10-20 16:35:05
 */
@Slf4j
@Service
public class TerminalUserVoServiceImpl implements TerminalUserVoService {

  @Autowired
  private TerminalUserService terminalUserService;

  @Autowired
  private TerminalUserRTerminalService terminalUserRTerminalService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public List<TerminalUserVo> findDetailsByIdsOrUserCodes(List<String> ids,
      List<String> userCodes) {
    List<TerminalUserVo> re = Lists.newArrayList();
    if (CollectionUtils.isEmpty(ids) && CollectionUtils.isEmpty(userCodes)) {
      return re;
    }
    List<TerminalUser> terminalUserList = terminalUserService
        .findDetailsByIdsOrUserCodes(ids, userCodes);
    if (CollectionUtils.isEmpty(terminalUserList)) {
      return re;
    }

    List<String> userCodeList = terminalUserList.stream()
        .filter(a -> StringUtils.isNotBlank(a.getUserCode())).map(TerminalUser::getUserCode)
        .collect(
            Collectors.toList());
    if (CollectionUtils.isEmpty(userCodeList)) {
      return re;
    }

    List<TerminalUserRTerminal> terminalList = terminalUserRTerminalService
        .findByUserCodes(userCodeList);

    re = this.buildTerminalUserVoList(terminalUserList, terminalList);
    return re;
  }


  @Override
  @Transactional
  public TerminalUserVo create(TerminalUserDto dto) {
    Validate.notNull(dto, "终端用户信息缺失");
    TerminalUser terminalUser = this.nebulaToolkitService
        .copyObjectByWhiteList(dto, TerminalUser.class, HashSet.class, ArrayList.class);
    this.terminalUserService.create(terminalUser);
    dto.setUserCode(terminalUser.getUserCode());
    this.bindExtInfo(dto);
    return this.buildByDtoAndTerminalUser(dto, terminalUser);
  }


  @Override
  @Transactional
  public TerminalUserVo update(TerminalUserDto dto) {
    Validate.notNull(dto, "终端用户信息缺失");
    TerminalUser terminalUser = this.nebulaToolkitService
        .copyObjectByWhiteList(dto, TerminalUser.class, HashSet.class, ArrayList.class);
    this.terminalUserService.update(terminalUser);
    dto.setUserCode(terminalUser.getUserCode());
    this.bindExtInfo(dto);
    return this.buildByDtoAndTerminalUser(dto, terminalUser);
  }

  /**
   * 保存关联终端信息
   *
   * @param dto
   */
  @Transactional
  public void bindExtInfo(TerminalUserDto dto) {
    Validate.notNull(dto, "终端用户信息缺失");
    List<TerminalUserRTerminal> terminalList = Lists.newArrayList();
    if (CollectionUtils.isNotEmpty(dto.getTerminalList())) {
      for (TerminalUserRTerminalDto item : dto.getTerminalList()) {
        item.setUserCode(dto.getUserCode());
      }
      terminalList = (List<TerminalUserRTerminal>) this.nebulaToolkitService
          .copyCollectionByWhiteList(dto.getTerminalList(), TerminalUserRTerminalDto.class,
              TerminalUserRTerminal.class,
              HashSet.class, ArrayList.class);
      terminalList.forEach(a -> a.setTerminalCode(dto.getUserCode()));
    }
    terminalUserRTerminalService.saveBatch(terminalList, dto.getUserCode());
  }

  /**
   * 构建返回vo
   *
   * @param terminalUserList
   * @param terminalList
   * @return
   */
  private List<TerminalUserVo> buildTerminalUserVoList(List<TerminalUser> terminalUserList,
      List<TerminalUserRTerminal> terminalList) {
    List<TerminalUserVo> re = Lists.newArrayList();
    Map<String, List<TerminalUserRTerminalVo>> map = Maps.newHashMap();
    if (CollectionUtils.isEmpty(terminalUserList)) {
      return re;
    }
    re = (List<TerminalUserVo>) this.nebulaToolkitService
        .copyCollectionByWhiteList(terminalUserList, TerminalUser.class, TerminalUserVo.class,
            HashSet.class, ArrayList.class);
    if (CollectionUtils.isNotEmpty(terminalList)) {
      List<TerminalUserRTerminalVo> voList = (List<TerminalUserRTerminalVo>) this.nebulaToolkitService
          .copyCollectionByWhiteList(terminalList, TerminalUserRTerminal.class,
              TerminalUserRTerminalVo.class,
              HashSet.class, ArrayList.class);
      map = voList.stream()
          .collect(Collectors.groupingBy(TerminalUserRTerminalVo::getUserCode));
    }
    for (TerminalUserVo item : re) {
      item.setTerminalList(map.get(item.getUserCode()));
    }
    return re;
  }

  /**
   * 构建响应vo信息
   *
   * @param dto
   * @param terminalUser
   * @return
   */
  private TerminalUserVo buildByDtoAndTerminalUser(TerminalUserDto dto, TerminalUser terminalUser) {
    if (Objects.isNull(terminalUser) || Objects.isNull(dto)) {
      return null;
    }
    TerminalUserVo vo = this.nebulaToolkitService
        .copyObjectByWhiteList(terminalUser, TerminalUserVo.class, HashSet.class, ArrayList.class);
    if (CollectionUtils.isNotEmpty(dto.getTerminalList())) {
      vo.setTerminalList(
          (List<TerminalUserRTerminalVo>) this.nebulaToolkitService
              .copyCollectionByWhiteList(dto.getTerminalList(),
                  TerminalUserRTerminalDto.class, TerminalUserRTerminalVo.class, HashSet.class,
                  ArrayList.class));
    }
    return vo;
  }
}
