package com.biz.crm.mdm.business.user.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.mdm.business.user.local.entity.UserRelWeChatEntity;
import com.biz.crm.mdm.business.user.local.mapper.UserRelWeChatMapper;
import com.biz.crm.mdm.business.user.sdk.dto.UserRelWeChatDto;
import com.biz.crm.mdm.business.user.sdk.dto.UserRelWeChatPageDto;
import com.biz.crm.mdm.business.user.sdk.dto.UserWeChatBindDto;
import com.biz.crm.mdm.business.user.sdk.vo.UserRelWeChatVo;
import org.apache.commons.lang3.StringUtils;
import org.apache.ibatis.annotations.Param;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * 用户与微信关联表(repository)
 *
 * @author xi.peng
 * @since 2021-11-05 15:05:07
 */
@Component
public class UserRelWeChatRepository extends ServiceImpl<UserRelWeChatMapper, UserRelWeChatEntity> {
  
  /**
   * 分页
   *
   * @param pageable
   * @param dto
   * @return
   */
  public Page<UserRelWeChatVo> findByConditions(Pageable pageable, @Param("dto") UserRelWeChatPageDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<UserRelWeChatPageDto> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    dto = Optional.ofNullable(dto).orElse(new UserRelWeChatPageDto());
    return baseMapper.findByConditions(page, dto);
  }
  
  /**
   * 根据ID获取详情
   *
   * @param id
   * @return
   */
  public UserRelWeChatEntity findById(String id) {
    return this.baseMapper.selectById(id);
  }

  /**
   * 逻辑删除
   *
   * @param tenantCode
   * @param userNames
   */
  public void deleteByUserNames(@Param("tenantCode") String tenantCode, @Param("userNames") List<String> userNames) {
    this.lambdaUpdate()
        .eq(UserRelWeChatEntity::getTenantCode, tenantCode)
        .in(UserRelWeChatEntity::getUserName, userNames)
        .remove();
  }

  /**
   * 查询用户关联的微信列表
   *
   * @param tenantCode
   * @param dto
   */
  public List<UserRelWeChatEntity> findByUserRelWeChatDto(@Param("tenantCode") String tenantCode, @Param("dto") UserRelWeChatDto dto) {
    dto = Optional.ofNullable(dto).orElse(new UserRelWeChatDto());
    if (StringUtils.isBlank(dto.getUserName())) {
      return new ArrayList<>();
    }
    return this.lambdaQuery()
        .eq(UserRelWeChatEntity::getTenantCode, tenantCode)
        .eq(UserRelWeChatEntity::getUserName, dto.getUserName())
        .eq(StringUtils.isNotBlank(dto.getOriginModule()), UserRelWeChatEntity::getOriginModule, dto.getOriginModule())
        .eq(Objects.nonNull(dto.getLoginStatus()), UserRelWeChatEntity::getLoginStatus, dto.getLoginStatus())
        .like(StringUtils.isNotBlank(dto.getNickName()), UserRelWeChatEntity::getNickName, dto.getNickName())
        .list();
  }

  /**
   * 通过用户微信绑定dto获取绑定关系信息
   *
   * @param dto 用户微信绑定dto
   * @return 绑定关系信息
   */
  public List<UserRelWeChatEntity> findByUserWeChatBindDto(UserWeChatBindDto dto) {
    return this.lambdaQuery()
        .eq(UserRelWeChatEntity::getTenantCode, dto.getTenantCode())
        .eq(UserRelWeChatEntity::getOriginModule, dto.getOriginModule())
        .eq(StringUtils.isNotBlank(dto.getUserName()), UserRelWeChatEntity::getUserName, dto.getUserName())
        .eq(StringUtils.isNotBlank(dto.getOpenId()), UserRelWeChatEntity::getOpenId, dto.getOpenId())
        .list();
  }
}
