package com.biz.crm.mdm.business.user.local.repository;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.mdm.business.user.local.entity.UserRelDingTalkEntity;
import com.biz.crm.mdm.business.user.local.mapper.UserRelDingTalkMapper;
import com.biz.crm.mdm.business.user.sdk.dto.UserRelDingTalkDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

/**
 * 用户与钉钉关联表(repository)
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.12.3 22:16
 */
@Component
public class UserRelDingTalkRepository extends ServiceImpl<UserRelDingTalkMapper, UserRelDingTalkEntity> {


    /**
     * 根据ID获取详情
     *
     * @param id
     * @return
     */
    public UserRelDingTalkEntity findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(UserRelDingTalkEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(UserRelDingTalkEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .in(UserRelDingTalkEntity::getId, id)
                .one();
    }


    /**
     * 查询用户关联的钉钉列表
     *
     * @param dto 查询参数
     */
    public List<UserRelDingTalkEntity> findByUserRelDingTalkDto(UserRelDingTalkDto dto) {
        dto = Optional.ofNullable(dto).orElse(new UserRelDingTalkDto());
        if (StringUtils.isBlank(dto.getUserName())) {
            return new ArrayList<>();
        }
        return this.lambdaQuery()
                .eq(UserRelDingTalkEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(UserRelDingTalkEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(UserRelDingTalkEntity::getUserName, dto.getUserName())
                .list();
    }


    /**
     * 根据钉钉用户ID查询数据
     *
     * @param dingTalkUserId 钉钉用户ID
     * @return
     */
    public List<UserRelDingTalkEntity> findByDingTalkUserId(String dingTalkUserId) {
        if (StringUtils.isBlank(dingTalkUserId)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .eq(UserRelDingTalkEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(UserRelDingTalkEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(UserRelDingTalkEntity::getDingTalkUserId, dingTalkUserId)
                .list();
    }

    /**
     * 根据钉钉用户ID集合查询数据
     *
     * @param dingTalkUserIds 钉钉用户ID集合
     * @return
     */
    public List<UserRelDingTalkEntity> findByDingTalkUserIds(List<String> dingTalkUserIds) {
        if (CollectionUtil.isEmpty(dingTalkUserIds)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .eq(UserRelDingTalkEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(UserRelDingTalkEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .in(UserRelDingTalkEntity::getDingTalkUserId, dingTalkUserIds)
                .list();
    }

    /**
     * 根据账号查询数据
     *
     * @param userName 用户名
     * @return
     */
    public List<UserRelDingTalkEntity> findByUserName(String userName) {
        if (StringUtils.isBlank(userName)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .eq(UserRelDingTalkEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(UserRelDingTalkEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .eq(UserRelDingTalkEntity::getUserName, userName)
                .list();
    }

    /**
     * 根据账号查询数据
     *
     * @param userNameList 用户名集合
     * @return
     */
    public List<UserRelDingTalkEntity> findByUserNames(List<String> userNameList) {
        if (CollectionUtil.isEmpty(userNameList)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .eq(UserRelDingTalkEntity::getTenantCode, TenantUtils.getTenantCode())
                .eq(UserRelDingTalkEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .in(UserRelDingTalkEntity::getUserName, userNameList)
                .list();
    }

    /**
     * 重构修改方法
     *
     * @param entity
     */
    public void updateEntity(UserRelDingTalkEntity entity) {
        LambdaUpdateWrapper<UserRelDingTalkEntity> lambdaUpdateWrapper = Wrappers.lambdaUpdate();
        lambdaUpdateWrapper.eq(UserRelDingTalkEntity::getTenantCode, TenantUtils.getTenantCode());
        lambdaUpdateWrapper.eq(UserRelDingTalkEntity::getId, entity.getId());
        this.baseMapper.update(entity, lambdaUpdateWrapper);
    }

    /**
     * 根据ID集合接触关联
     *
     * @param ids ID集合
     */
    public void removeByIds(List<String> ids) {
        if (CollectionUtil.isEmpty(ids)) {
            return;
        }
        this.lambdaUpdate()
                .eq(UserRelDingTalkEntity::getTenantCode, TenantUtils.getTenantCode())
                .in(UserRelDingTalkEntity::getId, ids)
                .remove();
    }

    /**
     * 根据账号集合接触关联
     *
     * @param userNameList 账号集合
     */
    public void deleteByUserNames(List<String> userNameList) {
        if (CollectionUtil.isEmpty(userNameList)) {
            return;
        }
        this.lambdaUpdate()
                .eq(UserRelDingTalkEntity::getTenantCode, TenantUtils.getTenantCode())
                .in(UserRelDingTalkEntity::getUserName, userNameList)
                .remove();
    }

    /**
     * 根据钉钉ID集合接触关联
     *
     * @param dingTalkUserIds 钉钉ID集合
     */
    public void deleteByDingTalkUserIds(List<String> dingTalkUserIds) {
        if (CollectionUtil.isEmpty(dingTalkUserIds)) {
            return;
        }
        this.lambdaUpdate()
                .eq(UserRelDingTalkEntity::getTenantCode, TenantUtils.getTenantCode())
                .in(UserRelDingTalkEntity::getDingTalkUserId, dingTalkUserIds)
                .remove();
    }

    /**
     * 根据ID集合接触关联
     *
     * @param ids ID集合
     */
    public void deleteByIds(List<String> ids) {
        if (CollectionUtil.isEmpty(ids)) {
            return;
        }
        this.lambdaUpdate()
                .eq(UserRelDingTalkEntity::getTenantCode, TenantUtils.getTenantCode())
                .in(UserRelDingTalkEntity::getId, ids)
                .remove();
    }


}

