package com.biz.crm.mdm.business.user.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.local.entity.UuidEntity;
import com.biz.crm.mdm.business.position.sdk.enums.PositionSourceEnum;
import com.biz.crm.mdm.business.user.local.entity.UserPositionEntity;
import com.biz.crm.mdm.business.user.local.repository.UserPositionRepository;
import com.biz.crm.mdm.business.user.sdk.service.UserPositionVoService;
import com.biz.crm.mdm.business.user.sdk.dto.UserPositionPageDto;
import com.biz.crm.mdm.business.user.sdk.dto.UserPositionDto;
import com.biz.crm.mdm.business.user.sdk.vo.UserPositionVo;
import com.biz.crm.mdm.business.user.sdk.event.UserPositionEventListener;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 职位-用户关联表(UserPosition)表服务实现类
 *
 * @author xi.peng
 * @since 2021-11-05 15:02:58
 */
@Service
public class UserPositionVoServiceImpl implements UserPositionVoService {

  @Autowired(required = false)
  private UserPositionRepository userPositionRepository;

  @Autowired(required = false)
  @Lazy
  private List<UserPositionEventListener> userPositionEventListeners;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<UserPositionVo> findByConditions(Pageable pageable, UserPositionPageDto dto) {
    return this.userPositionRepository.findByConditions(pageable, dto);
  }

  @Override
  public UserPositionVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    UserPositionEntity entity = this.userPositionRepository.findById(id);
    if (entity == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, UserPositionVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<UserPositionVo> findByUserName(String tenantCode, String userName) {
    if (StringUtils.isBlank(tenantCode) || StringUtils.isBlank(userName)) {
      return new ArrayList<>();
    }
    List<UserPositionEntity> entities = this.userPositionRepository.findByUserName(tenantCode, userName);
    if (CollectionUtils.isEmpty(entities)) {
      return new ArrayList<>();
    }
    return (List<UserPositionVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, UserPositionEntity.class, UserPositionVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public void create(UserPositionDto dto) {
    Validate.notNull(dto, "请求对象不能为空");
    dto.setTenantCode(TenantUtils.getTenantCode());
    UserPositionEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, UserPositionEntity.class, HashSet.class, ArrayList.class);
    //新增租户编号
    entity.setTenantCode(TenantUtils.getTenantCode());
    this.userPositionRepository.save(entity);
    // 发送通知
    if (CollectionUtils.isNotEmpty(userPositionEventListeners)) {
      UserPositionVo vo = this.nebulaToolkitService.copyObjectByWhiteList(entity, UserPositionVo.class, HashSet.class, ArrayList.class);
      this.userPositionEventListeners.forEach(event -> event.onCreate(vo));
    }
  }

  @Override
  @Transactional
  public void update(UserPositionDto dto) {
    Validate.notNull(dto, "请求对象不能为空");
    Validate.notBlank(dto.getId(), "ID不能为空");
    UserPositionEntity oldEntity = this.userPositionRepository.findById(dto.getId());
    Validate.notNull(oldEntity, "修改信息不存在");
    UserPositionEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, UserPositionEntity.class, HashSet.class, ArrayList.class);
    //重构修改方法
    this.userPositionRepository.updateByIdAndTenantCode(entity,TenantUtils.getTenantCode());
    // 发送修改通知
    if (CollectionUtils.isNotEmpty(userPositionEventListeners)) {
      UserPositionVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(oldEntity, UserPositionVo.class, HashSet.class, ArrayList.class);
      UserPositionVo newVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, UserPositionVo.class, HashSet.class, ArrayList.class);
      this.userPositionEventListeners.forEach(event -> event.onUpdate(oldVo, newVo));
    }
  }

  @Override
  public void deleteByUserNames(Collection<String> userNames) {
    if (CollectionUtils.isEmpty(userNames)) {
      return;
    }
    this.userPositionRepository.deleteByUserNames(TenantUtils.getTenantCode(), userNames);
  }

  @Override
  public void rebindByUserName(String userName, String primaryPositionCode, List<String> positionCodeList, List<UserPositionDto> positionList) {
    if (StringUtils.isBlank(userName)) {
      return;
    }
    this.userPositionRepository.deleteByUserNames(TenantUtils.getTenantCode(), Collections.singletonList(userName));
    Set<String> positionCodeSet = new HashSet<>();
    if (StringUtils.isNotBlank(primaryPositionCode)) {
      positionCodeSet.add(primaryPositionCode);
    }
    if (CollectionUtils.isNotEmpty(positionCodeList)) {
      this.userPositionRepository.deleteByPositionCodes(TenantUtils.getTenantCode(), positionCodeList);
      if (CollectionUtils.isEmpty(positionCodeSet)) {
        // 如果沒有主崗，則第一個作為主崗
        primaryPositionCode = positionCodeList.get(0);
      }
      positionCodeSet.addAll(positionCodeList);
    }
    if (CollectionUtils.isEmpty(positionCodeSet)) {
      return;
    }
    Map<String, UserPositionDto> userPositionMap = positionList.stream().collect(Collectors.toMap(UserPositionDto::getPositionCode, Function.identity()));
    List<UserPositionEntity> list = new ArrayList<>();
    for (String positionCode : positionCodeSet) {
      UserPositionEntity entity = new UserPositionEntity();
      entity.setTenantCode(TenantUtils.getTenantCode());
      entity.setUserName(userName);
      entity.setPositionCode(positionCode);
      entity.setMdgPositionCode(userPositionMap.get(positionCode).getMdgPositionCode());
      if (StringUtils.equals(primaryPositionCode, positionCode)) {
        entity.setPrimaryFlag(Boolean.TRUE);
        entity.setCurrentFlag(Boolean.TRUE);
      } else {
        entity.setPrimaryFlag(Boolean.FALSE);
        entity.setCurrentFlag(Boolean.FALSE);
      }
      list.add(entity);
    }
    this.userPositionRepository.saveBatch(list);
  }

  @Override
  @Transactional
  public void updateCurrentFlagPosition(String userName, String currentFlagPositionCode) {
    this.userPositionRepository.updateCurrentFlagFalseByUserName(TenantUtils.getTenantCode(), userName);
    this.userPositionRepository.updateCurrentFlagTrueByUserNameAndPositionCode(TenantUtils.getTenantCode(), userName, currentFlagPositionCode);
  }

  @Override
  public List<UserPositionVo> findAllUser(String tenantCode) {
    if (StringUtils.isBlank(tenantCode)) {
      return new ArrayList<>();
    }
    List<UserPositionEntity> entities = this.userPositionRepository.findAllUser(tenantCode);
    if (CollectionUtils.isEmpty(entities)) {
      return new ArrayList<>();
    }
    return (List<UserPositionVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, UserPositionEntity.class, UserPositionVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public Set<UserPositionVo> findByPositionCodeIn(Collection<String> positionCodes) {
    List<UserPositionEntity> userPositionList = userPositionRepository.findByPositionCodeIn(TenantUtils.getTenantCode(), positionCodes);
    Set<UserPositionVo> set = new HashSet<>();
    for (UserPositionEntity entity : userPositionList) {
      UserPositionVo userPositionVo = this.nebulaToolkitService.copyObjectByBlankList(entity, UserPositionVo.class, HashSet.class, ArrayList.class);
      set.add(userPositionVo);
    }
    return set;
  }

  /**
   * mdg一个用户只对应一个职位
   *
   * @param userPositionDtoList
   * @author huojia
   * @date 2022/12/31 1:09
   **/
  @Override
  public void rebindMdgUser(List<UserPositionDto> userPositionDtoList) {
    if (CollectionUtils.isEmpty(userPositionDtoList)) {
      return;
    }
    Set<String> userNameSet = userPositionDtoList.stream().map(UserPositionDto::getUserName).collect(Collectors.toSet());
    List<UserPositionEntity> userPositionEntities = userPositionRepository.listByUserNameList(TenantUtils.getTenantCode(), new ArrayList<>(userNameSet));
    List<String> mdgIds = userPositionEntities.stream()
            .filter(userPositionEntity -> PositionSourceEnum.MDG.getCode().equals(userPositionEntity.getPositionSource()))
            .map(UuidEntity::getId)
            .collect(Collectors.toList());
    List<UserPositionEntity> list = new ArrayList<>();
    String tenantCode = TenantUtils.getTenantCode();
    userPositionDtoList.forEach(userPositionDto -> {
      UserPositionEntity entity = new UserPositionEntity();
      entity.setTenantCode(TenantUtils.getTenantCode());
      entity.setUserName(userPositionDto.getUserName());
      entity.setPositionCode(userPositionDto.getPositionCode());
      entity.setPositionSource(PositionSourceEnum.MDG.getCode());
      entity.setMdgPositionCode(userPositionDto.getMdgPositionCode());
      entity.setTenantCode(tenantCode);
      if (userPositionDto.getPrimaryFlag()) {
        entity.setPrimaryFlag(Boolean.TRUE);
        entity.setCurrentFlag(Boolean.TRUE);
      } else {
        entity.setPrimaryFlag(Boolean.FALSE);
        entity.setCurrentFlag(Boolean.FALSE);
      }
      list.add(entity);
    });
    this.userPositionRepository.saveBatch(list);
    this.userPositionRepository.removeByIds(mdgIds);
  }

  @Override
  public List<UserPositionVo> listByUserNameList(List<String> userNameList) {
    if (CollectionUtils.isEmpty(userNameList)) {
      return new ArrayList<>();
    }
    List<UserPositionEntity> userPositionEntities = this.userPositionRepository.listByUserNameList(TenantUtils.getTenantCode(), userNameList);
    if (CollectionUtils.isEmpty(userPositionEntities)) {
      return new ArrayList<>();
    }
    return (List<UserPositionVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
            userPositionEntities, UserPositionEntity.class, UserPositionVo.class, LinkedHashSet.class, ArrayList.class
    );
  }
}
