package com.biz.crm.mdm.business.user.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.mdm.business.user.local.entity.UserRelDingTalkEntity;
import com.biz.crm.mdm.business.user.local.repository.UserRelDingTalkRepository;
import com.biz.crm.mdm.business.user.sdk.dto.UserRelDingTalkDto;
import com.biz.crm.mdm.business.user.sdk.service.UserRelDingTalkVoService;
import com.biz.crm.mdm.business.user.sdk.service.UserVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 用户与钉钉关联表(UserRelDingTalk)表服务实现类
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.12.3 22:36
 */
@Service
@Slf4j
public class UserRelDingTalkVoServiceImpl implements UserRelDingTalkVoService {

    @Autowired(required = false)
    private UserRelDingTalkRepository userRelDingTalkRepository;


    @Autowired(required = false)
    @Qualifier("nebulaToolkitService")
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private UserVoService userVoService;


    /**
     * 新增(新增前会删除账号已关联信息)
     *
     * @param dtoList 新增信息
     * @return java.util.List<com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:47
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public List<UserRelDingTalkVo> createBatch(List<UserRelDingTalkDto> dtoList) {
        this.createValidation(dtoList);
        List<String> userNameList = dtoList.stream()
                .map(UserRelDingTalkDto::getUserName)
                .distinct().collect(Collectors.toList());
        this.userRelDingTalkRepository.deleteByUserNames(userNameList);
        Collection<UserRelDingTalkEntity> entityList = this.nebulaToolkitService.copyCollectionByWhiteList(dtoList,
                UserRelDingTalkDto.class, UserRelDingTalkEntity.class, HashSet.class, ArrayList.class);
        this.userRelDingTalkRepository.saveBatch(entityList);
        return (List<UserRelDingTalkVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entityList,
                UserRelDingTalkEntity.class, UserRelDingTalkVo.class, HashSet.class, ArrayList.class);
    }

    /**
     * 新增验证
     *
     * @param dtoList 参数集合
     */
    private void createValidation(List<UserRelDingTalkDto> dtoList) {
        Assert.notEmpty(dtoList, "信息不能为空!");
        String tenantCode = TenantUtils.getTenantCode();
        dtoList.forEach(dto -> {
            Assert.hasLength(dto.getUserName(), "账号信息不能为空!");
            Assert.hasLength(dto.getDingTalkUserId(), "钉钉ID不能为空!");
            dto.setTenantCode(tenantCode);
        });
        List<String> userNameList = dtoList.stream()
                .map(UserRelDingTalkDto::getUserName)
                .distinct().collect(Collectors.toList());
        Set<String> userNameSet = userVoService.findUserNamesByUserNames(userNameList);
        Assert.notEmpty(userNameSet, "账号不存在!");
        dtoList.forEach(dto -> {
            Assert.isTrue(userNameSet.contains(dto.getUserName()), "账号[" + dto.getUserName() + "]不存在!");
        });

    }

    /**
     * 根据ID集合解除账号关联钉钉信息
     *
     * @param ids ID集合
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<String> ids) {
        if (CollectionUtil.isEmpty(ids)) {
            return;
        }
        this.userRelDingTalkRepository.deleteByIds(ids);
    }

    /**
     * 根据账号集合解除账号关联钉钉信息
     *
     * @param userNameList 账号集合
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByUserNames(List<String> userNameList) {
        if (CollectionUtil.isEmpty(userNameList)) {
            return;
        }
        this.userRelDingTalkRepository.deleteByUserNames(userNameList);
    }

    /**
     * 根据钉钉ID集合解除账号关联钉钉信息
     *
     * @param dingTalkUserIds 钉钉ID集合
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByDingTalkUserIds(List<String> dingTalkUserIds) {
        if (CollectionUtil.isEmpty(dingTalkUserIds)) {
            return;
        }
        this.userRelDingTalkRepository.deleteByDingTalkUserIds(dingTalkUserIds);
    }

    /**
     * 根据ID获取用户关联钉钉信息
     *
     * @param id ID
     * @return com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    public UserRelDingTalkVo findById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        UserRelDingTalkEntity entity = this.userRelDingTalkRepository.findById(id);
        if (entity == null) {
            return null;
        }
        return this.nebulaToolkitService.copyObjectByWhiteList(entity, UserRelDingTalkVo.class, HashSet.class, ArrayList.class);
    }

    /**
     * 根据钉钉ID获取用户关联钉钉信息
     *
     * @param dingTalkUserId 钉钉ID
     * @return java.util.List<com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    public List<UserRelDingTalkVo> findByDingTalkUserId(String dingTalkUserId) {
        if (StringUtils.isBlank(dingTalkUserId)) {
            return null;
        }
        List<UserRelDingTalkEntity> entityList = this.userRelDingTalkRepository.findByDingTalkUserId(dingTalkUserId);
        if (CollectionUtil.isEmpty(entityList)) {
            return Collections.emptyList();
        }
        return (List<UserRelDingTalkVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entityList, UserRelDingTalkEntity.class, UserRelDingTalkVo.class, HashSet.class, ArrayList.class);
    }

    /**
     * 根据钉钉ID获取用户关联钉钉信息
     *
     * @param dingTalkUserIds 钉钉ID
     * @return java.util.List<com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    public List<UserRelDingTalkVo> findByDingTalkUserIds(List<String> dingTalkUserIds) {
        if (CollectionUtil.isEmpty(dingTalkUserIds)) {
            return null;
        }
        List<UserRelDingTalkEntity> entityList = this.userRelDingTalkRepository.findByDingTalkUserIds(dingTalkUserIds);
        if (CollectionUtil.isEmpty(entityList)) {
            return Collections.emptyList();
        }
        return (List<UserRelDingTalkVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entityList, UserRelDingTalkEntity.class, UserRelDingTalkVo.class, HashSet.class, ArrayList.class);
    }

    /**
     * 根据账号获取用户关联钉钉信息
     *
     * @param userName 账号
     * @return java.util.List<com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    public List<UserRelDingTalkVo> findByUserName(String userName) {
        if (StringUtils.isBlank(userName)) {
            return Collections.emptyList();
        }
        List<UserRelDingTalkEntity> entityList = this.userRelDingTalkRepository.findByUserName(userName);
        if (CollectionUtil.isEmpty(entityList)) {
            return Collections.emptyList();
        }
        return (List<UserRelDingTalkVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entityList, UserRelDingTalkEntity.class, UserRelDingTalkVo.class, HashSet.class, ArrayList.class);

    }

    /**
     * 根据账号集合获取用户关联钉钉信息
     *
     * @param userNameList 账号集合
     * @return java.util.List<com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    public List<UserRelDingTalkVo> findByUserNames(List<String> userNameList) {
        if (CollectionUtil.isEmpty(userNameList)) {
            return Collections.emptyList();
        }
        List<UserRelDingTalkEntity> entityList = this.userRelDingTalkRepository.findByUserNames(userNameList);
        if (CollectionUtil.isEmpty(entityList)) {
            return Collections.emptyList();
        }
        return (List<UserRelDingTalkVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entityList, UserRelDingTalkEntity.class, UserRelDingTalkVo.class, HashSet.class, ArrayList.class);
    }

    /**
     * 根据账号集合获取用户关联钉钉信息
     * key userName
     * value 钉钉用户ID Set
     *
     * @param userNameList 账号集合
     * @return java.util.List<com.biz.crm.mdm.business.user.sdk.vo.UserRelDingTalkVo>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.3 22:43
     */
    @Override
    public Map<String, Set<String>> findMapByUserNames(List<String> userNameList) {
        if (CollectionUtil.isEmpty(userNameList)) {
            return Collections.emptyMap();
        }
        List<UserRelDingTalkEntity> entityList = this.userRelDingTalkRepository.findByUserNames(userNameList);
        if (CollectionUtil.isEmpty(entityList)) {
            return Collections.emptyMap();
        }
        return entityList.stream()
                .filter(k -> StringUtils.isBlank(k.getUserName()))
                .filter(k -> StringUtils.isBlank(k.getDingTalkUserId()))
                .collect(Collectors
                        .groupingBy(UserRelDingTalkEntity::getUserName,
                                Collectors.mapping(UserRelDingTalkEntity::getDingTalkUserId,
                                        Collectors.toSet())));
    }

    /**
     * 根据账号获取用户关联钉钉用户ID
     *
     * @param userName 账号
     * @return java.util.Set<java.lang.String>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.5 11:27
     */
    @Override
    public Set<String> findDingTalkUserIdByUserName(String userName) {
        if (StringUtils.isBlank(userName)) {
            return Collections.emptySet();
        }
        List<UserRelDingTalkEntity> entityList = this.userRelDingTalkRepository.findByUserName(userName);
        if (CollectionUtil.isEmpty(entityList)) {
            return Collections.emptySet();
        }
        return entityList.stream()
                .filter(k -> StringUtils.isNotBlank(k.getDingTalkUserId()))
                .map(UserRelDingTalkEntity::getDingTalkUserId)
                .collect(Collectors.toSet());
    }

    /**
     * 根据账号获取用户关联钉钉用户ID
     *
     * @param userNameList 账号集合
     * @return java.util.Set<java.lang.String>
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.12.5 11:27
     */
    @Override
    public Set<String> findDingTalkUserIdByUserNames(List<String> userNameList) {
        if (CollectionUtil.isEmpty(userNameList)) {
            return Collections.emptySet();
        }
        List<UserRelDingTalkEntity> entityList = this.userRelDingTalkRepository.findByUserNames(userNameList);
        if (CollectionUtil.isEmpty(entityList)) {
            return Collections.emptySet();
        }
        return entityList.stream()
                .filter(k -> StringUtils.isNotBlank(k.getDingTalkUserId()))
                .map(UserRelDingTalkEntity::getDingTalkUserId)
                .collect(Collectors.toSet());
    }
}
