package com.biz.crm.mdm.business.user.local.service.internal;

import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.biz.crm.mdm.business.position.sdk.dto.PositionDto;
import com.biz.crm.mdm.business.position.sdk.enums.PositionSourceEnum;
import com.biz.crm.mdm.business.position.sdk.service.PositionVoService;
import com.biz.crm.mdm.business.position.sdk.vo.PositionVo;
import com.biz.crm.mdm.business.user.local.entity.UserEntity;
import com.biz.crm.mdm.business.user.local.repository.UserRepository;
import com.biz.crm.mdm.business.user.local.service.UserPositionService;
import com.biz.crm.mdm.business.user.local.service.UserService;
import com.biz.crm.mdm.business.user.sdk.constant.UserConstant;
import com.biz.crm.mdm.business.user.sdk.dto.UserDto;
import com.biz.crm.mdm.business.user.sdk.dto.UserPositionDto;
import com.biz.crm.mdm.business.user.sdk.enums.UserSourceEnum;
import com.biz.crm.mdm.business.user.sdk.enums.UserTypeEnum;
import com.biz.crm.mdm.business.user.sdk.service.UserVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserPositionVo;
import com.biz.crm.mn.common.base.constant.CommonConstant;
import com.biz.crm.mn.common.base.util.DateUtil;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.dto.MasterDataMdgBaseDto;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.enums.EffStatusEnum;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.service.MasterDataMdgService;
import com.biz.crm.mn.third.system.master.data.mdg.sdk.vo.MasterDataMdgUserVo;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 用户表(User)表服务实现类
 *
 * @author xi.peng
 * @since 2021-11-04 16:47:03
 */
@Slf4j
@Service
public class UserServiceImpl implements UserService {

    @Autowired(required = false)
    private UserRepository userRepository;

    @Autowired(required = false)
    private UserVoService userVoService;

    @Autowired(required = false)
    private OrgVoService orgVoService;

    @Autowired(required = false)
    private RedisMutexService redisMutexService;

    @Autowired(required = false)
    private PositionVoService positionVoService;

    @Autowired(required = false)
    private UserPositionService userPositionService;

    @Autowired(required = false)
    private MasterDataMdgService masterDataMdgService;

    @Autowired(required = false)
    private GenerateCodeService generateCodeService;


    /**
     * 批量拉取用户数据
     *
     * @param dto
     * @author huojia
     * @date 2023/1/4 11:59
     **/
    @Override
    public void pullUserList(MasterDataMdgBaseDto dto) {
        if (ObjectUtils.isEmpty(dto)) {
            dto = new MasterDataMdgBaseDto();
        }
        if (StringUtils.isEmpty(dto.getPageNum())) {
            dto.setPageNum("1");
        }
        if (StringUtils.isEmpty(dto.getPageSize())) {
            dto.setPageSize(CommonConstant.MAX_PAGE_SIZE_STR);
        }
        boolean lock = true;
        String lockKey = DateUtil.format(new Date(), DateUtil.DEFAULT_YEAR_MONTH_DAY);
        if (!StringUtils.isEmpty(dto.getUdate())) {
            lockKey = dto.getUdate();
        }
        try {
            lock = this.lock(lockKey);
            if (!lock) {
                return;
            }
            List<MasterDataMdgUserVo> masterDataMdgUserVos = masterDataMdgService.pullUserList(dto);
            if (CollectionUtils.isEmpty(masterDataMdgUserVos)) {
                return;
            }
            // 根据用户编码查询用户，区分更新、新增操作
            Set<String> userNameSet = masterDataMdgUserVos.stream().map(MasterDataMdgUserVo::getEMPLID).collect(Collectors.toSet());
            List<UserEntity> userEntities = userRepository.findByUserNames(new ArrayList<>(userNameSet));
            Map<String, UserEntity> userMap = userEntities.stream().collect(Collectors.toMap(UserEntity::getUserName, Function.identity()));
            // 查询手机号
            Set<String> phoneSet = masterDataMdgUserVos.stream().map(MasterDataMdgUserVo::getMNCELLPHONE).collect(Collectors.toSet());
            List<UserEntity> userPhoneList = userRepository.listByUserPhone(new ArrayList<>(phoneSet));
            Map<String, String> phoneMap = userPhoneList.stream().collect(Collectors.toMap(UserEntity::getUserPhone, UserEntity::getUserName));
            // 校验数据，用户上对应的职位都是已有职位
            List<MasterDataMdgUserVo> successList = new ArrayList<>();
            List<UserDto> pullList = this.userValidate(masterDataMdgUserVos, successList);
            List<UserPositionDto> userPositionList = this.positionValidate(successList);
            Map<String, UserPositionDto> userPositionMap = userPositionList.stream().collect(Collectors.toMap(UserPositionDto::getUserName, Function.identity()));
            // 区分更新、新增
            List<UserDto> saveList = new ArrayList<>();
            List<UserDto> updateList = new ArrayList<>();
            // 区分新增、编辑
            pullList.forEach(pull -> {
                if (!userPositionMap.containsKey(pull.getUserName())) {
                    return;
                }
                pull.setPositionList(Collections.singletonList(userPositionMap.get(pull.getUserName())));
                pull.setUserType(UserTypeEnum.USER.getCode());
                if (userMap.containsKey(pull.getUserName())) {
                    UserEntity userEntity = userMap.get(pull.getUserName());
                    pull.setId(userEntity.getId());
                    pull.setUserPassword(userEntity.getUserPassword());
                    pull.setMd5Flag(BooleanEnum.FALSE.getCapital());
                    pull.setUserCode(userEntity.getUserCode());
                    updateList.add(pull);
                } else {
                    if (phoneMap.containsKey(pull.getUserPhone())) {
                        return;
                    }
                    pull.setMd5Flag(BooleanEnum.FALSE.getCapital());
                    pull.setUserPassword("e19481fb576bade223bc94e280cb6323");
                    saveList.add(pull);
                }
            });
            if (!CollectionUtils.isEmpty(saveList)) {
                log.info(">>>>>>>>>>-----------------" + "本次共拉取到新增数据：" + saveList.size() + "条--------------------<<<<<<<<<<<");
                userVoService.createMdgBatch(saveList);
            }
            if (!CollectionUtils.isEmpty(updateList)) {
                log.info(">>>>>>>>>>-----------------" + "本次共拉取到更新数据：" + updateList.size() + "条--------------------<<<<<<<<<<<");
                userVoService.updateMdgBatch(updateList);
            }
        } finally {
            if (lock) {
                this.unLock(lockKey);
            }
        }
    }

    /**
     * 校验职位数据，同时生成新职位
     *
     * @return masterDataMdgUserVos
     * @author huojia
     * @date 2023/1/4 14:39
     **/
    private List<UserPositionDto> positionValidate(List<MasterDataMdgUserVo> masterDataMdgUserVos) {
        Set<String> mdgPositionCodeSet = masterDataMdgUserVos.stream()
                .map(MasterDataMdgUserVo::getPOSITIONNBR)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
        Set<String> userNameSet = masterDataMdgUserVos.stream().map(MasterDataMdgUserVo::getEMPLID).collect(Collectors.toSet());
        Set<String> mdgParentCodeSet = masterDataMdgUserVos.stream().map(MasterDataMdgUserVo::getMNJOBEMPLID).collect(Collectors.toSet());
        // 查询职位明细
        List<PositionVo> mdgPositionVoList = positionVoService.findByMdgPositionCodeList(new ArrayList<>(mdgPositionCodeSet));
        Map<String, List<PositionVo>> mdgPositionVoMap = mdgPositionVoList.stream()
                .collect(Collectors.groupingBy(PositionVo::getMdgPositionCode));
        // 根据用户编码查询用户职位关系
        List<UserPositionVo> userPositionVoList = userPositionService.listByUserNameList(new ArrayList<>(userNameSet));
        List<UserPositionVo> parentUserPositionVoList = userPositionService.listByUserNameList(new ArrayList<>(mdgParentCodeSet));
        List<String> parentCodeList = parentUserPositionVoList.stream().map(UserPositionVo::getPositionCode).collect(Collectors.toList());
        List<PositionVo> parentList = positionVoService.findByIdsOrCodes(null, parentCodeList);
        Map<String, PositionVo> parentMap = parentList.stream().collect(Collectors.toMap(PositionVo::getPositionCode, Function.identity()));
        Map<String, List<UserPositionVo>> parentUserPositionMap = parentUserPositionVoList.stream().collect(Collectors.groupingBy(UserPositionVo::getUserName));
        Map<String, List<UserPositionVo>> userPositionMap = userPositionVoList.stream().collect(Collectors.groupingBy(UserPositionVo::getUserName));
        // 记录重复的职位数据
        Map<String, List<UserPositionDto>> dupPositionMap = new HashMap<>();
        // 组装用户职位
        List<UserPositionDto> positionList = new ArrayList<>();
        masterDataMdgUserVos.forEach(masterDataMdgUserVo -> {
            UserPositionDto userPositionDto = new UserPositionDto();
            userPositionDto.setUserName(masterDataMdgUserVo.getEMPLID());
            userPositionDto.setMdgPositionCode(masterDataMdgUserVo.getPOSITIONNBR());
            userPositionDto.setPositionSource(PositionSourceEnum.MDG.getCode());
            userPositionDto.setParentUserName(masterDataMdgUserVo.getMNJOBEMPLID());
            userPositionDto.setOrgCode(masterDataMdgUserVo.getDEPTID());
            AtomicBoolean existFlag = new AtomicBoolean(false);
            // 获取职位编码，如果之前存在关系，直接使用之前的关系
            if (userPositionMap.containsKey(masterDataMdgUserVo.getEMPLID())) {
                List<UserPositionVo> mdgUserPositionList = userPositionMap.get(masterDataMdgUserVo.getEMPLID()).stream()
                        .filter(userPositionVo -> PositionSourceEnum.MDG.getCode().equals(userPositionVo.getPositionSource())
                                && userPositionDto.getMdgPositionCode().equals(userPositionVo.getMdgPositionCode()))
                        .collect(Collectors.toList());
                if (!CollectionUtils.isEmpty(mdgUserPositionList)) {
                    userPositionDto.setPositionCode(mdgUserPositionList.get(0).getPositionCode());
                    userPositionDto.setPrimaryFlag(mdgUserPositionList.get(0).getPrimaryFlag());
                    userPositionDto.setOperationType(false);
                    existFlag.set(true);
                }
            }
            if (!existFlag.get()) {
                // 不存在该用户、该职位关系，创建职位
                // 之前存在该用户的其他关系，则必定非主职位，不存在关系，则设置为主职位
                if (userPositionMap.containsKey(masterDataMdgUserVo.getEMPLID())) {
                    userPositionDto.setPrimaryFlag(false);
                } else {
                    userPositionDto.setPrimaryFlag(true);
                }
                userPositionDto.setOperationType(true);
                if (dupPositionMap.containsKey(userPositionDto.getMdgPositionCode())) {
                    UserPositionDto dupPosition = dupPositionMap.get(userPositionDto.getMdgPositionCode()).stream().max(Comparator.comparingInt(UserPositionDto::getSuffixSequence)).get();
                    userPositionDto.setPositionCode(generateCodeService.generateCode("ZW", 1).get(0));
                    userPositionDto.setPositionName(dupPosition.getPositionName().split("_")[0] + "_" + (dupPosition.getSuffixSequence() + 1));
                    userPositionDto.setSuffixSequence(dupPosition.getSuffixSequence() + 1);
                    List<UserPositionDto> userPositionDtoList = dupPositionMap.get(userPositionDto.getMdgPositionCode());
                    userPositionDtoList.add(userPositionDto);
                    dupPositionMap.put(userPositionDto.getMdgPositionCode(), userPositionDtoList);
                } else {
                    List<UserPositionDto> userPositionDtoList = new ArrayList<>();
                    PositionVo positionVo = mdgPositionVoMap.get(userPositionDto.getMdgPositionCode()).stream()
                            .filter(mdgPositionVo -> mdgPositionVo.getSuffixSequence() != null)
                            .max(Comparator.comparingInt(PositionVo::getSuffixSequence)).get();
                    userPositionDto.setPositionCode(generateCodeService.generateCode("ZW", 1).get(0));
                    userPositionDto.setPositionName(positionVo.getPositionName().split("_")[0] + "_" + (positionVo.getSuffixSequence() + 1));
                    userPositionDto.setSuffixSequence(positionVo.getSuffixSequence() + 1);
                    userPositionDtoList.add(userPositionDto);
                    dupPositionMap.put(userPositionDto.getMdgPositionCode(), userPositionDtoList);
                }
            }
            positionList.add(userPositionDto);
        });
        // 组装上级数据：1、上级数据在本次同步的数据中  2、上级数据在之前的数据中
        Map<String, UserPositionDto> userPositionDtoMap = positionList.stream().collect(Collectors.toMap(UserPositionDto::getUserName, Function.identity()));
        positionList.forEach(position -> {
            if (StringUtils.isEmpty(position.getParentUserName())) {
                return;
            }
            if (userPositionDtoMap.containsKey(position.getParentUserName())) {
                position.setParentCode(userPositionDtoMap.get(position.getParentUserName()).getPositionCode());
            } else if (parentUserPositionMap.containsKey(position.getParentUserName())) {
                parentUserPositionMap.get(position.getParentUserName()).forEach(parentUserPositionVo -> {
                    if (PositionSourceEnum.MDG.getCode().equals(parentUserPositionVo.getPositionSource())) {
                        if (parentMap.containsKey(parentUserPositionVo.getPositionCode())) {
                            position.setMdgParentCode(parentMap.get(parentUserPositionVo.getPositionCode()).getMdgPositionCode());
                            position.setParentCode(parentMap.get(parentUserPositionVo.getPositionCode()).getPositionCode());
                        }
                    }
                });
            }
        });
        return positionList;
    }

    /**
     * 构建职位数据
     *
     * @param pullList
     * @author huojia
     * @date 2023/1/4 11:29
     **/
    private void buildPosition(List<UserDto> pullList) {
        // 生成职位数据
        List<UserPositionDto> userPositionDtoList = new ArrayList<>();
        pullList.forEach(pull -> {
            userPositionDtoList.add(pull.getPositionList().get(0));
        });
        // 查询用户职位关系
        Set<String> userNameSet = userPositionDtoList.stream().map(UserPositionDto::getUserName).collect(Collectors.toSet());
        List<UserPositionVo> userPositionVoList = userPositionService.listByUserNameList(new ArrayList<>(userNameSet));
        Map<String, String> userMdgPosMap = userPositionVoList.stream().collect(Collectors.toMap(UserPositionVo::getUserName, UserPositionVo::getMdgPositionCode));
        // 查询职位，上级职位
        Set<String> mdgPositionCodeSet = userPositionDtoList.stream().map(UserPositionDto::getMdgPositionCode).collect(Collectors.toSet());
        Set<String> mdgParentCodeSet = Optional.ofNullable(userPositionDtoList.stream().map(UserPositionDto::getMdgParentCode).filter(Objects::nonNull).collect(Collectors.toSet())).orElse(new HashSet<>());
        mdgPositionCodeSet.addAll(mdgParentCodeSet);
        List<PositionVo> positionVoList = positionVoService.findByMdgPositionCodeList(new ArrayList<>(mdgPositionCodeSet));
        Map<String, List<PositionVo>> mdgPositionMap = positionVoList.stream().collect(Collectors.groupingBy(PositionVo::getMdgPositionCode));
        // 根据职位获取对应的序列号，同时记录要新增的职位
        List<PositionDto> savePositionList = new ArrayList<>();
        // 同步过来的数据不去管角色问题，只要不覆盖原角色即可，每天更新的角色上的职位数据，对应的mdg职位编码肯定在系统中存在（职位接口会同步过来）
        // 所以不用管职位上的组织、角色、名称问题，只需要把同mdg职位编码的数据同步即可
        pullList.forEach(pull -> {
            pull.getPositionList().forEach(userPositionDto -> {
                // 判断是否有用户、mdg职位对应关系
                // 没有用户职位关系则新增，有则不管（数据的更新会在职位表）
                if (!userMdgPosMap.containsKey(userPositionDto.getUserName())
                        || !userMdgPosMap.get(userPositionDto.getUserName()).equals(userPositionDto.getPositionCode())) {
                    userPositionDto.setOperationType(true);
                }
            });
        });
    }

    /**
     * 用户校验
     *
     * @param masterDataMdgUserVos
     * @return java.util.List<com.biz.crm.mdm.business.user.sdk.dto.UserDto>
     * @author huojia
     * @date 2022/12/30 22:34
     **/
    private List<UserDto> userNewValidate(List<MasterDataMdgUserVo> masterDataMdgUserVos) {
        Map<String, MasterDataMdgUserVo> masterDataMdgUserVoMap = masterDataMdgUserVos.stream()
                .collect(Collectors.toMap(MasterDataMdgUserVo::getEMPLID, Function.identity(), (oldVo, newVo) -> newVo));
        Set<String> mdgPositionCodeSet = masterDataMdgUserVos.stream()
                .map(MasterDataMdgUserVo::getPOSITIONNBR)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
        Set<String> userNameSet = masterDataMdgUserVos.stream().map(MasterDataMdgUserVo::getEMPLID).collect(Collectors.toSet());
        Set<String> mdgParentCodeSet = masterDataMdgUserVos.stream().map(MasterDataMdgUserVo::getMNJOBEMPLID).collect(Collectors.toSet());
        Map<String, List<MasterDataMdgUserVo>> codeMap = new HashMap<>(8);
        // 查询职位明细
        List<PositionVo> mdgPositionVoList = positionVoService.findByMdgPositionCodeList(new ArrayList<>(mdgPositionCodeSet));
        Map<String, PositionVo> mdgPositionVoMap = mdgPositionVoList.stream()
                .collect(Collectors.toMap(PositionVo::getMdgPositionCode, Function.identity(), (oldVo, newVo) -> newVo));
        // 查询组织数据
        Set<String> orgCodeSet = masterDataMdgUserVos.stream().map(MasterDataMdgUserVo::getDEPTID).filter(Objects::nonNull).collect(Collectors.toSet());
        List<OrgVo> orgVos = Optional.ofNullable(orgVoService.findByOrgCodes(new ArrayList<>(orgCodeSet))).orElse(new ArrayList<>());
        Map<String, OrgVo> orgVoMap = orgVos.stream().collect(Collectors.toMap(OrgVo::getOrgCode, Function.identity()));
        // 根据用户编码查询用户职位关系
        List<UserPositionVo> userPositionVoList = userPositionService.listByUserNameList(new ArrayList<>(userNameSet));
        List<UserPositionVo> parentUserPositionVoList = userPositionService.listByUserNameList(new ArrayList<>(mdgParentCodeSet));
        List<String> positionCodeList = parentUserPositionVoList.stream().map(UserPositionVo::getPositionCode).collect(Collectors.toList());
        List<PositionVo> byIdsOrCodes = positionVoService.findByIdsOrCodes(null, positionCodeList);
        Map<String, PositionVo> positionMap = byIdsOrCodes.stream().collect(Collectors.toMap(PositionVo::getPositionCode, Function.identity()));
        Map<String, List<UserPositionVo>> parentUserPositionMap = parentUserPositionVoList.stream().collect(Collectors.groupingBy(UserPositionVo::getUserName));
        Map<String, List<UserPositionVo>> userPositionMap = userPositionVoList.stream().collect(Collectors.groupingBy(UserPositionVo::getUserName));
        List<UserDto> pullList = new ArrayList<>();
        // 记录失败数据
        List<MasterDataMdgUserVo> failList = new ArrayList<>();
        masterDataMdgUserVos.forEach(masterDataMdgUserVo -> {
            // 只拉取有效状态数据
            if (StringUtils.isEmpty(masterDataMdgUserVo.getHRSTATUS())
                    || EffStatusEnum.DISABLE.getCode().equals(masterDataMdgUserVo.getHRSTATUS())) {
                return;
            }
            if (StringUtils.isEmpty(masterDataMdgUserVo.getEMPLID())) {
                failList.add(masterDataMdgUserVo);
                return;
            }
            if (StringUtils.equals(masterDataMdgUserVo.getEMPLID(), masterDataMdgUserVo.getMNJOBEMPLID())) {
                failList.add(masterDataMdgUserVo);
                return;
            }
            if (StringUtils.isEmpty(masterDataMdgUserVo.getDEPTID())) {
                failList.add(masterDataMdgUserVo);
                return;
            }
            if (!mdgPositionVoMap.containsKey(masterDataMdgUserVo.getPOSITIONNBR())) {
                failList.add(masterDataMdgUserVo);
                return;
            }
            if (orgVoMap.containsKey(masterDataMdgUserVo.getDEPTID())) {
                failList.add(masterDataMdgUserVo);
                return;
            }
            UserDto userDto = new UserDto();
            List<MasterDataMdgUserVo> userVos = Optional.ofNullable(codeMap.get(masterDataMdgUserVo.getEMPLID())).orElse(new ArrayList<>());
            userVos.add(masterDataMdgUserVo);
            if (userVos.size() > 1) {
                throw new RuntimeException("数据：" + JSON.toJSONString(userVos) + "重复拉取，请检查！");
            }
            codeMap.put(masterDataMdgUserVo.getEMPLID(), userVos);
            // 组装用户数据
            userDto.setUserName(masterDataMdgUserVo.getEMPLID());
            userDto.setFullName(masterDataMdgUserVo.getNAME());
            userDto.setUserPhone(masterDataMdgUserVo.getMNCELLPHONE());
            userDto.setCostCenterCode(masterDataMdgUserVo.getMN_SALARY_CENTER());
            userDto.setEmployeeStatus(masterDataMdgUserVo.getHRSTATUS());
            userDto.setUserSource(UserSourceEnum.MDG.getCode());
            userDto.setMdgParentUserName(masterDataMdgUserVo.getMNJOBEMPLID());
            // 组装用户职位
            List<UserPositionDto> positionList = new ArrayList<>();
            UserPositionDto userPositionDto = new UserPositionDto();
            userPositionDto.setUserName(userDto.getUserName());
            userPositionDto.setMdgPositionCode(masterDataMdgUserVo.getPOSITIONNBR());
            userPositionDto.setPositionSource(PositionSourceEnum.MDG.getCode());
            // 获取上级：上级在职位数据中存在
            if (masterDataMdgUserVoMap.containsKey(masterDataMdgUserVo.getMNJOBEMPLID())) {
                MasterDataMdgUserVo parentUser = masterDataMdgUserVoMap.get(masterDataMdgUserVo.getMNJOBEMPLID());
                if (ObjectUtils.isEmpty(parentUser)) {
                    userPositionDto.setMdgParentCode(parentUser.getPOSITIONNBR());
                }
            } else if (parentUserPositionMap.containsKey(masterDataMdgUserVo.getMNJOBEMPLID())) {
                parentUserPositionMap.get(masterDataMdgUserVo.getMNJOBEMPLID()).forEach(parentUserPositionVo -> {
                    if (PositionSourceEnum.MDG.getCode().equals(parentUserPositionVo.getPositionSource())) {
                        userPositionDto.setMdgParentCode(positionMap.get(parentUserPositionVo.getPositionCode()).getMdgPositionCode());
                    }
                });
            } else {
                failList.add(masterDataMdgUserVo);
                return;
            }
            // 如果是之前存在的数据，需要查看之前用户职位关系，是否主职位；如果不存在，默认主职位
            // 同时保留之前已维护的角色
            if (userPositionMap.containsKey(userDto.getUserName())) {
                userPositionMap.get(userDto.getUserName()).forEach(userPositionVo -> {
                    if (userPositionVo.getPositionCode().equals(masterDataMdgUserVo.getPOSITIONNBR())) {
                        userPositionDto.setPrimaryFlag(userPositionVo.getPrimaryFlag());
                    }
                });
            } else {
                userPositionDto.setPrimaryFlag(true);
            }
            userPositionDto.setOperationType(false);
            positionList.add(userPositionDto);
            userDto.setPositionList(positionList);
            pullList.add(userDto);
        });
        log.info("拉取用户失败的数据————————————————————————————>" + JSON.toJSONString(failList));
        return pullList;
    }

    /**
     * 校验职位级别、职位数据
     *
     * @param masterDataMdgUserVos
     * @return java.util.List<com.biz.crm.mdm.business.user.sdk.dto.UserDto>
     * @author huojia
     * @date 2022/12/23 21:39
     **/
    private List<UserDto> userValidate(List<MasterDataMdgUserVo> masterDataMdgUserVos, List<MasterDataMdgUserVo> successList) {
        Set<String> mdgPositionCodeSet = masterDataMdgUserVos.stream()
                .map(MasterDataMdgUserVo::getPOSITIONNBR)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
        Map<String, List<MasterDataMdgUserVo>> codeMap = new HashMap<>(8);
        // 查询职位明细
        List<PositionVo> mdgPositionVoList = positionVoService.findByMdgPositionCodeList(new ArrayList<>(mdgPositionCodeSet));
        Map<String, PositionVo> mdgPositionVoMap = mdgPositionVoList.stream()
                .collect(Collectors.toMap(PositionVo::getMdgPositionCode, Function.identity(), (oldVo, newVo) -> newVo));
        // 查询组织数据
        Set<String> orgCodeSet = masterDataMdgUserVos.stream().map(MasterDataMdgUserVo::getDEPTID).filter(Objects::nonNull).collect(Collectors.toSet());
        List<OrgVo> orgVos = Optional.ofNullable(orgVoService.findByOrgCodes(new ArrayList<>(orgCodeSet))).orElse(new ArrayList<>());
        Map<String, OrgVo> orgVoMap = orgVos.stream().collect(Collectors.toMap(OrgVo::getOrgCode, Function.identity()));
        List<UserDto> pullList = new ArrayList<>();
        // 记录失败数据
        List<MasterDataMdgUserVo> failList = new ArrayList<>();
        // 记录重复手机号
        Map<String, List<String>> dupPhoneMap = new HashMap<>();
        masterDataMdgUserVos.forEach(masterDataMdgUserVo -> {
            // 只拉取有效状态数据
            if (StringUtils.isEmpty(masterDataMdgUserVo.getHRSTATUS())
                    || EffStatusEnum.DISABLE.getCode().equals(masterDataMdgUserVo.getHRSTATUS())) {
                failList.add(masterDataMdgUserVo);
                return;
            }
            if (StringUtils.isEmpty(masterDataMdgUserVo.getEMPLID())) {
                failList.add(masterDataMdgUserVo);
                return;
            }
            if (StringUtils.equals(masterDataMdgUserVo.getEMPLID(), masterDataMdgUserVo.getMNJOBEMPLID())) {
                failList.add(masterDataMdgUserVo);
                return;
            }
            if (StringUtils.isEmpty(masterDataMdgUserVo.getDEPTID())) {
                failList.add(masterDataMdgUserVo);
                return;
            }
            if (!mdgPositionVoMap.containsKey(masterDataMdgUserVo.getPOSITIONNBR())) {
                failList.add(masterDataMdgUserVo);
                return;
            }
            if (!orgVoMap.containsKey(masterDataMdgUserVo.getDEPTID())) {
                failList.add(masterDataMdgUserVo);
                return;
            }
            if (StringUtils.isEmpty(masterDataMdgUserVo.getMNCELLPHONE())) {
                failList.add(masterDataMdgUserVo);
                return;
            }
            UserDto userDto = new UserDto();
            List<MasterDataMdgUserVo> userVos = Optional.ofNullable(codeMap.get(masterDataMdgUserVo.getEMPLID())).orElse(new ArrayList<>());
            userVos.add(masterDataMdgUserVo);
            if (userVos.size() > 1) {
                throw new RuntimeException("数据：" + JSON.toJSONString(userVos) + "重复拉取，请检查！");
            }
            // 手机号去重
            if (dupPhoneMap.containsKey(masterDataMdgUserVo.getMNCELLPHONE())) {
                List<String> dupPhoneList = dupPhoneMap.get(masterDataMdgUserVo.getMNCELLPHONE());
                dupPhoneList.add(masterDataMdgUserVo.getEMPLID());
                dupPhoneMap.put(masterDataMdgUserVo.getMNCELLPHONE(), dupPhoneList);
                failList.add(masterDataMdgUserVo);
                return;
            } else {
                List<String> dupPhoneList = new ArrayList<>();
                dupPhoneList.add(masterDataMdgUserVo.getEMPLID());
                dupPhoneMap.put(masterDataMdgUserVo.getMNCELLPHONE(), dupPhoneList);
            }
            codeMap.put(masterDataMdgUserVo.getEMPLID(), userVos);
            // 组装用户数据
            userDto.setUserName(masterDataMdgUserVo.getEMPLID());
            userDto.setFullName(masterDataMdgUserVo.getNAME());
            userDto.setUserPhone(masterDataMdgUserVo.getMNCELLPHONE());
            userDto.setCostCenterCode(masterDataMdgUserVo.getMN_SALARY_CENTER());
            userDto.setEmployeeStatus(masterDataMdgUserVo.getHRSTATUS());
            userDto.setUserSource(UserSourceEnum.MDG.getCode());
            userDto.setMdgParentUserName(masterDataMdgUserVo.getMNJOBEMPLID());
            pullList.add(userDto);
            successList.add(masterDataMdgUserVo);
        });
        log.info("拉取用户失败的数据共" + failList.size() + "条————————————————————————————>" + JSON.toJSONString(failList));
        List<String> userNameList = new ArrayList<>();
        if (MapUtils.isNotEmpty(dupPhoneMap)) {
            dupPhoneMap.forEach((key, value) -> {
                if (value.size() > 1) {
                    userNameList.addAll(value);
                }
            });
        }
        log.info("手机号码重复的用户共：" + userNameList.size() + "条————————————————————————————>" + JSON.toJSONString(userNameList));
        return pullList;
    }

    /**
     * 解锁
     *
     * @param yearMonthDay
     * @author huojia
     * @date 2022/12/10 14:30
     **/
    private void unLock(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取员工解锁失败，日期不能为空！");
        }
        redisMutexService.unlock(UserConstant.USER_LOCK + yearMonthDay);
    }

    /**
     * 加锁
     *
     * @param yearMonthDay
     * @return boolean
     * @author huojia
     * @date 2022/12/10 14:30
     **/
    private boolean lock(String yearMonthDay) {
        if (StringUtils.isEmpty(yearMonthDay)) {
            throw new RuntimeException("拉取员工加锁失败，日期不能为空！");
        }
        return this.redisMutexService.tryLock(UserConstant.USER_LOCK + yearMonthDay, TimeUnit.HOURS, 12);
    }
}
