package com.biz.crm.mdm.business.user.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.enums.LockStateEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.org.sdk.common.constant.OrgCodeConstant;
import com.biz.crm.mdm.business.org.sdk.service.OrgPositionVoService;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgPositionVo;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.biz.crm.mdm.business.position.sdk.dto.PositionDto;
import com.biz.crm.mdm.business.position.sdk.dto.PositionRelationDto;
import com.biz.crm.mdm.business.position.sdk.dto.RelationDataDto;
import com.biz.crm.mdm.business.position.sdk.service.PositionVoService;
import com.biz.crm.mdm.business.position.sdk.vo.PositionVo;
import com.biz.crm.mdm.business.user.local.entity.UserEntity;
import com.biz.crm.mdm.business.user.local.entity.UserPositionEntity;
import com.biz.crm.mdm.business.user.local.repository.UserPositionRepository;
import com.biz.crm.mdm.business.user.local.repository.UserRepository;
import com.biz.crm.mdm.business.user.sdk.constant.UserConstant;
import com.biz.crm.mdm.business.user.sdk.dto.UserChangePasswordDto;
import com.biz.crm.mdm.business.user.sdk.dto.UserConditionDto;
import com.biz.crm.mdm.business.user.sdk.dto.UserDto;
import com.biz.crm.mdm.business.user.sdk.dto.UserForceChangePasswordDto;
import com.biz.crm.mdm.business.user.sdk.dto.UserPageDto;
import com.biz.crm.mdm.business.user.sdk.dto.UserPositionDto;
import com.biz.crm.mdm.business.user.sdk.enums.UserTypeEnum;
import com.biz.crm.mdm.business.user.sdk.event.UserEventListener;
import com.biz.crm.mdm.business.user.sdk.service.UserPositionVoService;
import com.biz.crm.mdm.business.user.sdk.service.UserVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserPositionVo;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.Aes128Utils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.security.sdk.password.PasswordValidateService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import joptsimple.internal.Strings;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.Pageable;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.DigestUtils;

import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 用户表(User)表服务实现类
 *
 * @author xi.peng
 * @since 2021-11-04 16:47:03
 */
@Slf4j
@Service
public class UserVoServiceImpl implements UserVoService {

  @Autowired(required = false)
  private UserRepository userRepository;

  @Autowired(required = false)
  private PositionVoService positionVoService;

  @Autowired(required = false)
  private OrgVoService orgVoService;

  @Autowired(required = false)
  private UserPositionVoService userPositionVoService;

  @Autowired(required = false)
  @Lazy
  private List<UserEventListener> userEventListeners;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private UserPositionRepository userPositionRepository;

  @Autowired(required = false)
  private OrgPositionVoService orgPositionVoService;

  @Autowired(required = false)
  private StringRedisTemplate stringRedisTemplate;

  @Autowired
  private PasswordValidateService passwordValidateService;

  @Override
  public Page<UserVo> findByConditions(Pageable pageable, UserPageDto dto) {
    return this.userRepository.findByConditions(pageable, dto);
  }

  @Override
  public UserVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    //重构查询方法
    UserEntity entity = this.userRepository.findByIdAndTenantCode(id, TenantUtils.getTenantCode());
    if (entity == null) {
      return null;
    }
    UserVo userVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, UserVo.class, HashSet.class, ArrayList.class);
    if (StringUtils.isNotBlank(userVo.getSysLoginPort())) {
      userVo.setSysLoginPorts(Arrays.asList(userVo.getSysLoginPort().split(",")));
    }
    List<UserPositionEntity> userPositions = this.userPositionRepository.findByUserName(TenantUtils.getTenantCode(), userVo.getUserName());
    if (CollectionUtils.isEmpty(userPositions)) {
      return userVo;
    }
    //组装职位关联的组织数据
    List<UserPositionVo> userPositionVos = (List<UserPositionVo>) this.nebulaToolkitService.copyCollectionByWhiteList(userPositions, UserPositionEntity.class, UserPositionVo.class, HashSet.class, ArrayList.class);
    if (!CollectionUtils.isEmpty(userPositions)) {
      List<String> positionCodes = userPositions.stream().map(UserPositionEntity::getPositionCode).collect(Collectors.toList());
      List<OrgPositionVo> orgPositionVos = this.orgPositionVoService.findByPositionCodes(positionCodes);
      //组装职位关联的组织数据
      if (!CollectionUtils.isEmpty(orgPositionVos)) {
        Map<String, List<OrgPositionVo>> orgPositionVoMap = orgPositionVos.stream().collect(Collectors.groupingBy(OrgPositionVo::getPositionCode));
        for (UserPositionVo userPositionVo : userPositionVos) {
          List<OrgPositionVo> orgPositionVoList = orgPositionVoMap.get(userPositionVo.getPositionCode());
          if (CollectionUtils.isEmpty(orgPositionVoList)) {
            continue;
          }
          Set<String> orgCodes = orgPositionVoList.stream().map(OrgPositionVo::getOrgCode).collect(Collectors.toSet());
          userPositionVo.setOrgCodes(orgCodes);
        }
      }
    }
    //组装职位数据
    userVo.setPositionList(userPositionVos);
    return userVo;
  }

  @Override
  public UserVo findByUserName(String userName) {
    if (StringUtils.isBlank(userName)) {
      return null;
    }
    String tenantCode = TenantUtils.getTenantCode();
    UserEntity entity = this.userRepository.findByUserName(tenantCode, userName);
    if (entity == null) {
      return null;
    }
    UserVo userVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, UserVo.class, HashSet.class, ArrayList.class);
    if (StringUtils.isNotBlank(userVo.getSysLoginPort())) {
      userVo.setSysLoginPorts(Arrays.asList(userVo.getSysLoginPort().split(",")));
    }
    List<UserPositionVo> userPosits = userPositionVoService.findByUserName(tenantCode, userName);
    if (!CollectionUtils.isEmpty(userPosits)) {
      Optional<UserPositionVo> currentOptional = userPosits.stream().filter(f -> Boolean.TRUE.equals(f.getCurrentFlag())).findFirst();
      if (currentOptional.isPresent()) {
        userVo.setPositionCode(currentOptional.get().getPositionCode());
        OrgVo orgVo = orgPositionVoService.findByPositionCode(currentOptional.get().getPositionCode());
        if (ObjectUtils.isNotEmpty(orgVo)) {
          userVo.setOrgCode(orgVo.getOrgCode());
          userVo.setOrgName(orgVo.getOrgName());
        }
      }
    }
    return userVo;
  }

  @Override
  @Transactional
  public void create(UserDto dto) {
    this.createValidation(dto);
    this.validationPosition(dto);
    if (StringUtils.isBlank(dto.getUserCode())) {
      // redis生物料编码code
      List<String> codeList = this.generateCodeService.generateCode(UserConstant.USER_CODE, 1);
      Validate.isTrue(CollectionUtils.isNotEmpty(codeList), "添加信息时，生成用户编码失败！");
      UserEntity dbEntity = this.userRepository.findByUserCode(TenantUtils.getTenantCode(), codeList.get(0));
      Validate.isTrue(Objects.isNull(dbEntity), String.format("添加信息时，用户编码[%s]已存在！", codeList.get(0)));
      dto.setUserCode(codeList.get(0));
    } else {
      UserEntity dbEntity = this.userRepository.findByUserCode(TenantUtils.getTenantCode(), dto.getUserCode());
      Validate.isTrue(Objects.isNull(dbEntity), String.format("添加信息时，用户编码[%s]已存在！", dto.getUserCode()));
    }

    //基本信息
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    dto.setLockState(LockStateEnum.UNLOCK.getCode());
    dto.setForceChangePassword(Boolean.TRUE);
    dto.setUpdatePasswordDate(new Date());


    // 企业用户，企业采购用户
    Validate.isTrue(StringUtils.equalsAny(dto.getUserType(), UserTypeEnum.USER.getCode(), UserTypeEnum.USER_GUIDE.getCode()), "添加信息时,用户类型非企业用户【%s】和企业导购用户【%s】", UserTypeEnum.USER.getCode(), UserTypeEnum.USER_GUIDE.getCode());
    UserEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, UserEntity.class, HashSet.class, ArrayList.class);
    //新增租户编号
    entity.setTenantCode(TenantUtils.getTenantCode());
    this.userRepository.saveOrUpdate(entity);
    // 更新用户-职位关联关系
    this.rebindUserPositionRelation(dto);

    // 发送新增通知
    if (CollectionUtils.isNotEmpty(userEventListeners)) {
      UserVo vo = this.nebulaToolkitService.copyObjectByWhiteList(entity, UserVo.class, HashSet.class, ArrayList.class);
      this.userEventListeners.forEach(event -> event.onCreate(vo));
    }
  }

  /**
   * 批量新增用户（拉取MDG用户数据专用）
   *
   * @param saveList
   * @author huojia
   * @date 2022/12/24 16:32
   **/
  @Override
  public void createMdgBatch(List<UserDto> saveList) {
    // 批量校验数据
    this.createValidationBatch(saveList);
    // 企业用户，企业采购用户
    List<UserEntity> userEntities = (List<UserEntity>) this.nebulaToolkitService.copyCollectionByWhiteList(saveList, UserDto.class, UserEntity.class, LinkedHashSet.class, ArrayList.class);
    // 更新用户-职位关联关系
    this.rebindUserPositionRelationBatch(saveList, userEntities);
  }

  /**
   * 批量编辑用户（拉取MDG用户数据专用）
   *
   * @param updateList
   * @author huojia
   * @date 2022/12/24 17:39
   **/
  @Override
  public void updateMdgBatch(List<UserDto> updateList) {
    this.updateValidationBatch(updateList);
    //重构查询方法
    List<UserEntity> userEntities = (List<UserEntity>) this.nebulaToolkitService.copyCollectionByWhiteList(updateList, UserDto.class, UserEntity.class, LinkedHashSet.class, ArrayList.class);
    // 更新用户-职位关联关系
    this.rebindUserPositionRelationBatch(updateList, userEntities);
  }

  @Override
  public List<UserVo> listByUserPhone(List<String> userPhoneList) {
    if (CollectionUtils.isEmpty(userPhoneList)) {
      return Lists.newArrayList();
    }
    List<UserEntity> userEntityList = userRepository.listByUserPhone(userPhoneList);
    return (List<UserVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
            userEntityList, UserEntity.class, UserVo.class, LinkedHashSet.class, ArrayList.class
    );
  }

  @Override
  public List<UserVo> findUserByOrgCodes(List<String> orgCodeList) {
    return userRepository.findUserByOrgCodes(orgCodeList);
  }

  /**
   * 按userName查询用户信息
   *
   * @param userNameList 用户
   * @return 用户列表
   */
  @Override
  public List<UserVo> findByUserNames(List<String> userNameList) {
    if (CollectionUtils.isEmpty(userNameList)) {
      return Lists.newArrayList();
    }
    List<UserEntity> userEntityList = userRepository.findByUserNames(userNameList);
    return (List<UserVo>) this.nebulaToolkitService.copyCollectionByWhiteList(
            userEntityList, UserEntity.class, UserVo.class, LinkedHashSet.class, ArrayList.class
    );
  }

  /**
   * 按userName查询用户详细信息
   *
   * @param userName 用户账号
   * @return 用户信息
   */
  @Override
  public UserVo findUserAllMsgByUserName(String userName) {
    if (StringUtils.isBlank(userName)) {
      return null;
    }
    return userRepository.findUserAllMsgByUserName(userName);
  }


  /**
   * 批量校验数据
   *
   * @param updateList
   * @author huojia
   * @date 2022/12/24 17:40
   **/
  private void updateValidationBatch(List<UserDto> updateList) {
    List<String> userNameList = updateList.stream().map(UserDto::getUserName).filter(Objects::nonNull).collect(Collectors.toList());
    List<UserEntity> byUserNames = this.userRepository.findByUserNames(TenantUtils.getTenantCode(), userNameList);
    Map<String, UserEntity> userEntityMap = byUserNames.stream().collect(Collectors.toMap(UserEntity::getUserName, Function.identity()));
    updateList.forEach(update -> {
      Validate.notBlank(update.getUserPassword(), "密码不能为空");
      Validate.notBlank(update.getId(), "ID不能为空");
      Validate.notBlank(update.getUserCode(), "用户编码不能为空");
      Validate.isTrue(CollectionUtils.isNotEmpty(update.getPositionList()), "关联的职位不能为空!");
      UserEntity userEntity = userEntityMap.get(update.getUserName());
      Validate.notNull(userEntity, "用户不存在或已经删除");
      for (UserPositionDto item : update.getPositionList()) {
        //item.setId(null);
        Validate.notBlank(item.getOrgCode(), "组织不能为空");
        Validate.notNull(item.getOperationType(), "是否新增职位不能为空");
        Validate.notNull(item.getPrimaryFlag(), "是否主职位不能为空");
        if (item.getOperationType()) {
          // Validate.notBlank(item.getPositionLevelCode(), "职位级别不能为空");
          // item.setPositionCode(null);
        } else {
          Validate.notBlank(item.getPositionCode(), "职位不能为空");
          //item.setPositionLevelCode("");
          if (StringUtils.isNotBlank(item.getParentCode())) {
            Validate.isTrue(!item.getParentCode().equals(item.getPositionCode()), "上级职位不能是自己");
          }
        }
      }
    });
  }

  /**
   * 批量绑定职位
   *
   * @param saveList
   * @author huojia
   * @date 2022/12/24 16:22
   **/
  @Transactional(rollbackFor = Exception.class)
  void rebindUserPositionRelationBatch(List<UserDto> saveList, List<UserEntity> userEntities) {
    List<PositionDto> positionDtoList = new ArrayList<>();
    List<UserPositionDto> userPositionDtoList = new ArrayList<>();
    saveList.forEach(save -> {
      userPositionDtoList.add(save.getPositionList().get(0));
      for (UserPositionDto item : save.getPositionList()) {
        //获取组织
        PositionDto positionDto = this.nebulaToolkitService.copyObjectByWhiteList(item, PositionDto.class, HashSet.class, ArrayList.class);
        // mdg的职位组织关系已经确定，无需再维护
        List<PositionRelationDto> positionRelationDtos = new ArrayList<>();
        PositionRelationDto positionRelationDto = new PositionRelationDto();
        positionRelationDto.setRelationKey(OrgCodeConstant.KEY);
        positionRelationDto.setRelationName(OrgCodeConstant.NAME);
        List<RelationDataDto> relationDataDtos = new ArrayList<>();
        RelationDataDto relationDataDto = new RelationDataDto();
        relationDataDto.setCode(item.getOrgCode());
        relationDataDtos.add(relationDataDto);
        positionRelationDto.setRelationData(relationDataDtos);
        positionRelationDtos.add(positionRelationDto);
        positionDto.setRelationData(positionRelationDtos);
        positionDtoList.add(positionDto);
      }
    });
    // 更新用户
    this.userRepository.saveOrUpdateBatch(userEntities);
    // 更新职位上下级
    this.positionVoService.updateMdgBatch(positionDtoList);
    // 重新绑定MDG用户职位关系
    this.userPositionVoService.rebindMdgUser(userPositionDtoList);
  }

  /**
   * 批量校验数据
   *
   * @param saveList
   * @author huojia
   * @date 2022/12/24 15:59
   **/
  private void createValidationBatch(List<UserDto> saveList) {
    List<String> userNameList = saveList.stream().map(UserDto::getUserName).filter(Objects::nonNull).collect(Collectors.toList());
    List<UserEntity> byUserNames = this.userRepository.findByUserNames(TenantUtils.getTenantCode(), userNameList);
    Map<String, UserEntity> userEntityMap = byUserNames.stream().collect(Collectors.toMap(UserEntity::getUserName, Function.identity()));
    List<String> codeList = this.generateCodeService.generateCode(UserConstant.USER_CODE, saveList.size());
    ListIterator<String> codeIterator = codeList.listIterator();
    saveList.forEach(save -> {
      save.setId(null);
      if (userEntityMap.containsKey(save.getUserName())) {
        throw new RuntimeException("登录账号已存在（或已逻辑删除）");
      }
      Validate.isTrue(CollectionUtils.isNotEmpty(save.getPositionList()), "关联的职位不能为空!");
      if (BooleanEnum.TRUE.getCapital().equals(save.getMd5Flag())) {
        String password = Aes128Utils.decrypt(save.getUserPassword(), UserConstant.ENCRYPT_KEY, Aes128Utils.EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
        String md5Password = DigestUtils.md5DigestAsHex(password.getBytes(StandardCharsets.UTF_8));
        save.setUserPassword(md5Password);
      }
      save.setUserCode(codeIterator.next());
      save.setTenantCode(TenantUtils.getTenantCode());
      save.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
      save.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
      save.setLockState(EnableStatusEnum.ENABLE.getCode());
      save.setForceChangePassword(Boolean.TRUE);
      save.setUpdatePasswordDate(new Date());
      for (UserPositionDto item : save.getPositionList()) {
        //item.setId(null);
        // Validate.notBlank(item.getOrgCode(), "组织不能为空");
        Validate.notNull(item.getOperationType(), "是否新增职位不能为空");
        Validate.notNull(item.getPrimaryFlag(), "是否主职位不能为空");
        Validate.notBlank(item.getPositionCode(), "职位不能为空");
        if (StringUtils.isNotBlank(item.getParentCode())) {
          Validate.isTrue(!item.getParentCode().equals(item.getPositionCode()), "上级职位不能是自己");
        }
      }
    });
  }

  @Override
  @Transactional
  public void update(UserDto dto) {
    this.updateValidation(dto);
    this.validationPosition(dto);
    //重构查询方法
    UserEntity oldEntity = this.userRepository.findByIdAndTenantCode(dto.getId(), TenantUtils.getTenantCode());
    Validate.notNull(oldEntity, "修改信息不存在");
    UserEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, UserEntity.class, HashSet.class, ArrayList.class);
    //新增租户编号
    entity.setTenantCode(TenantUtils.getTenantCode());
    this.userRepository.saveOrUpdate(entity);
    // 更新用户-职位关联关系
    this.rebindUserPositionRelation(dto);
    // 发送修改通知
    if (CollectionUtils.isNotEmpty(userEventListeners)) {
      UserVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(oldEntity, UserVo.class, HashSet.class, ArrayList.class);
      UserVo materialVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, UserVo.class, HashSet.class, ArrayList.class);
      this.userEventListeners.forEach(event -> event.onUpdate(oldVo, materialVo));
    }
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<UserEntity> entities = this.userRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.userRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    // TODO 指定用户全平台退出登录
    // 发送启用通知
    if (CollectionUtils.isNotEmpty(userEventListeners)) {
      List<UserVo> voList = (List<UserVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, UserEntity.class, UserVo.class, HashSet.class, ArrayList.class);
      this.userEventListeners.forEach(event -> event.onEnable(voList));
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<UserEntity> entities = this.userRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.userRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    // 发送禁用通知
    if (CollectionUtils.isNotEmpty(userEventListeners)) {
      List<UserVo> voList = (List<UserVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, UserEntity.class, UserVo.class, HashSet.class, ArrayList.class);
      this.userEventListeners.forEach(event -> event.onDisable(voList));
    }
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<UserEntity> entities = this.userRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.userRepository.updateDelFlagByIds(ids);
    List<String> userNameList =
            entities.stream()
                    .filter(a -> StringUtils.isNotBlank(a.getUserName()))
                    .map(UserEntity::getUserName)
                    .collect(Collectors.toList());
    if (CollectionUtils.isNotEmpty(userNameList)) {
      this.userPositionVoService.deleteByUserNames(userNameList);
    }
    // 发送删除通知
    if (CollectionUtils.isNotEmpty(userEventListeners)) {
      List<UserVo> voList = (List<UserVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, UserEntity.class, UserVo.class, HashSet.class, ArrayList.class);
      this.userEventListeners.forEach(event -> event.onDelete(voList));
    }
  }

  /**
   * 更新用户-职位关联关系
   * 1、根据用户账号移除原用户与职位关联信息
   * 2、职位信息检查
   * （1）检查职位信息关联的组织是否存在
   * （2）新增职位先创建记录，已有职位校验是否存在
   * 3、重新创建用户与职位关联关系
   *
   * @param dto 用户对象
   */
  public void rebindUserPositionRelation(UserDto dto) {
    if (CollectionUtils.isEmpty(dto.getPositionList())) {
      this.userPositionVoService.deleteByUserNames(Collections.singletonList(dto.getUserName()));
      return;
    }
    for (UserPositionDto item : dto.getPositionList()) {
      //获取组织
      OrgVo org = orgVoService.findByOrgCode(item.getOrgCode());
      Validate.notNull(org, "组织不存在");
      PositionDto positionDto = this.nebulaToolkitService.copyObjectByWhiteList(item, PositionDto.class, HashSet.class, ArrayList.class);
      positionDto.setRoleCodeList(item.getRoleCodeList());
      // 维护职位与组织关系
      List<PositionRelationDto> positionRelationDtos = new ArrayList<>();
      PositionRelationDto positionRelationDto = new PositionRelationDto();
      positionRelationDto.setRelationKey(OrgCodeConstant.KEY);
      positionRelationDto.setRelationName(OrgCodeConstant.NAME);
      List<RelationDataDto> relationDataDtos = new ArrayList<>();
      RelationDataDto relationDataDto = new RelationDataDto();
      relationDataDto.setCode(org.getOrgCode());
      relationDataDtos.add(relationDataDto);
      positionRelationDto.setRelationData(relationDataDtos);
      positionRelationDtos.add(positionRelationDto);
      positionDto.setRelationData(positionRelationDtos);
      if (item.getOperationType()) {
        //新增职位关系
//        positionDto.setPositionName(null);
        PositionVo positionVo = positionVoService.create(positionDto);
        item.setPositionCode(positionVo.getPositionCode());
      } else {
        //已有职位关系
        Validate.notBlank(item.getPositionCode(), "职位编码不能为空");
        PositionVo positionVo = positionVoService.findByPositionCode(item.getPositionCode());
        Validate.notNull(positionVo, "职位不存在");
        positionDto.setId(positionVo.getId());
        item.setMdgPositionCode(positionVo.getMdgPositionCode());
        positionVoService.update(positionDto);
      }
    }
    List<String> primaryPositionCodeList = dto.getPositionList().stream().filter(i -> Boolean.TRUE.equals(i.getPrimaryFlag())).map(UserPositionDto::getPositionCode).collect(Collectors.toList());
    Validate.isTrue(primaryPositionCodeList.size() <= 1, "一个用户有且只有一个主职位");
    String primaryPositionCode = null;
    if (CollectionUtils.isNotEmpty(primaryPositionCodeList)) {
      primaryPositionCode = primaryPositionCodeList.get(0);
    }
    List<String> positionCodeList = dto.getPositionList().stream().map(UserPositionDto::getPositionCode).distinct().collect(Collectors.toList());
    this.userPositionVoService.rebindByUserName(dto.getUserName(), primaryPositionCode, positionCodeList, dto.getPositionList());
  }

  /**
   * 新增校验差异部分
   *
   * @param dto
   */
  private void createValidation(UserDto dto) {
    dto.setId(null);
    this.validation(dto);
    if (CollectionUtils.isNotEmpty(dto.getSysLoginPorts())) {
      dto.setSysLoginPort(Strings.join(dto.getSysLoginPorts(), ","));
    }
    UserEntity byUserNameCoverDel = this.userRepository.findByUserNameCoverDel(TenantUtils.getTenantCode(), dto.getUserName());
    Validate.isTrue(Objects.isNull(byUserNameCoverDel), "登录账号已存在（或已逻辑删除）");
    Validate.notBlank(dto.getUserPassword(), "用户密码不能为空！");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getPositionList()), "关联的职位不能为空!");
    log.info("HttpServletRequest userPassword:{}", dto.getUserPassword());
    //对密码进行 md5 加密
    String password = Aes128Utils.decrypt(dto.getUserPassword(), UserConstant.ENCRYPT_KEY, Aes128Utils.EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
    String md5Password = DigestUtils.md5DigestAsHex(password.getBytes(StandardCharsets.UTF_8));
    dto.setUserPassword(md5Password);
  }

  /**
   * 编辑校验差异部分
   *
   * @param dto
   */
  private void updateValidation(UserDto dto) {
    this.validation(dto);
    Validate.notBlank(dto.getUserPassword(), "密码不能为空");
    Validate.notBlank(dto.getId(), "ID不能为空");
    Validate.notBlank(dto.getUserCode(), "用户编码不能为空");
    UserEntity dbEntity = this.userRepository.findByIdAndTenantCode(dto.getId(), TenantUtils.getTenantCode());
    Validate.notNull(dbEntity, "不存在或已经删除");
    Validate.isTrue(dbEntity.getUserName().equals(dto.getUserName()), "用户账号不能修改");
    Validate.isTrue(CollectionUtils.isNotEmpty(dto.getPositionList()), "关联的职位不能为空!");
    if (StringUtils.isNotBlank(dto.getUserPassword())) {
      log.info("HttpServletRequest userPassword:{}", dto.getUserPassword());
      //对密码进解密
      String password = Aes128Utils.decrypt(dto.getUserPassword(), UserConstant.ENCRYPT_KEY, Aes128Utils.EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
      this.passwordValidateService.validate(password);
      if (!password.equals(dbEntity.getUserPassword())) {
        // 如果不是原来的密码需要md5加密
        String md5Password = DigestUtils.md5DigestAsHex(password.getBytes(StandardCharsets.UTF_8));
        dto.setUserPassword(md5Password);
        // 密码发生变化，记录更改时间 updatePasswordDate
        dto.setUpdatePasswordDate(new Date());
      } else {
        // 否则还是原来的密码
        dto.setUserPassword(dbEntity.getUserPassword());
      }
    } else {
      // 否则还是原来的密码
      dto.setUserPassword(dbEntity.getUserPassword());
    }
  }

  /**
   * 新增编辑校验公共部分
   * 1、必填参数校验
   * 2、校验手机号、邮箱、身份证号是否占用
   * 3、职位列表校验
   *
   * @param dto
   */
  private void validation(UserDto dto) {
    Validate.notNull(dto, "参数对象不能为空");
    Validate.notBlank(dto.getUserName(), "用户帐号不能为空");
    Validate.notBlank(dto.getFullName(), "用户姓名不能为空");
    Validate.notBlank(dto.getUserPhone(), "电话号码不能为空");
    List<UserEntity> UserByPhone = this.userRepository.findByUserPhone(TenantUtils.getTenantCode(), dto.getUserPhone());
    if (StringUtils.isNotBlank(dto.getId())) {
      //更新操作下校验电话号码
      List<UserEntity> list = UserByPhone.stream().filter(x -> StringUtils.isNotBlank(dto.getId()) && !dto.getId().equals(x.getId()))
              .collect(Collectors.toList());
      Validate.isTrue(CollectionUtils.isEmpty(list), "电话号码已经被占用");
    } else {
      //创建操作下校验电话号码
      Validate.isTrue(CollectionUtils.isEmpty(UserByPhone), "电话号码已经被占用");
    }
    if (StringUtils.isNotBlank(dto.getEmail())) {
      List<UserEntity> list = this.userRepository.findByEmail(TenantUtils.getTenantCode(), dto.getEmail())
              .stream().filter(x -> StringUtils.isNotBlank(dto.getId()) && !dto.getId().equals(x.getId()))
              .collect(Collectors.toList());
      Validate.isTrue(CollectionUtils.isEmpty(list), "邮箱已经被占用");
    }
    if (StringUtils.isNotBlank(dto.getIdentityCardNumber())) {
      List<UserEntity> list = this.userRepository.findByIdentityCardNumber(TenantUtils.getTenantCode(), dto.getIdentityCardNumber())
              .stream().filter(x -> StringUtils.isNotBlank(dto.getId()) && !dto.getId().equals(x.getId()))
              .collect(Collectors.toList());
      Validate.isTrue(CollectionUtils.isEmpty(list), "身份证号码已经存在");
    }
    if (CollectionUtils.isEmpty(dto.getPositionList())) {
      return;
    }
    List<String> primaryPositionCodeList = dto.getPositionList().stream().filter(i -> Boolean.TRUE.equals(i.getPrimaryFlag())).map(UserPositionDto::getPositionCode).collect(Collectors.toList());
    Validate.isTrue(primaryPositionCodeList.size() <= 1, "一个用户有且只有一个主职位");
    for (UserPositionDto item : dto.getPositionList()) {
      //item.setId(null);
      Validate.notBlank(item.getOrgCode(), "组织不能为空");
      Validate.notNull(item.getOperationType(), "是否新增职位不能为空");
      Validate.notNull(item.getPrimaryFlag(), "是否主职位不能为空");
      if (item.getOperationType()) {
        // Validate.notBlank(item.getPositionLevelCode(), "职位级别不能为空");
        // TODO：前端传错了，先将就
        item.setPositionName(item.getPositionCode());
        item.setPositionCode(null);
      } else {
        Validate.notBlank(item.getPositionCode(), "职位不能为空");
        //item.setPositionLevelCode("");
        if (StringUtils.isNotBlank(item.getParentCode())) {
          Validate.isTrue(!item.getParentCode().equals(item.getPositionCode()), "上级职位不能是自己");
        }
      }
    }
  }

  @Override
  @Transactional
  public void deleteUserLockByIds(List<String> ids) {
    Validate.notEmpty(ids, "ID集合不能为空");
    List<UserEntity> list = this.userRepository.findByIds(ids);
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    String tenantCode = TenantUtils.getTenantCode();
    /**
     * 登录方式
     * {@link com.bizunited.nebula.security.sdk.loginform.LoginDetails}
     */
    list.forEach(item -> {
      for (int i = 1; i < 7; i++) {
        String account = StringUtils.join(i + "", "_", item.getUserName());
        String redisKey = String.format(UserConstant.LOGIN_FAILED_KEY, tenantCode, account);
        this.stringRedisTemplate.delete(redisKey);
      }
    });
  }

  @Override
  @Transactional
  public void updatePasswordByIds(UserForceChangePasswordDto dto) {
    Validate.notEmpty(dto.getIds(), "ID集合不能为空");
    Validate.notBlank(dto.getPassword(), "缺失密码");
    log.info("HttpServletRequest password:{}", dto.getPassword());
    List<UserEntity> entities = this.userRepository.findByIds(dto.getIds());
    Validate.notEmpty(entities, "不存在或已删除");
    Set<String> idSet = entities.stream().map(UserEntity::getId).collect(Collectors.toSet());
    dto.getIds().forEach(id -> Validate.isTrue(idSet.contains(id), String.format("不存在或已删除ID:%s", id)));
    String password = Aes128Utils.decrypt(dto.getPassword(), UserConstant.ENCRYPT_KEY, Aes128Utils.EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
    String passwordEncryption = DigestUtils.md5DigestAsHex(password.getBytes(StandardCharsets.UTF_8));
    this.userRepository.updatePasswordByIds(dto.getIds(), passwordEncryption);
    // TODO 更新缓存
  }

  @Override
  @Transactional
  public void updatePasswordByUserName(UserChangePasswordDto dto) {
    Validate.notBlank(dto.getUserName(), "缺失用户名");
    Validate.notBlank(dto.getOldPassword(), "缺失原密码");
    Validate.notBlank(dto.getNewPassword(), "缺失新密码");
    log.info("HttpServletRequest oldPassword:{}", dto.getOldPassword());
    log.info("HttpServletRequest newPassword:{}", dto.getNewPassword());
    // 校验修改密码强度
    this.isStrongPwd(dto.getNewPassword());
    UserEntity entity = this.userRepository.findByUserName(TenantUtils.getTenantCode(), dto.getUserName());
    Validate.notNull(entity, "用户不存在或已删除");
    String oldPassword = Aes128Utils.decrypt(dto.getOldPassword(), UserConstant.ENCRYPT_KEY, Aes128Utils.EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
    String newPassword = Aes128Utils.decrypt(dto.getNewPassword(), UserConstant.ENCRYPT_KEY, Aes128Utils.EncodeType.CBC, Aes128Utils.Padding.PKCS_7_PADDING);
    String oldPasswordEncryption = DigestUtils.md5DigestAsHex(oldPassword.getBytes(StandardCharsets.UTF_8));
    String newPasswordEncryption = DigestUtils.md5DigestAsHex(newPassword.getBytes(StandardCharsets.UTF_8));
    Validate.isTrue(entity.getUserPassword().equals(oldPasswordEncryption), "原密码输入错误");
    Validate.isTrue(!oldPasswordEncryption.equals(newPasswordEncryption), "新密码不能与原密码相同");
    entity.setUserPassword(newPasswordEncryption);
    entity.setUpdatePasswordDate(new Date());
    entity.setForceChangePassword(Boolean.FALSE);
    //重构修改方法
    this.userRepository.updateByIdAndTenantCode(entity, TenantUtils.getTenantCode());
  }

  @Override
  public Set<String> findUserNamesByCodesAndUserType(List<String> orgCodes, List<String> positionLevelCodes, String userType) {
    if (CollectionUtils.isEmpty(orgCodes) && CollectionUtils.isEmpty(positionLevelCodes)) {
      return new HashSet<>();
    }
    Set<String> positionCodes = new HashSet<>();
    //1、根据组织查询职位编码集合
    Set<String> positionCodesByOrgCodes = this.orgPositionVoService.findPositionCodesByOrgCodes(orgCodes);
    //2、通过职位级别查询职位编码集合
    Set<String> positionCodesByPositionLevelCodes = this.positionVoService.findPositionCodesByPositionLevelCodes(positionLevelCodes);
    //如果1和2查询的信息都不为空则取交集，如果1不为空且2为空则取1作为职位结果，如果1为空且2不为空则取2作为职位结果，其他情况直接结束后续
    if (!CollectionUtils.isEmpty(positionCodesByOrgCodes) && !CollectionUtils.isEmpty(positionCodesByPositionLevelCodes)) {
      positionCodesByOrgCodes.retainAll(positionCodesByOrgCodes);
      positionCodes = positionCodesByOrgCodes;
    } else if (!CollectionUtils.isEmpty(positionCodesByOrgCodes) && CollectionUtils.isEmpty(positionCodesByPositionLevelCodes)) {
      positionCodes = positionCodesByOrgCodes;
    } else if (CollectionUtils.isEmpty(positionCodesByOrgCodes) && !CollectionUtils.isEmpty(positionCodesByPositionLevelCodes)) {
      positionCodes = positionCodesByPositionLevelCodes;
    } else {
      return new HashSet<>();
    }
    List<UserPositionEntity> byPositionCode = this.userPositionRepository.findByPositionCodeIn(TenantUtils.getTenantCode(), positionCodes);
    if (CollectionUtils.isEmpty(byPositionCode)) {
      return new HashSet<>();
    }
    Set<String> userNames = byPositionCode.stream().map(UserPositionEntity::getUserName).collect(Collectors.toSet());
    List<UserEntity> byUserNamesAndUserType = this.userRepository.findByUserNamesAndUserType(TenantUtils.getTenantCode(), userNames, userType);
    if (CollectionUtils.isEmpty(byUserNamesAndUserType)) {
      return new HashSet<>();
    }
    return byUserNamesAndUserType.stream().map(UserEntity::getUserName).collect(Collectors.toSet());
  }

  @Override
  public UserVo findDetailsByPhone(String phone) {
    if (StringUtils.isBlank(phone)) {
      return null;
    }
    List<UserEntity> userEntities = this.userRepository.findByUserPhone(TenantUtils.getTenantCode(), phone);
    if (CollectionUtils.isEmpty(userEntities)) {
      return null;
    }
    UserVo userVo = this.nebulaToolkitService.copyObjectByWhiteList(userEntities.get(0), UserVo.class, HashSet.class, ArrayList.class);
    List<UserPositionEntity> userPositions = this.userPositionRepository.findByUserName(TenantUtils.getTenantCode(), userVo.getUserName());
    if (CollectionUtils.isEmpty(userPositions)) {
      return userVo;
    }
    //组装职位关联的组织数据
    List<UserPositionVo> userPositionVos = (List<UserPositionVo>) this.nebulaToolkitService.copyCollectionByWhiteList(userPositions, UserPositionEntity.class, UserPositionVo.class, HashSet.class, ArrayList.class);
    if (!CollectionUtils.isEmpty(userPositions)) {
      List<String> positionCodes = userPositions.stream().map(UserPositionEntity::getPositionCode).collect(Collectors.toList());
      List<OrgPositionVo> orgPositionVos = this.orgPositionVoService.findByPositionCodes(positionCodes);
      //组装职位关联的组织数据
      if (!CollectionUtils.isEmpty(orgPositionVos)) {
        Map<String, List<OrgPositionVo>> orgPositionVoMap = orgPositionVos.stream().collect(Collectors.groupingBy(OrgPositionVo::getPositionCode));
        for (UserPositionVo userPositionVo : userPositionVos) {
          List<OrgPositionVo> orgPositionVoList = orgPositionVoMap.get(userPositionVo.getPositionCode());
          if (CollectionUtils.isEmpty(orgPositionVoList)) {
            continue;
          }
          Set<String> orgCodes = orgPositionVoList.stream().map(OrgPositionVo::getOrgCode).collect(Collectors.toSet());
          userPositionVo.setOrgCodes(orgCodes);
        }
      }
    }
    //组装职位数据
    userVo.setPositionList(userPositionVos);
    return userVo;
  }

  @Override
  public UserVo findByPhone(String phone) {
    if (StringUtils.isBlank(phone)) {
      return null;
    }
    List<UserEntity> userEntities = this.userRepository.findByUserPhone(TenantUtils.getTenantCode(), phone);
    if (CollectionUtils.isEmpty(userEntities)) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(userEntities.get(0), UserVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public UserVo findRelationByUserNameAndOrgCodesOrOrgTypes(String userName, List<String> orgCodes, List<String> orgTypes) {
    if (StringUtils.isBlank(userName)) {
      return null;
    }
    UserEntity userEntity = this.userRepository.findByUserName(TenantUtils.getTenantCode(), userName);
    if (userEntity == null) {
      return null;
    }
    UserVo userVo = this.nebulaToolkitService.copyObjectByWhiteList(userEntity, UserVo.class, HashSet.class, ArrayList.class);
    List<UserPositionEntity> userPositionEntities = this.userPositionRepository.findByUserName(TenantUtils.getTenantCode(), userName);

    List<String> positionCodes = Optional.ofNullable(userPositionEntities).orElse(Lists.newArrayList()).stream().map(UserPositionEntity::getPositionCode).collect(Collectors.toList());
    List<OrgPositionVo> orgPositionVos = this.orgPositionVoService.findByPositionCodes(positionCodes);
    if (CollectionUtils.isEmpty(orgPositionVos)) {
      return userVo;
    }
    // 找到用户岗位关系间接关联的组织编码集合
    List<String> userRelationOrgCodes = orgPositionVos.stream().map(OrgPositionVo::getOrgCode).collect(Collectors.toList());
    List<OrgVo> userRelationOrgVos = this.orgVoService.findByOrgCodes(userRelationOrgCodes);
    if (CollectionUtils.isEmpty(userRelationOrgVos)) {
      // 说明用户岗位关系没有关联任何组织信息，直接返回用户基础信息
      return userVo;
    }

    if (CollectionUtils.isNotEmpty(orgCodes)) {
      List<OrgVo> allChildrenByOrgCodes = this.orgVoService.findAllChildrenByOrgCodes(orgCodes);
      List<String> allChildrenOrgCodes = allChildrenByOrgCodes
              .stream()
              .map(OrgVo::getOrgCode)
              .distinct()
              .collect(Collectors.toList());
      // 从orgCodes包含所有子集的组织类型集合中找到与用户关系组织重叠的部分
      List<String> relationOrgCodes = userRelationOrgVos
              .stream()
              .filter(op -> StringUtils.isNotBlank(op.getOrgCode()) && allChildrenOrgCodes.contains(op.getOrgCode()))
              .map(OrgVo::getOrgCode)
              .distinct()
              .collect(Collectors.toList());
      userVo.setRelationOrgCodes(relationOrgCodes);
    }

    if (CollectionUtils.isNotEmpty(orgTypes)) {
      List<OrgVo> allChildrenByOrgTypes = this.orgVoService.findAllChildrenByOrgTypes(orgTypes);
      if (CollectionUtils.isNotEmpty(allChildrenByOrgTypes)) {
        List<String> allChildrenOrgTypes = allChildrenByOrgTypes
                .stream()
                .map(OrgVo::getOrgType)
                .distinct()
                .collect(Collectors.toList());
        // 从orgTypes包含所有子集的组织类型集合中找到与用户关系组织重叠的部分
        List<String> relationOrgTypes = userRelationOrgVos
                .stream()
                .filter(op -> StringUtils.isNotBlank(op.getOrgType()) && allChildrenOrgTypes.contains(op.getOrgType()))
                .map(OrgVo::getOrgType)
                .distinct()
                .collect(Collectors.toList());
        userVo.setRelationOrgTypes(relationOrgTypes);
      }
    }

    return userVo;
  }

  @Override
  public Set<String> findUserNamesByUserConditionDto(UserConditionDto dto) {
    if (Objects.isNull(dto)) {
      return Sets.newLinkedHashSet();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    return this.userRepository.findUserNamesByUserConditionDto(dto);
  }

  @Override
  public Set<String> findPositionCodesByUserNames(List<String> userNames) {
    if (CollectionUtils.isEmpty(userNames)) {
      return new HashSet<>(0);
    }
    List<UserPositionEntity> byUserNames = this.userPositionRepository.findByUserNames(TenantUtils.getTenantCode(), userNames);
    if (CollectionUtils.isEmpty(byUserNames)) {
      return new HashSet<>(0);
    }
    //默认获取主职位的角色
    return byUserNames.stream().filter(UserPositionEntity::getPrimaryFlag).map(UserPositionEntity::getPositionCode).collect(Collectors.toSet());
  }


  /**
   * 根据账号集合查询账号信息
   *
   * @param userNameList 账号集合
   * @return java.util.Set<java.lang.String>
   * @author: huxmld
   * @version: v1.0.0
   * @date: 2022.12.3 23:00
   */
  @Override
  public Set<String> findUserNamesByUserNames(List<String> userNameList) {
    if (CollectionUtils.isEmpty(userNameList)) {
      return new HashSet<>(0);
    }
    List<UserEntity> byUserNames = this.userRepository.findByUserNames(TenantUtils.getTenantCode(), userNameList);
    if (CollectionUtils.isEmpty(byUserNames)) {
      return new HashSet<>(0);
    }
    return byUserNames.stream().map(UserEntity::getUserName)
            .collect(Collectors.toSet());
  }

  /**
   * 先判断传入密码的长度是否够12位，其次判断是否包含各个类型的符号.
   * 密码要求: 大小写字母特殊字符以及数字, 且长度大于12位.
   *
   * @param pwd
   * @return
   */
  private void isStrongPwd(String pwd) {
    // 密码长度必须大于等于8位数
    final Integer MIN_LENGTH = 8;
    // 小写字母表
    final String SMALL_LETTERS = "abcdefghijklmnopqrstuvwxyz";
    // 大写字母表
    final String BIG_LETTERS = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    // 特殊字符
    final String SPECIAL_CHARS = "~!@#$%^&*()_+-={}[],./<>?:|";
    // 数字
    final String NUMBERS = "1234567890";
    //检查长度是否足够
    boolean isLongEnough = pwd.length() >= MIN_LENGTH;
    Validate.isTrue(isLongEnough, "密码长度必须大于等于8位数");
    //长度够了
    boolean hasBigLetters = false;
    boolean hasSmallLetters = false;
    boolean hasNumbers = false;
    boolean hasSpecialChars = false;
    int len = pwd.length();
    for (int i = 0; i < len; i++) {
      if (!hasSmallLetters && SMALL_LETTERS.indexOf(pwd.charAt(i)) > -1) {
        hasSmallLetters = true;
      }
      if (!hasBigLetters && BIG_LETTERS.indexOf(pwd.charAt(i)) > -1) {
        hasBigLetters = true;
      }
      if (!hasNumbers && NUMBERS.indexOf(pwd.charAt(i)) > -1) {
        hasNumbers = true;
      }
      if (!hasSpecialChars && SPECIAL_CHARS.indexOf(pwd.charAt(i)) > -1) {
        hasSpecialChars = true;
      }
    }
    Validate.isTrue(hasBigLetters, "密码里面没有大写字母！");
    Validate.isTrue(hasSmallLetters, "密码里面没有小写字母！");
    Validate.isTrue(hasNumbers, "密码里面没有数字！");
    Validate.isTrue(hasSpecialChars, "密码里面没有特殊字符！");
  }

  /**
   * 校验职位是否存在主主职位
   *
   * @param dto
   */
  private void validationPosition(UserDto dto) {
    Validate.notNull(dto, "参数不能为空！");
    if (CollectionUtils.isEmpty(dto.getPositionList())) {
      return;
    }
    List<UserPositionDto> flag = dto.getPositionList().stream().filter(o -> o.getPrimaryFlag() == true).collect(Collectors.toList());
    Validate.isTrue(CollectionUtils.isNotEmpty(flag), "必须有一个主职位!");
  }

  @Override
  public List<UserVo> findByUserCodes(Set<String> userCodes) {
    if (CollectionUtils.isEmpty(userCodes)) {
      return new ArrayList<>(0);
    }
    List<UserEntity> userEntities = this.userRepository.findByUserCodes(TenantUtils.getTenantCode(), userCodes);
    if (CollectionUtils.isEmpty(userEntities)) {
      return null;
    }
    return (List<UserVo>) this.nebulaToolkitService.copyCollectionByWhiteList(userEntities, UserEntity.class, UserVo.class, HashSet.class, ArrayList.class);

  }
}
