package com.biz.crm.mdm.business.user.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.LoginFromTypeEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.mdm.business.user.local.entity.UserRelWeChatEntity;
import com.biz.crm.mdm.business.user.local.repository.UserRelWeChatRepository;
import com.biz.crm.mdm.business.user.sdk.dto.UserRelWeChatDto;
import com.biz.crm.mdm.business.user.sdk.dto.UserRelWeChatPageDto;
import com.biz.crm.mdm.business.user.sdk.dto.UserWeChatBindDto;
import com.biz.crm.mdm.business.user.sdk.enums.UserTypeEnum;
import com.biz.crm.mdm.business.user.sdk.event.UserRelWeChatEventListener;
import com.biz.crm.mdm.business.user.sdk.service.UserRelWeChatVoService;
import com.biz.crm.mdm.business.user.sdk.service.UserVoService;
import com.biz.crm.mdm.business.user.sdk.vo.UserRelWeChatVo;
import com.biz.crm.mdm.business.user.sdk.vo.UserVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 用户与微信关联表(UserRelWeChat)表服务实现类
 *
 * @author xi.peng
 * @since 2021-11-05 15:05:07
 */
@Service
public class UserRelWeChatVoServiceImpl implements UserRelWeChatVoService {

  @Autowired(required = false)
  private UserRelWeChatRepository userRelWeChatRepository;

  @Autowired(required = false)
  @Lazy
  private List<UserRelWeChatEventListener> userRelWeChatEventListeners;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private LoginUserService loginUserService;
  @Autowired(required = false)
  private UserVoService userVoService;

  @Override
  public Page<UserRelWeChatVo> findByConditions(Pageable pageable, UserRelWeChatPageDto dto) {

    return this.userRelWeChatRepository.findByConditions(pageable, dto);
  }

  @Override
  public UserRelWeChatVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    UserRelWeChatEntity entity = this.userRelWeChatRepository.findById(id);
    if (entity == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, UserRelWeChatVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  @Transactional
  public void create(UserRelWeChatDto dto) {
    Validate.notNull(dto, "请求对象不能为空");
    dto.setTenantCode(TenantUtils.getTenantCode());
    UserRelWeChatEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, UserRelWeChatEntity.class, HashSet.class, ArrayList.class);
    this.userRelWeChatRepository.save(entity);
    // 发送通知
    if (CollectionUtils.isNotEmpty(userRelWeChatEventListeners)) {
      UserRelWeChatVo vo = this.nebulaToolkitService.copyObjectByWhiteList(entity, UserRelWeChatVo.class, HashSet.class, ArrayList.class);
      this.userRelWeChatEventListeners.forEach(event -> event.onCreate(vo));
    }
  }

  @Override
  @Transactional
  public void update(UserRelWeChatDto dto) {
    Validate.notNull(dto, "请求对象不能为空");
    Validate.notBlank(dto.getId(), "ID不能为空");
    UserRelWeChatEntity oldEntity = this.userRelWeChatRepository.findById(dto.getId());
    Validate.notNull(oldEntity, "修改信息不存在");
    UserRelWeChatEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, UserRelWeChatEntity.class, HashSet.class, ArrayList.class);
    this.userRelWeChatRepository.updateById(entity);
    // 发送修改通知
    if (CollectionUtils.isNotEmpty(userRelWeChatEventListeners)) {
      UserRelWeChatVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(oldEntity, UserRelWeChatVo.class, HashSet.class, ArrayList.class);
      UserRelWeChatVo newVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, UserRelWeChatVo.class, HashSet.class, ArrayList.class);
      this.userRelWeChatEventListeners.forEach(event -> event.onUpdate(oldVo, newVo));
    }
  }

  @Override
  @Transactional
  public void deleteByUserNames(List<String> userNames) {
    if (CollectionUtils.isEmpty(userNames)) {
      return;
    }
    this.userRelWeChatRepository.deleteByUserNames(TenantUtils.getTenantCode(), userNames);
    //TODO 退出登录APPLET_SFA & APPLET_DMS，此逻辑需结合登录机制完成闭环
  }

  @Override
  public List<UserRelWeChatVo> findByUserRelWeChatDto(UserRelWeChatDto dto) {
    List<UserRelWeChatEntity> entities = this.userRelWeChatRepository.findByUserRelWeChatDto(TenantUtils.getTenantCode(), dto);
    if (CollectionUtils.isEmpty(entities)) {
      return new ArrayList<>();
    }
    return (List<UserRelWeChatVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, UserRelWeChatEntity.class, UserRelWeChatVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public void createByUserWeChatBindDto(UserWeChatBindDto dto) {
    this.createValidation(dto);
    UserWeChatBindDto queryDto = new UserWeChatBindDto();
    queryDto.setOriginModule(dto.getOriginModule());
    queryDto.setOpenId(dto.getOpenId());
    List<UserRelWeChatEntity> collect = this.userRelWeChatRepository.findByUserWeChatBindDto(queryDto);
    if (CollectionUtils.isNotEmpty(collect)) {
      collect = collect.stream().filter(x -> !dto.getUserName().equals(x.getUserName()))
          .collect(Collectors.toList());
    }
    if (CollectionUtils.isNotEmpty(collect)) {
      this.userRelWeChatRepository.removeByIds(collect.stream().map(UserRelWeChatEntity::getId).collect(Collectors.toList()));
    }
    List<UserRelWeChatEntity> existCheck = this.userRelWeChatRepository.findByUserWeChatBindDto(dto);
    if (CollectionUtils.isEmpty(existCheck)) {
      UserRelWeChatEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, UserRelWeChatEntity.class, HashSet.class, ArrayList.class);
      entity.setLoginStatus(Boolean.TRUE);
      this.userRelWeChatRepository.save(entity);
    } else {
      UserRelWeChatEntity entity = existCheck.get(0);
      entity.setLoginStatus(Boolean.TRUE);
      this.userRelWeChatRepository.updateById(existCheck.get(0));
    }
  }

  @Override
  public List<UserRelWeChatVo> findByOpenid(String openid) {
    if (StringUtils.isBlank(openid)) {
      return new ArrayList<>(0);
    }
    List<UserRelWeChatEntity> entities = this.userRelWeChatRepository.findByOpenid(TenantUtils.getTenantCode(), openid);
    if (CollectionUtils.isEmpty(entities)) {
      return new ArrayList<>();
    }
    return (List<UserRelWeChatVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, UserRelWeChatEntity.class, UserRelWeChatVo.class, HashSet.class, ArrayList.class);

  }

  @Override
  public List<UserRelWeChatVo> findByUserName(String userName) {
    if (StringUtils.isBlank(userName)) {
      return new ArrayList<>(0);
    }
    List<UserRelWeChatEntity> entities = this.userRelWeChatRepository.findByUserName(TenantUtils.getTenantCode(), userName);
    if (CollectionUtils.isEmpty(entities)) {
      return new ArrayList<>();
    }
    return (List<UserRelWeChatVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, UserRelWeChatEntity.class, UserRelWeChatVo.class, HashSet.class, ArrayList.class);

  }

  /**
   * 在创建UserRelWeChat模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(UserWeChatBindDto dto) {

    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setId(null);
    Validate.notNull(dto, "请求对象不能为空");
    Validate.notBlank(dto.getOpenId(), "缺失openId");
    Validate.notBlank(dto.getOriginModule(), "缺失关联小程序模块");
    AbstractCrmUserIdentity loginDetails = this.loginUserService.getAbstractLoginUser();
    UserVo userVo = this.userVoService.findByUserName(loginDetails.getAccount());
    Validate.notNull(userVo, "用户不存在");
    dto.setUserName(userVo.getUserName());

    //TODO 此处展示设置为需要唯一检测,crm2.0是需要获取配置来设置此值的
    Boolean uniqueCheck = Boolean.TRUE;
    if (LoginFromTypeEnum.SFA_MINI.getValue().equals(dto.getOriginModule())) {
      Validate.isTrue(UserTypeEnum.USER.getCode().equals(userVo.getUserType()), "非企业用户不能绑定SFA小程序");
    } else if (LoginFromTypeEnum.DMS_MINI.getValue().equals(dto.getOriginModule())) {
      Validate.isTrue(UserTypeEnum.CUSTOMER.getCode().equals(userVo.getUserType()), "非客户用户不能绑定DMS小程序");
      //TODO 截止2022/05/16，系统还没有终端小程序，如果以后增加请再维护好对应枚举值LoginFromTypeEnum
      //} else if (LoginFromTypeEnum.APPLET_TERMINAL.getValue().equals(dto.getOriginModule())) {
      //  Validate.isTrue(UserTypeEnum.CUSTOMER.getCode().equals(userVo.getUserType()) || UserTypeEnum.TERMINAL.getCode().equals(userVo.getUserType()), "非客户用户或终端用户不能绑定终端小程序");
    } else {
      throw new IllegalArgumentException("小程序模块不存在");
    }

    if (Boolean.TRUE.equals(uniqueCheck)) {
      UserWeChatBindDto queryDto = new UserWeChatBindDto();
      queryDto.setOriginModule(dto.getOriginModule());
      queryDto.setUserName(dto.getUserName());
      List<UserRelWeChatEntity> entities = this.userRelWeChatRepository.findByUserWeChatBindDto(queryDto);
      if (CollectionUtils.isNotEmpty(entities)) {
        entities = entities.stream().filter(x -> !dto.getOpenId().equals(x.getOpenId())).collect(Collectors.toList());
      }
      Validate.isTrue(CollectionUtils.isEmpty(entities), "该用户已经绑定了微信，不能重复绑定");
    }
  }
}
