package com.biz.crm.mdm.business.user.local.repository;

import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.mdm.business.user.local.entity.UserPositionEntity;
import com.biz.crm.mdm.business.user.local.mapper.UserPositionMapper;
import com.biz.crm.mdm.business.user.sdk.dto.UserPositionPageDto;
import com.biz.crm.mdm.business.user.sdk.vo.UserPositionVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.ibatis.annotations.Param;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import javax.swing.plaf.LabelUI;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

/**
 * 职位-用户关联表(repository)
 *
 * @author xi.peng
 * @since 2021-11-05 15:02:58
 */
@Component
public class UserPositionRepository extends ServiceImpl<UserPositionMapper, UserPositionEntity> {

  /**
   * 分页
   *
   * @param pageable
   * @param dto
   * @return
   */
  public Page<UserPositionVo> findByConditions(Pageable pageable, @Param("dto") UserPositionPageDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<UserPositionPageDto> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    dto = Optional.ofNullable(dto).orElse(new UserPositionPageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());//设置租户编号信息
    return baseMapper.findByConditions(page, dto);
  }

  /**
   * 根据ID获取详情
   *
   * @param id
   * @return
   */
  public UserPositionEntity findById(String id) {
    return this.lambdaQuery()
        .eq(UserPositionEntity::getTenantCode,TenantUtils.getTenantCode())
        .in(UserPositionEntity::getId,id)
        .one();
  }

  /**
   * 根据用户账号获取用户与职位关系集合
   *
   * @param tenantCode 租户编码
   * @param userName   用户账号
   * @return List<UserPositionEntity>
   */
  public List<UserPositionEntity> findByUserName(String tenantCode, String userName) {
    return this.lambdaQuery()
            .eq(UserPositionEntity::getTenantCode, tenantCode)
            .eq(UserPositionEntity::getUserName, userName)
            .list();
  }

  /**
   * 根据用户账号获取用户与职位关系集合
   *
   * @param tenantCode    租户编码
   * @param positionCodes 职位编码集合
   * @return List<UserPositionEntity>
   */
  public List<UserPositionEntity> findByPositionCodeIn(String tenantCode, Collection<String> positionCodes) {
    return this.lambdaQuery()
            .eq(UserPositionEntity::getTenantCode, tenantCode)
            .in(UserPositionEntity::getPositionCode, positionCodes)
            .list();
  }

  /**
   * 根据用户账号获取用户与职位关系集合
   *
   * @param tenantCode 租户编码
   * @param userNames  用户账号集合
   * @return List<UserPositionEntity>
   */
  public List<UserPositionEntity> findByUserNames(String tenantCode, Collection<String> userNames) {
    return this.lambdaQuery()
            .eq(UserPositionEntity::getTenantCode, tenantCode)
            .in(UserPositionEntity::getUserName, userNames)
            .list();
  }

  /**
   * 根据用户账号逻辑删除
   *
   * @param tenantCode 租户编码
   * @param userNames  用户账号集合
   */
  public void deleteByUserNames(String tenantCode, Collection<String> userNames) {
    this.lambdaUpdate()
            .eq(UserPositionEntity::getTenantCode, tenantCode)
            .in(UserPositionEntity::getUserName, userNames)
            .remove();
  }

  /**
   * 根据职位编码逻辑删除
   *
   * @param tenantCode    租户编码
   * @param positionCodes 职位编码集合
   */
  public void deleteByPositionCodes(String tenantCode, Collection<String> positionCodes) {
    this.lambdaUpdate()
            .eq(UserPositionEntity::getTenantCode, tenantCode)
            .in(UserPositionEntity::getPositionCode, positionCodes)
            .remove();
  }

  /**
   * 根据用户账号更新所有关联记录"是否当前职位"为“否”
   *
   * @param tenantCode 租户编码
   * @param userName   用户账号
   */
  public void updateCurrentFlagFalseByUserName(String tenantCode, String userName) {
    this.lambdaUpdate()
            .eq(UserPositionEntity::getTenantCode, tenantCode)
            .eq(UserPositionEntity::getUserName, userName)
            .eq(UserPositionEntity::getCurrentFlag, Boolean.TRUE)
            .set(UserPositionEntity::getCurrentFlag, Boolean.FALSE)
            .update();
  }

  /**
   * 根据用户账号及当前职位编码更新关联记录"是否当前职位"为“是”
   *
   * @param tenantCode 租户编码
   * @param userName   用户账号
   * @param userName   职位
   */
  public void updateCurrentFlagTrueByUserNameAndPositionCode(String tenantCode, String userName, String positionCode) {
    this.lambdaUpdate()
            .eq(UserPositionEntity::getTenantCode, tenantCode)
            .eq(UserPositionEntity::getUserName, userName)
            .eq(UserPositionEntity::getPositionCode, positionCode)
            .set(UserPositionEntity::getCurrentFlag, Boolean.TRUE)
            .update();
  }

  /**
   * 查询所有用户
   */
  public List<UserPositionEntity> findAllUser(String tenantCode) {
    return this.lambdaQuery()
            .eq(UserPositionEntity::getTenantCode, tenantCode)
            .list();
  }

  /**
   * 重构修改方法
   * @param entity
   * @param tenantCode
   */
  public void updateByIdAndTenantCode(UserPositionEntity entity, String tenantCode) {
    LambdaUpdateWrapper<UserPositionEntity>lambdaUpdateWrapper= Wrappers.lambdaUpdate();
    lambdaUpdateWrapper.eq(UserPositionEntity::getTenantCode,tenantCode);
    lambdaUpdateWrapper.in(UserPositionEntity::getId,entity.getId());
    this.baseMapper.update(entity,lambdaUpdateWrapper);
  }
}
