package com.biz.crm.mdm.business.visitor.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.visitor.local.entity.VisitorPhoneEntity;
import com.biz.crm.mdm.business.visitor.local.repository.VisitorPhoneRepository;
import com.biz.crm.mdm.business.visitor.sdk.service.VisitorAccountVoService;
import com.biz.crm.mdm.business.visitor.sdk.service.VisitorPhoneVoService;
import com.biz.crm.mdm.business.visitor.sdk.dto.VisitorPhonePageDto;
import com.biz.crm.mdm.business.visitor.sdk.dto.VisitorPhoneDto;
import com.biz.crm.mdm.business.visitor.sdk.vo.VisitorAccountVo;
import com.biz.crm.mdm.business.visitor.sdk.vo.VisitorPhoneVo;
import com.biz.crm.mdm.business.visitor.sdk.event.VisitorPhoneEventListener;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;

import lombok.extern.slf4j.Slf4j;

/**
 * 游客手机表(VisitorPhone)表服务实现类
 *
 * @author xi.peng
 * @since 2022-08-30 17:23:01
 */
@Slf4j
@Service
public class VisitorPhoneVoServiceImpl implements VisitorPhoneVoService {

  @Autowired
  private VisitorPhoneRepository visitorPhoneRepository;

  @Autowired
  private VisitorAccountVoService visitorAccountVoService;

  @Autowired(required = false)
  private List<VisitorPhoneEventListener> visitorPhoneEventListeners;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<VisitorPhoneVo> findByConditions(Pageable pageable, VisitorPhonePageDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new VisitorPhonePageDto());
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    Page<VisitorPhonePageDto> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.visitorPhoneRepository.findByConditions(page, dto);
  }

  @Override
  public VisitorPhoneVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    VisitorPhoneEntity entity = this.visitorPhoneRepository.findById(id);
    if (entity == null) {
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, VisitorPhoneVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<VisitorPhoneVo> findByAccount(String account) {
    if (StringUtils.isBlank(account)) {
      return null;
    }
    List<VisitorPhoneEntity> entities = this.visitorPhoneRepository.findByAccount(TenantUtils.getTenantCode(), account);
    if (CollectionUtils.isEmpty(entities)) {
      return null;
    }
    return (List<VisitorPhoneVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, VisitorPhoneEntity.class, VisitorPhoneVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public List<VisitorPhoneVo> findByDefaultAccount() {
    VisitorAccountVo visitorAccountVo = this.visitorAccountVoService.findDefaultAccount();
    Validate.notNull(visitorAccountVo, "请联系管理员维护默认游客账号！");
    return this.findByAccount(visitorAccountVo.getAccount());
  }

  @Override
  @Transactional
  public void create(VisitorPhoneDto dto) {
    this.createValidation(dto);
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    // 扩展逻辑：自定义选择游客账号做关联
    Validate.notNull(dto.getAccount(), "请选择游客账号！");
    VisitorAccountVo visitorAccountVo = this.visitorAccountVoService.findByAccount(dto.getAccount());
    Validate.notNull(visitorAccountVo, "游客账号不存在或已删除！");
    if (BooleanEnum.FALSE.getCapital().equals(visitorAccountVo.getAttribute())) {
      List<VisitorPhoneEntity> entities = this.visitorPhoneRepository.findByAccount(TenantUtils.getTenantCode(), dto.getAccount());
      if (CollectionUtils.isNotEmpty(entities)) {
        Optional<VisitorPhoneEntity> visitorPhoneEntityOptional = entities.stream().filter(vp-> vp.getPhone().equals(dto.getPhone())).findFirst();
        Validate.isTrue(!visitorPhoneEntityOptional.isPresent(), "当前游客账号只可被一个用户使用！");
      }
    }
    VisitorPhoneEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, VisitorPhoneEntity.class, HashSet.class, ArrayList.class);
    //新增租户编号
    entity.setTenantCode(TenantUtils.getTenantCode());
    this.visitorPhoneRepository.save(entity);
    // 发送通知
    if (CollectionUtils.isNotEmpty(visitorPhoneEventListeners)) {
      VisitorPhoneVo vo = this.nebulaToolkitService.copyObjectByWhiteList(entity, VisitorPhoneVo.class, HashSet.class, ArrayList.class);
      this.visitorPhoneEventListeners.forEach(event -> event.onCreate(vo));
    }
  }

  @Override
  @Transactional
  public void createByDefaultAccount(VisitorPhoneDto dto) {
    this.createValidation(dto);
    dto.setTenantCode(TenantUtils.getTenantCode());
    dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    // 标品逻辑：使用默认游客账号做关联
    VisitorAccountVo visitorAccountVo = this.visitorAccountVoService.findDefaultAccount();
    Validate.notNull(visitorAccountVo, "请联系管理员维护默认游客账号");
    if (BooleanEnum.FALSE.getCapital().equals(visitorAccountVo.getAttribute())) {
      List<VisitorPhoneEntity> entities = this.visitorPhoneRepository.findByAccount(TenantUtils.getTenantCode(), dto.getAccount());
      if (CollectionUtils.isNotEmpty(entities)) {
        Optional<VisitorPhoneEntity> visitorPhoneEntityOptional = entities.stream().filter(vp-> vp.getPhone().equals(dto.getPhone())).findFirst();
        Validate.isTrue(!visitorPhoneEntityOptional.isPresent(), "当前游客账号只可被一个用户使用！");
      }
    }
    VisitorPhoneEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, VisitorPhoneEntity.class, HashSet.class, ArrayList.class);
    //新增租户编号
    entity.setTenantCode(TenantUtils.getTenantCode());
    this.visitorPhoneRepository.save(entity);
    // 发送通知
    if (CollectionUtils.isNotEmpty(visitorPhoneEventListeners)) {
      VisitorPhoneVo vo = this.nebulaToolkitService.copyObjectByWhiteList(entity, VisitorPhoneVo.class, HashSet.class, ArrayList.class);
      this.visitorPhoneEventListeners.forEach(event -> event.onCreate(vo));
    }
  }

  @Override
  @Transactional
  public void update(VisitorPhoneDto dto) {
    this.updateValidation(dto);
    VisitorPhoneEntity oldEntity = this.visitorPhoneRepository.findById(dto.getId());
    Validate.notNull(oldEntity, "修改信息不存在");
    VisitorPhoneEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, VisitorPhoneEntity.class, HashSet.class, ArrayList.class);
   //重构修改方法
    this.visitorPhoneRepository.updateByIdAndTenantCode(entity,TenantUtils.getTenantCode());
    // 发送修改通知
    if (CollectionUtils.isNotEmpty(visitorPhoneEventListeners)) {
      VisitorPhoneVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(oldEntity, VisitorPhoneVo.class, HashSet.class, ArrayList.class);
      VisitorPhoneVo newVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, VisitorPhoneVo.class, HashSet.class, ArrayList.class);
      this.visitorPhoneEventListeners.forEach(event -> event.onUpdate(oldVo, newVo));
    }
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<VisitorPhoneEntity> entities = this.visitorPhoneRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.visitorPhoneRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    // 发送启用通知
    if (CollectionUtils.isNotEmpty(visitorPhoneEventListeners)) {
      List<VisitorPhoneVo> voList = (List<VisitorPhoneVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, VisitorPhoneEntity.class, VisitorPhoneVo.class, HashSet.class, ArrayList.class);
      this.visitorPhoneEventListeners.forEach(event -> event.onEnable(voList));
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<VisitorPhoneEntity> entities = this.visitorPhoneRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.visitorPhoneRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    // 发送禁用通知
    if (CollectionUtils.isNotEmpty(visitorPhoneEventListeners)) {
      List<VisitorPhoneVo> voList = (List<VisitorPhoneVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, VisitorPhoneEntity.class, VisitorPhoneVo.class, HashSet.class, ArrayList.class);
      this.visitorPhoneEventListeners.forEach(event -> event.onDisable(voList));
    }
  }

  @Override
  @Transactional
  public void updateLoginTimesByPhone(String phone) {
    Validate.notBlank(phone, "手机号不能为空！");
    boolean bool = this.visitorPhoneRepository.updateLoginTimesByPhone(TenantUtils.getTenantCode(), phone);
    Validate.isTrue(bool, "根据手机号更新登录次数失败！");
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    List<VisitorPhoneEntity> entities = this.visitorPhoneRepository.findByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities), "不存在或已删除！");
    this.visitorPhoneRepository.updateDelFlagByIds(ids);
    // 发送删除通知
    if (CollectionUtils.isNotEmpty(visitorPhoneEventListeners)) {
      List<VisitorPhoneVo> voList = (List<VisitorPhoneVo>) this.nebulaToolkitService.copyCollectionByWhiteList(entities, VisitorPhoneEntity.class, VisitorPhoneVo.class, HashSet.class, ArrayList.class);
      this.visitorPhoneEventListeners.forEach(event -> event.onDelete(voList));
    }
  }

  private void createValidation(VisitorPhoneDto dto) {
    // TODO 具体实现
    this.validation(dto);
  }

  private void updateValidation(VisitorPhoneDto dto) {
    // TODO 具体实现
    this.validation(dto);
    Validate.notBlank(dto.getId(), "ID不能为空");
    Validate.notBlank(dto.getAccount(), "关联账号不能为空");
  }

  private void validation(VisitorPhoneDto dto) {
    // TODO 具体实现
    Validate.notNull(dto, "游客手机对象不能为空");
    Validate.notNull(dto.getPhone(), "用户唯一识别号不能为空");
  }
}
