package com.biz.crm.mdm.business.warehouse.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.warehouse.local.entity.WarehouseEntity;
import com.biz.crm.mdm.business.warehouse.local.repository.WarehouseCoverageRepository;
import com.biz.crm.mdm.business.warehouse.local.repository.WarehouseRepository;
import com.biz.crm.mdm.business.warehouse.local.service.WarehouseCoverageService;
import com.biz.crm.mdm.business.warehouse.local.service.WarehouseService;
import com.biz.crm.mdm.business.warehouse.sdk.dto.WarehouseDto;
import com.biz.crm.mdm.business.warehouse.sdk.dto.WarehousePageDto;
import com.biz.crm.mdm.business.warehouse.sdk.event.WarehouseEventListener;
import com.biz.crm.mdm.business.warehouse.sdk.vo.WarehouseEventVo;
import com.biz.crm.mdm.business.warehouse.sdk.vo.WarehouseVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.function.SerializableBiConsumer;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 仓库表服务实现类
 *
 * @author ning.zhang
 * @date 2021-11-19 10:27:14
 */
@Service("warehouseService")
public class WarehouseServiceImpl implements WarehouseService {

  @Autowired(required = false)
  private WarehouseRepository warehouseRepository;
  @Autowired(required = false)
  private WarehouseCoverageRepository warehouseCoverageRepository;
  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private GenerateCodeService generateCodeService;
  @Autowired(required = false)
  private WarehouseCoverageService warehouseCoverageService;
  @Autowired(required = false)
  @Lazy
  private List<WarehouseEventListener> listeners;
  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  @Override
  @Transactional
  public WarehouseEntity create(WarehouseDto dto) {
    this.createValidation(dto);
    if (StringUtils.isEmpty(dto.getWarehouseCode())) {
      dto.setWarehouseCode(this.generateCodeService.generateCode("CK", 1).get(0));
    }
    List<WarehouseEntity> list = this.warehouseRepository.findByCodes(Lists.newArrayList(dto.getWarehouseCode()));
    Validate.isTrue(CollectionUtils.isEmpty(list), "仓库编码已经存在");
    WarehouseEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, WarehouseEntity.class, HashSet.class, ArrayList.class);
    entity.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    entity.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    //新增租户编号
    entity.setTenantCode(TenantUtils.getTenantCode());
    this.warehouseRepository.save(entity);
    //绑定仓库关联数据
    this.warehouseCoverageService.update(dto.getCoverageList(), dto.getWarehouseCode());
    //新增仓库事件通知
    if (CollectionUtils.isNotEmpty(listeners)) {
      WarehouseVo newObj = this.nebulaToolkitService.copyObjectByBlankList(entity, WarehouseVo.class, HashSet.class, ArrayList.class);
      WarehouseEventVo warehouseEventVo = new WarehouseEventVo();
      warehouseEventVo.setNewWarehouse(newObj);
      warehouseEventVo.setOldWarehouse(null);
      listeners.forEach(listener -> {
        listener.onCreate(warehouseEventVo);
      });
    }
    return entity;
  }

  @Override
  @Transactional
  public WarehouseEntity update(WarehouseDto dto) {
    this.updateValidation(dto);
    WarehouseEntity entity = this.warehouseRepository.getById(dto.getId());
    Validate.notNull(entity, "仓库信息不存在");
    Validate.isTrue(entity.getWarehouseCode().equals(dto.getWarehouseCode()), "仓库编码不能修改");
    Validate.isTrue(!entity.getWarehouseDefault(), "无法修改默认仓库");
    WarehouseEntity updateEntity = this.nebulaToolkitService.copyObjectByWhiteList(dto, WarehouseEntity.class, HashSet.class, ArrayList.class);
    this.warehouseRepository.updateById(updateEntity);
    //绑定仓库关联数据
    this.warehouseCoverageService.update(dto.getCoverageList(), dto.getWarehouseCode());
    //更新仓库事件通知
    WarehouseVo oldVo = this.nebulaToolkitService.copyObjectByWhiteList(entity, WarehouseVo.class, HashSet.class, ArrayList.class);
    WarehouseVo newVo = this.nebulaToolkitService.copyObjectByBlankList(dto, WarehouseVo.class, HashSet.class, ArrayList.class);
    WarehouseEventVo eventVo = new WarehouseEventVo();
    eventVo.setOldWarehouse(oldVo);
    eventVo.setNewWarehouse(newVo);
    if (CollectionUtils.isNotEmpty(listeners)) {
      listeners.forEach(listener -> {
        listener.onUpdate(eventVo);
      });
    }
    //通过事件引擎发送事件通知
    SerializableBiConsumer<WarehouseEventListener, WarehouseEventVo> onUpdate = WarehouseEventListener::onUpdate;
    this.nebulaNetEventClient.publish(eventVo, WarehouseEventListener.class, onUpdate);
    return updateEntity;
  }

  @Override
  @Transactional
  public void deleteBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "缺失id");
    List<WarehouseEntity> entities = this.warehouseRepository.listByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "数据删除个数不匹配");
    for(WarehouseEntity entity : entities){
      Validate.isTrue(EnableStatusEnum.DISABLE.getCode().equals(entity.getEnableStatus()), "[%s]仓库已启用不能删除", entity.getWarehouseName());
    }
    List<String> warehouseCodes = entities.stream().map(WarehouseEntity::getWarehouseCode).collect(Collectors.toList());
    this.warehouseRepository.updateDelFlagByIds(ids);
    //物理删除覆盖区域
    this.warehouseCoverageRepository.deleteByWarehouseCodes(warehouseCodes);
    List<WarehouseVo> vos = (List<WarehouseVo>) nebulaToolkitService.copyCollectionByWhiteList(entities, WarehouseEntity.class, WarehouseVo.class, HashSet.class, ArrayList.class);
    WarehouseEventVo eventVo = new WarehouseEventVo();
    eventVo.setWarehouseVoList(vos);
    //删除仓库事件通知
    SerializableBiConsumer<WarehouseEventListener, WarehouseEventVo> sf = WarehouseEventListener::onDelete;
    nebulaNetEventClient.publish(eventVo, WarehouseEventListener.class, sf);
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "缺失id");
    List<WarehouseEntity> entities = this.warehouseRepository.listByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "数据启用个数不匹配");
    this.warehouseRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
    //启用仓库事件通知
    if (CollectionUtils.isNotEmpty(listeners)) {
      List<WarehouseVo> voList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(entities, WarehouseEntity.class
          , WarehouseVo.class, HashSet.class, ArrayList.class));
      WarehouseEventVo eventVo = new WarehouseEventVo();
      eventVo.setWarehouseVoList(voList);
      listeners.forEach(listener -> {
        listener.onEnable(eventVo);
      });
    }
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "缺失id");
    List<WarehouseEntity> entities = this.warehouseRepository.listByIds(ids);
    Validate.isTrue(CollectionUtils.isNotEmpty(entities) && entities.size() == ids.size(), "数据禁用个数不匹配");
    this.warehouseRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
    //禁用仓库事件通知
    if (CollectionUtils.isNotEmpty(listeners)) {
      List<WarehouseVo> voList = Lists.newArrayList(nebulaToolkitService.copyCollectionByWhiteList(entities, WarehouseEntity.class
          , WarehouseVo.class, HashSet.class, ArrayList.class));
      WarehouseEventVo eventVo = new WarehouseEventVo();
      eventVo.setWarehouseVoList(voList);
      listeners.forEach(listener -> {
        listener.onDisable(eventVo);
      });
    }
  }

  @Override
  public Page<WarehouseEntity> findByConditions(Pageable pageable, WarehousePageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new WarehousePageDto());
    return this.warehouseRepository.findByConditions(pageable, dto);
  }

  @Override
  public WarehouseEntity findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    return this.warehouseRepository.findById(id);
  }

  /**
   * 在创建warehouse模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void createValidation(WarehouseDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    dto.setId(null);
    Validate.notNull(dto.getWarehouseDefault(), "缺失默认仓库标识");
    Validate.notBlank(dto.getWarehouseName(), "缺失仓库名称");
    Validate.notBlank(dto.getWarehouseAddress(), "缺失仓库地址");
    Validate.isTrue(!dto.getWarehouseDefault(), "无法创建默认仓库");
    Validate.notNull(dto.getLatitude(), "缺失纬度");
    Validate.notNull(dto.getLongitude(), "缺失经度");
    Validate.isTrue(dto.getWarehouseName().length() < 128, "仓库名称，在进行编辑时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(StringUtils.isBlank(dto.getWarehouseCode()) || dto.getWarehouseCode().length() < 64, "仓库编码，在进行编辑时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dto.getWarehouseAddress().length() < 255, "仓库地址，在进行编辑时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(StringUtils.isBlank(dto.getWarehouseHead()) || dto.getWarehouseHead().length() < 32, "仓库责任人，在进行编辑时填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(StringUtils.isBlank(dto.getContactPhone()) || dto.getContactPhone().length() < 64, "联系人电话，在进行编辑时填入值超过了限定长度(64)，请检查!");
    if(dto.getWarehouseDefault()){
      Validate.isTrue(CollectionUtils.isEmpty(dto.getCoverageList()), "设置为默认仓库是，不能填写覆盖区域");
    }else {
      Validate.isTrue(CollectionUtils.isNotEmpty(dto.getCoverageList()), "设置为非默认仓库是，必须填写覆盖区域");
    }
  }

  /**
   * 在修改warehouse模型对象之前，检查对象各属性的正确性，其主键属性必须没有值
   *
   * @param dto 检查对象
   */
  private void updateValidation(WarehouseDto dto) {
    Validate.notNull(dto, "进行当前操作时，信息对象必须传入!");
    Validate.notBlank(dto.getId(), "修改信息时，id不能为空！");
    Validate.notBlank(dto.getWarehouseCode(), "缺失仓库编码");
    Validate.notBlank(dto.getWarehouseName(), "缺失仓库名称");
    Validate.notBlank(dto.getWarehouseAddress(), "缺失仓库地址");
    Validate.notNull(dto.getLatitude(), "缺失纬度");
    Validate.notNull(dto.getLongitude(), "缺失经度");
    Validate.notNull(dto.getWarehouseDefault(), "缺失默认仓库标识");
    Validate.isTrue(!dto.getWarehouseDefault(), "无法修改默认仓库");
    Validate.isTrue(dto.getWarehouseName().length() < 128, "仓库名称，在进行编辑时填入值超过了限定长度(128)，请检查!");
    Validate.isTrue(dto.getWarehouseCode().length() < 64, "仓库编码，在进行编辑时填入值超过了限定长度(64)，请检查!");
    Validate.isTrue(dto.getWarehouseAddress().length() < 255, "仓库地址，在进行编辑时填入值超过了限定长度(255)，请检查!");
    Validate.isTrue(StringUtils.isBlank(dto.getWarehouseHead()) || dto.getWarehouseHead().length() < 32, "仓库责任人，在进行编辑时填入值超过了限定长度(32)，请检查!");
    Validate.isTrue(StringUtils.isBlank(dto.getContactPhone()) || dto.getContactPhone().length() < 64, "联系人电话，在进行编辑时填入值超过了限定长度(64)，请检查!");
  }
}
