package com.biz.crm.mdm.business.warehouse.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.mdm.business.region.sdk.service.RegionVoService;
import com.biz.crm.mdm.business.warehouse.local.entity.WarehouseEntity;
import com.biz.crm.mdm.business.warehouse.local.repository.WarehouseRepository;
import com.biz.crm.mdm.business.warehouse.local.service.WarehouseService;
import com.biz.crm.mdm.business.warehouse.sdk.dto.WarehousePageDto;
import com.biz.crm.mdm.business.warehouse.sdk.service.WarehouseVoService;
import com.biz.crm.mdm.business.warehouse.sdk.vo.WarehouseVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.collect.Lists;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

/**
 * 仓库vo服务接口实现类
 *
 * @author ning.zhang
 * @date 2022/3/9
 */
@Service
public class WarehouseVoServiceImpl implements WarehouseVoService {
  @Autowired(required = false)
  private WarehouseRepository warehouseRepository;
  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;
  @Autowired(required = false)
  private WarehouseService warehouseService;
  @Autowired(required = false)
  private RegionVoService regionVoService;

  @Override
  public WarehouseVo findDetailsByCode(String warehouseCode) {
    WarehouseEntity entity = this.warehouseRepository.findDetailsByCode(warehouseCode);
    Validate.notNull(entity, "仓库信息不存在");
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, WarehouseVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 通过覆盖区域市Code获取仓库信息
   * @param cityCode
   * @return
   */
  @Override
  public WarehouseVo findDetailsByCityCode(String cityCode) {
    if(StringUtils.isBlank(cityCode)){
      return null;
    }
    WarehouseEntity entity = this.warehouseRepository.findDetailsByCityCode(cityCode);
    if(entity == null){
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, WarehouseVo.class, HashSet.class, ArrayList.class);
  }

  /**
   * 获取默认仓库
   * @param bool
   * @return
   */
  @Override
  public WarehouseVo findDetailsByWarehouseDefault(Boolean bool) {
    if(bool == null){
      return null;
    }
    WarehouseEntity entity = this.warehouseRepository.findDetailsByWarehouseDefault(bool);
    if(entity == null){
      return null;
    }
    return this.nebulaToolkitService.copyObjectByWhiteList(entity, WarehouseVo.class, HashSet.class, ArrayList.class);
  }

  @Override
  public Page<WarehouseVo> findByConditions(Pageable pageable, WarehousePageDto dto) {
    pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    dto = ObjectUtils.defaultIfNull(dto, new WarehousePageDto());
    Page<WarehouseEntity> entityPage = this.warehouseService.findByConditions(pageable, dto);
    Page<WarehouseVo> pageResult = new Page<>(entityPage.getCurrent(), entityPage.getSize(), entityPage.getTotal());
    if (CollectionUtils.isNotEmpty(entityPage.getRecords())) {
      pageResult.setRecords(convertEntityToVo(entityPage.getRecords()));
    }
    return pageResult;
  }

  /**
   * 仓库实体列表转VO列表
   *
   * @param entities 仓库实体列表
   * @return 仓库VO列表
   */
  private List<WarehouseVo> convertEntityToVo(List<WarehouseEntity> entities) {
    List<WarehouseVo> list = Lists.newArrayList(this.nebulaToolkitService.copyCollectionByWhiteList(entities, WarehouseEntity.class
        , WarehouseVo.class, HashSet.class, ArrayList.class));
    List<String> regionCodes = Lists.newArrayList();
    list.forEach(warehouseVo -> {
      if (StringUtils.isNotBlank(warehouseVo.getCityCode())) {
        regionCodes.add(warehouseVo.getCityCode());
      }
      if (StringUtils.isNotBlank(warehouseVo.getProvinceCode())) {
        regionCodes.add(warehouseVo.getProvinceCode());
      }
      if (StringUtils.isNotBlank(warehouseVo.getDistrictCode())) {
        regionCodes.add(warehouseVo.getDistrictCode());
      }
    });
    if (CollectionUtils.isEmpty(regionCodes)) {
      return list;
    }
    Map<String, String> regionCodeNameMap = this.regionVoService.findRegionCodeNameMap(regionCodes);
    list.forEach(warehouseVo -> {
      warehouseVo.setProvinceName(regionCodeNameMap.get(warehouseVo.getProvinceCode()));
      warehouseVo.setCityName(regionCodeNameMap.get(warehouseVo.getCityCode()));
      warehouseVo.setDistrictName(regionCodeNameMap.get(warehouseVo.getDistrictCode()));
    });
    return list;
  }

}
