package com.biz.crm.mdm.business.warehouse.local.controller;

import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.mdm.business.warehouse.local.entity.WarehouseEntity;
import com.biz.crm.mdm.business.warehouse.local.service.WarehouseService;
import com.biz.crm.mdm.business.warehouse.sdk.dto.WarehouseDto;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * 仓库信息: Warehouse: 仓库管理
 *
 * @author ning.zhang
 * @date 2021/11/19
 */
@Slf4j
@RestController
@RequestMapping("/v1/warehouse/warehouse")
@Api(tags = "仓库信息: Warehouse: 仓库管理")
public class WarehouseController {

  @Autowired(required = false)
  private WarehouseService warehouseService;

  /**
   * 新增仓库信息
   *
   * @param dto 参数dto
   * @return 新增仓库数据
   */
  @ApiOperation(value = "新增仓库")
  @PostMapping("")
  public Result create(@RequestBody WarehouseDto dto) {
    try {
      return Result.ok(this.warehouseService.create(dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 更新仓库信息
   *
   * @param dto 参数dto
   * @return 更新后仓库数据
   */
  @ApiOperation(value = "更新仓库信息")
  @PatchMapping("")
  public Result update(@RequestBody WarehouseDto dto) {
    try {
      return Result.ok(this.warehouseService.update(dto));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 删除仓库
   *
   * @param ids 仓库id集合
   * @return 操作结果
   */
  @ApiOperation(value = "删除")
  @DeleteMapping("/deleteBatch")
  public Result deleteBatch(@RequestParam("ids") List<String> ids) {
    try {
      this.warehouseService.deleteBatch(ids);
      return Result.ok("删除成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 启用仓库
   *
   * @param ids 仓库id集合
   * @return 操作结果
   */
  @ApiOperation(value = "启用")
  @PatchMapping("/enableBatch")
  public Result enableBatch(@RequestBody List<String> ids) {
    try {
      this.warehouseService.enableBatch(ids);
      return Result.ok("启用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 禁用仓库
   *
   * @param ids 仓库id集合
   * @return 操作结果
   */
  @ApiOperation(value = "禁用")
  @PatchMapping("/disableBatch")
  public Result disableBatch(@RequestBody List<String> ids) {
    try {
      this.warehouseService.disableBatch(ids);
      return Result.ok("禁用成功");
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }

  /**
   * 通过ID获取仓库信息
   *
   * @param id 仓库ID
   * @return 操作结果
   */
  @ApiOperation(value = "通过ID获取仓库信息")
  @GetMapping("/findById")
  public Result<WarehouseEntity> findById(@RequestParam(value = "id", required = false) @ApiParam(name = "id", value = "主键ID") String id) {
    try {
      return Result.ok(this.warehouseService.findById(id));
    } catch (RuntimeException e) {
      log.error(e.getMessage(), e);
      return Result.error(e.getMessage());
    }
  }
}
