package com.biz.crm.mdm.business.warehouse.local.repository;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.warehouse.local.entity.WarehouseCoverageEntity;
import com.biz.crm.mdm.business.warehouse.local.entity.WarehouseEntity;
import com.biz.crm.mdm.business.warehouse.local.mapper.WarehouseMapper;
import com.biz.crm.mdm.business.warehouse.sdk.dto.WarehousePageDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * 仓库表的数据库访问类 {@link WarehouseEntity}
 *
 * @author ning.zhang
 * @date 2021-11-19 10:27:13
 */
@Component
public class WarehouseRepository extends ServiceImpl<WarehouseMapper, WarehouseEntity> {

  /**
   * 根据仓库编码列表获取仓库列表
   *
   * @param warehouseCodes 仓库编码集合
   * @return 仓库列表
   */
  public List<WarehouseEntity> findByCodes(List<String> warehouseCodes) {
    return this.lambdaQuery()
        .eq(WarehouseEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .in(WarehouseEntity::getWarehouseCode, warehouseCodes)
        .eq(WarehouseEntity::getTenantCode, TenantUtils.getTenantCode())
        .list();
  }

  /**
   * 批量删除
   *
   * @param ids ID集合
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate()
        .in(WarehouseEntity::getId, ids)
        .eq(WarehouseEntity::getTenantCode, TenantUtils.getTenantCode())
        .set(WarehouseEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

  /**
   * 批量更新启用禁用状态
   *
   * @param ids          ID集合
   * @param enableStatus 启用禁用状态
   */
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enableStatus) {
    this.lambdaUpdate()
        .in(WarehouseEntity::getId, ids)
        .eq(WarehouseEntity::getTenantCode, TenantUtils.getTenantCode())
        .set(WarehouseEntity::getEnableStatus, enableStatus.getCode())
        .update();
  }

  /**
   * 仓库分页列表
   *
   * @param dto      请求参数dto
   * @param pageable 分页信息
   * @return Page<WarehouseEntity>仓库分页信息
   */
  public Page<WarehouseEntity> findByConditions(Pageable pageable, WarehousePageDto dto) {
    Page<WarehouseEntity> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.baseMapper.findByConditions(page, dto, TenantUtils.getTenantCode());
  }

  /**
   * 通过ID获取仓库信息
   *
   * @param id 仓库ID
   * @return 仓库信息
   */
  public WarehouseEntity findById(String id) {
    return this.baseMapper.findById(id, TenantUtils.getTenantCode());
  }

  /**
   * 通过code获取仓库信息
   *
   * @param warehouseCode
   * @return
   */
  public WarehouseEntity findDetailsByCode(String warehouseCode) {
    return this.lambdaQuery()
        .eq(WarehouseEntity::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(WarehouseEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(WarehouseEntity::getWarehouseCode, warehouseCode)
        .eq(WarehouseEntity::getTenantCode, TenantUtils.getTenantCode())
        .one();
  }

  /**
   * 通过默认仓库标识获取仓库信息
   *
   * @param warehouseDefault 默认仓库标识
   * @return 仓库信息
   */
  public List<WarehouseEntity> findByWarehouseDefault(Boolean warehouseDefault) {
    return this.lambdaQuery()
        .eq(WarehouseEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .in(WarehouseEntity::getWarehouseDefault, warehouseDefault)
        .eq(WarehouseEntity::getTenantCode, TenantUtils.getTenantCode())
        .list();
  }

  /**
   * 通过覆盖区域市Code获取仓库信息
   * @param cityCode
   * @return
   */
  public WarehouseEntity findDetailsByCityCode(String cityCode) {
    return this.baseMapper.findDetailsByCityCode(cityCode, TenantUtils.getTenantCode());
  }

  /**
   * 获取默认仓库
   * @param bool
   * @return
   */
  public WarehouseEntity findDetailsByWarehouseDefault(Boolean bool) {
    return this.baseMapper.findDetailsByWarehouseDefault(bool, TenantUtils.getTenantCode());
  }
}
