package com.biz.crm.mdm.business.warehouse.local.service.internal;

import com.biz.crm.mdm.business.warehouse.local.entity.WarehouseCoverageEntity;
import com.biz.crm.mdm.business.warehouse.local.repository.WarehouseCoverageRepository;
import com.biz.crm.mdm.business.warehouse.local.service.WarehouseCoverageService;
import com.biz.crm.mdm.business.warehouse.sdk.dto.WarehouseCoverageDto;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 仓库覆盖区域表服务实现类
 *
 * @author ning.zhang
 * @date 2021-11-19 10:27:45
 */
@Slf4j
@Service("warehouseCoverageService")
public class WarehouseCoverageServiceImpl implements WarehouseCoverageService {

  @Autowired(required = false)
  private WarehouseCoverageRepository warehouseCoverageRepository;
  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  @Transactional
  public void update(List<WarehouseCoverageDto> dtoList, String warehouseCode) {
    Validate.notBlank(warehouseCode, "仓库编码不能为空");
    this.warehouseCoverageRepository.deleteByWarehouseCode(warehouseCode);
    if (CollectionUtils.isEmpty(dtoList)) {
      return;
    }
    this.validate(dtoList,warehouseCode);
    //过滤省份为空的数据
    List<WarehouseCoverageEntity> entities = dtoList.stream()
        .filter(dto -> Objects.nonNull(dto) && StringUtils.isNotEmpty(dto.getProvinceCode()))
        .map(dto -> {
          WarehouseCoverageEntity entity = this.nebulaToolkitService.copyObjectByWhiteList(dto, WarehouseCoverageEntity.class, HashSet.class, ArrayList.class);
          entity.setWarehouseCode(warehouseCode);
          //新增租户编号
          entity.setTenantCode(TenantUtils.getTenantCode());
          return entity;
        }).collect(Collectors.toList());
    if (CollectionUtils.isEmpty(entities)) {
      return;
    }
    this.warehouseCoverageRepository.saveBatch(entities);
  }

  /**
   * 验证区域是否重复
   * @param dtoList
   */
  private void validate(List<WarehouseCoverageDto> dtoList,String warehouseCode) {
    //先校验当前仓库是否存在重复区域
    Set<String> set = new HashSet<>();
    for(WarehouseCoverageDto item : dtoList){
      String provinceAndCity = item.getProvinceCode().concat(item.getCityCode());
      if(!set.add(provinceAndCity)){
        Validate.isTrue(false, "当前仓库覆盖区域存在重复城市:[%s][%s]", item.getProvinceName(), item.getCityName());
      }
    }

    for(WarehouseCoverageDto dto : dtoList){
      Validate.notBlank(dto.getCityCode(), "城市不能为空");
      //查询 同一租户下 其他仓库是否存在当前覆盖城市(去掉本身)
      String tenantCode = TenantUtils.getTenantCode();
      tenantCode = StringUtils.isEmpty(tenantCode) ? "default" : tenantCode;
      List<WarehouseCoverageEntity> list = warehouseCoverageRepository.lambdaQuery()
              .eq(WarehouseCoverageEntity::getCityCode, dto.getCityCode())
              .eq(WarehouseCoverageEntity::getTenantCode, TenantUtils.getTenantCode())
              .ne(WarehouseCoverageEntity::getWarehouseCode, warehouseCode)
              .list();
      Validate.isTrue(CollectionUtils.isEmpty(list), "[%s][%s]已有[%s]个仓库覆盖", dto.getProvinceName(), dto.getCityName(), list.size());
    }
  }
}
