package com.biz.crm.mdm.business.customer.channel.local.repository;


import java.util.Collections;
import java.util.List;
import java.util.Optional;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.mdm.business.customer.channel.local.entity.CustomerChannel;
import com.biz.crm.mdm.business.customer.channel.local.mapper.CustomerChannelMapper;
import com.biz.crm.mdm.business.customer.channel.sdk.dto.CustomerChannelDto;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;


/**
 * 客户渠道主表(CustomerChannel)表数据库访问层
 *
 * @author HanJiaJun
 * @since 2022-10-31 17:04:11
 */
@Component
public class CustomerChannelRepository extends ServiceImpl<CustomerChannelMapper, CustomerChannel> {

    @Autowired
    private CustomerChannelMapper customerChannelMapper;

    /**
     * 分页查询数据
     *
     * @param pageable 分页对象
     * @param dto      实体对象
     * @return
     */
    public Page<CustomerChannelVo> findByConditions(Pageable pageable, CustomerChannelDto dto) {
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        dto = Optional.ofNullable(dto).orElse(new CustomerChannelDto());
        dto.setTenantCode(TenantUtils.getTenantCode());
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        Page<CustomerChannelVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return customerChannelMapper.findByConditions(page, dto);
    }

    /**
     * 禁用
     *
     * @param idList
     */
    public void disableBatch(List<String> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return;
        }
        this.lambdaUpdate()
                .set(CustomerChannel::getEnableStatus, EnableStatusEnum.DISABLE.getCode())
                .in(CustomerChannel::getId, idList)
                .eq(CustomerChannel::getTenantCode, TenantUtils.getTenantCode())
                .update();
    }

    /**
     * 启用
     *
     * @param idList
     */
    public void enableBatch(List<String> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return;
        }
        this.lambdaUpdate()
                .set(CustomerChannel::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                .in(CustomerChannel::getId, idList)
                .eq(CustomerChannel::getTenantCode, TenantUtils.getTenantCode())
                .update();
    }

    /**
     * 删除
     *
     * @param idList
     */
    public void delete(List<String> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return;
        }
        this.lambdaUpdate()
                .set(CustomerChannel::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .in(CustomerChannel::getId, idList)
                .eq(CustomerChannel::getTenantCode, TenantUtils.getTenantCode())
                .update();
    }

    /**
     * 根据id获取详情
     *
     * @param id 员工ID
     * @return
     */
    public CustomerChannel findById(String id) {
        if (!StringUtils.hasText(id)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(CustomerChannel::getId, id)
                .eq(CustomerChannel::getTenantCode, TenantUtils.getTenantCode())
                .eq(CustomerChannel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .one();
    }

    /**
     * 根据id集合获取详情集合
     *
     * @param idList ID集合
     * @return
     */
    public List<CustomerChannel> findByIds(List<String> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(CustomerChannel::getId, idList)
                .eq(CustomerChannel::getTenantCode, TenantUtils.getTenantCode())
                .eq(CustomerChannel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }

    /**
     * 根据编码获取详情
     *
     * @param code 编码
     * @return
     */
    public CustomerChannel findByCode(String code) {
        if (!StringUtils.hasText(code)) {
            return null;
        }
        return this.lambdaQuery()
                .eq(CustomerChannel::getCustomerChannelCode, code)
                .eq(CustomerChannel::getTenantCode, TenantUtils.getTenantCode())
                .eq(CustomerChannel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .one();
    }

    /**
     * 根据编码集合获取详情集合
     *
     * @param codeList 编码集合
     * @return
     */
    public List<CustomerChannel> findByCodes(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return Collections.emptyList();
        }
        return this.lambdaQuery()
                .in(CustomerChannel::getCustomerChannelCode, codeList)
                .eq(CustomerChannel::getTenantCode, TenantUtils.getTenantCode())
                .eq(CustomerChannel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
                .list();
    }


    /**
     * 根据所属客户渠道编码查询所属客户渠道区域
     * @param customerChannelCode
     * @return
     */
    public CustomerChannel findByIndustryDeliveryWayName(String customerChannelCode) {
        if (StringUtils.isEmpty(customerChannelCode)) {
            return new CustomerChannel();
        }
        return this.lambdaQuery()
                .in(CustomerChannel::getCustomerChannelCode, customerChannelCode)
                .eq(CustomerChannel::getTenantCode, TenantUtils.getTenantCode())
                .one();
    }

    /**
     * 根据渠道名称查渠道信息
     *
     * @param name
     * @return
     */
    public List<CustomerChannelVo> findByName(String name) {
        return customerChannelMapper.findByName(name);
    }

    /**
     * 根据编码或者名称查询
     * @param codeOrName
     * @return {@link List}<{@link CustomerChannelVo}>
     */
    public List<CustomerChannelVo> findByCodeOrName(List<String> codeOrName) {
        Assert.notEmpty(codeOrName, "编码名称查询条件不能为空");
        CustomerChannelDto dto = new CustomerChannelDto();
        dto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        dto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        dto.setTenantCode(TenantUtils.getTenantCode());
        dto.setCodeOrNameList(codeOrName);
        return this.baseMapper.findByCustomerChannelDto(dto);
    }
}

