package com.biz.crm.mdm.business.customer.channel.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.mdm.business.customer.channel.local.entity.CustomerChannel;
import com.biz.crm.mdm.business.customer.channel.local.repository.CustomerChannelRepository;
import com.biz.crm.mdm.business.customer.channel.sdk.dto.CustomerChannelDto;
import com.biz.crm.mdm.business.customer.channel.sdk.service.CustomerChannelVoService;
import com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.util.*;

/**
 * 客户渠道信息(CustomerChannel)表服务实现类
 *
 * @author: huxmld
 * @version: v1.0.0
 * @date: 2022.11.4 23:13
 */
@Slf4j
@Service
public class CustomerChannelVoServiceImpl implements CustomerChannelVoService {

    @Autowired(required = false)
    private CustomerChannelRepository customerChannelRepository;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;


    /**
     * 查询分页列表
     *
     * @param pageable 分页参数
     * @param dto      查询参数
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.mdm.business.customer.channel.sdk.vo.CustomerChannelVo>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:52
     */
    @Override
    public Page<CustomerChannelVo> findByConditions(Pageable pageable, CustomerChannelDto dto) {
        return this.customerChannelRepository.findByConditions(pageable, dto);
    }

    /**
     * 按id查询详情
     *
     * @param id ID
     * @return java.util.List<com.biz.crm.mdm.business.customer.channel.local.entity.CustomerChannel>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:09
     */
    @Override
    public CustomerChannelVo findById(String id) {
        if (StringUtils.isEmpty(id)) {
            return null;
        }
        CustomerChannel promoters = this.customerChannelRepository.findById(id);
        if (Objects.isNull(promoters)) {
            return null;
        }
        return this.nebulaToolkitService.copyObjectByBlankList(
                promoters, CustomerChannelVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 按id集合查询详情
     *
     * @param idList ID集合
     * @return java.util.List<com.biz.crm.mdm.business.customer.channel.local.entity.CustomerChannel>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:09
     */
    @Override
    public List<CustomerChannelVo> findByIds(List<String> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return Collections.emptyList();
        }
        List<CustomerChannel> promotersList = this.customerChannelRepository.findByIds(idList);
        if (CollectionUtils.isEmpty(promotersList)) {
            return Collections.emptyList();
        }

        return (List<CustomerChannelVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                promotersList, CustomerChannel.class, CustomerChannelVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 按编码查询详情
     *
     * @param code 编码
     * @return java.util.List<com.biz.crm.mdm.business.customer.channel.local.entity.CustomerChannel>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:09
     */
    @Override
    public CustomerChannelVo findByCode(String code) {
        if (StringUtils.isEmpty(code)) {
            return null;
        }
        CustomerChannel entity = this.customerChannelRepository.findByCode(code);
        if (Objects.isNull(entity)) {
            return null;
        }
        return this.nebulaToolkitService.copyObjectByBlankList(
                entity, CustomerChannelVo.class,
                HashSet.class, ArrayList.class);
    }


    /**
     * 按编码集合查询详情
     *
     * @param codeList 编码集合
     * @return java.util.List<com.biz.crm.mdm.business.customer.channel.local.entity.CustomerChannel>
     * @describe: 简述
     * @author: huxmld
     * @version: v1.0.0
     * @date: 2022.11.4 23:09
     */
    @Override
    public List<CustomerChannelVo> findByCodes(List<String> codeList) {
        if (CollectionUtils.isEmpty(codeList)) {
            return Collections.emptyList();
        }
        List<CustomerChannel> entityList = this.customerChannelRepository.findByCodes(codeList);
        if (CollectionUtils.isEmpty(entityList)) {
            return Collections.emptyList();
        }
        return (List<CustomerChannelVo>) this.nebulaToolkitService.copyCollectionByBlankList(
                entityList, CustomerChannel.class, CustomerChannelVo.class,
                HashSet.class, ArrayList.class);
    }

    /**
     * 根据所属客户渠道编码查询所属客户渠道区域
     * @param customerChannelCode
     * @return
     */
    @Override
    public String findByCustomerChannelName(String customerChannelCode) {
        Validate.notEmpty(customerChannelCode,"客户渠道编码不能为空");
        CustomerChannel customerChannel =  this.customerChannelRepository.findByIndustryDeliveryWayName(customerChannelCode);
        if (ObjectUtils.isEmpty(customerChannel)) {
            return null;
        }
        return customerChannel.getCustomerChannelName();
    }

    /**
     * 根据渠道名称查渠道信息
     *
     * @param name
     * @return
     */
    @Override
    public List<CustomerChannelVo> findByName(String name) {
        if (StringUtils.isBlank(name)){
            return null;
        }
        return this.customerChannelRepository.findByName(name);
    }

    /**
     * 根据编码或者名称查询
     * @param codeOrName
     * @return {@link List}<{@link CustomerChannelVo}>
     */
    @Override
    public List<CustomerChannelVo> findByCodeOrName(List<String> codeOrName) {
        return this.customerChannelRepository.findByCodeOrName(codeOrName);
    }
}
