package com.biz.crm.mdm.business.terminal.channel.local.repository;


import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.local.entity.UuidFlagOpEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.vo.LazyTreeVo;
import com.biz.crm.mdm.business.terminal.channel.local.dto.TerminalChannelSelectDto;
import com.biz.crm.mdm.business.terminal.channel.local.entity.MdmTerminalChannel;
import com.biz.crm.mdm.business.terminal.channel.local.mapper.MdmTerminalChannelMapper;
import com.biz.crm.mdm.business.terminal.channel.sdk.vo.TerminalChannelLazyTreeVo;
import com.biz.crm.mn.common.base.constant.CommonSelectConstant;
import com.biz.crm.mn.common.base.vo.CommonSelectVo;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.*;


/**
 * 终端渠道主表(MdmTerminalChannel)表数据库访问层
 *
 * @author HanJiaJun
 * @since 2022-11-04 14:33:37
 */
@Component
public class MdmTerminalChannelRepository extends ServiceImpl<MdmTerminalChannelMapper, MdmTerminalChannel> {

  @Autowired
  private MdmTerminalChannelMapper mdmTerminalChannelMapper;

  /**
   * 分页查询数据
   *
   * @param pageable           分页对象
   * @param mdmTerminalChannel 实体对象
   * @return
   */
  public Page<MdmTerminalChannel> findByConditions(Pageable pageable, MdmTerminalChannel mdmTerminalChannel) {
    Page<MdmTerminalChannel> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    //新增租户编号
    mdmTerminalChannel.setTenantCode(TenantUtils.getTenantCode());
    Page<MdmTerminalChannel> pageList = mdmTerminalChannelMapper.findByConditions(page, mdmTerminalChannel);
    return pageList;
  }

  public List<MdmTerminalChannel> findSelectByConditions(TerminalChannelSelectDto dto) {
    return mdmTerminalChannelMapper.findSelectByConditions(dto, CommonSelectConstant.COMMON_SELECT_DEFAULT_SIZE);
  }


  public MdmTerminalChannel findByCode(String terminalChannelCode) {
    return this.lambdaQuery()
        .eq(MdmTerminalChannel::getTerminalChannelCode, terminalChannelCode)
        .eq(MdmTerminalChannel::getTenantCode, TenantUtils.getTenantCode())
        .eq(MdmTerminalChannel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .one();
  }

  public List<MdmTerminalChannel> findChildrenListByParentCode(String parentChannelCode) {
    return this.lambdaQuery()
        .eq(MdmTerminalChannel::getParentChannelCode, parentChannelCode)
        .eq(MdmTerminalChannel::getTenantCode, TenantUtils.getTenantCode())
        .eq(MdmTerminalChannel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .list();
  }

  /**
   * 通过渠道编码更新当前的RuleCode和levelNum
   *
   * @param terminalChannelCode
   * @param ruleCodeByParentCode
   * @param levelNum
   */
  public void updateRuleCodeAndLevelNumByChannelCode(String terminalChannelCode, String ruleCodeByParentCode, int levelNum) {
    this.lambdaUpdate()
        .eq(MdmTerminalChannel::getTerminalChannelCode, terminalChannelCode)
        .eq(MdmTerminalChannel::getTenantCode,TenantUtils.getTenantCode())
        .set(MdmTerminalChannel::getRuleCode, ruleCodeByParentCode)
        .set(MdmTerminalChannel::getLevelNum, levelNum)
        .update();
  }

  /**
   * 逻辑删除
   *
   * @param idList
   */
  public void updateDelFlagByids(List<String> idList) {
    this.lambdaUpdate()
        .in(MdmTerminalChannel::getId, idList)
        .eq(MdmTerminalChannel::getTenantCode,TenantUtils.getTenantCode())
        .set(MdmTerminalChannel::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }


  public List<MdmTerminalChannel> findByIds(List<String> idList) {
    return this.lambdaQuery()
        .in(MdmTerminalChannel::getId, idList)
        .eq(MdmTerminalChannel::getTenantCode,TenantUtils.getTenantCode())
        .list();
  }


  public List<MdmTerminalChannel> findChildrenListByParentCodes(Set<String> parentCodes) {
    return this.lambdaQuery()
        .in(MdmTerminalChannel::getParentChannelCode, parentCodes)
        .eq(MdmTerminalChannel::getTenantCode, TenantUtils.getTenantCode())
        .eq(MdmTerminalChannel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .list();
  }

  /**
   * 通过规则编码和 启禁用状态查询
   *
   * @param parentRuleCode
   * @param enableCode
   * @return
   */
  public List<MdmTerminalChannel> findbyRuleCodesAndEnableStatus(Set<String> parentRuleCode, String enableCode) {
    return this.lambdaQuery()
        .in(MdmTerminalChannel::getRuleCode, parentRuleCode)
        .eq(MdmTerminalChannel::getTenantCode, TenantUtils.getTenantCode())
        .eq(MdmTerminalChannel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(MdmTerminalChannel::getEnableStatus, enableCode)
        .list();
  }

  /**
   * 批量启用
   *
   * @param idList
   * @param enableCode
   */
  public void updateEnableStatusByIds(List<String> idList, String enableCode) {
    this.lambdaUpdate()
        .in(MdmTerminalChannel::getId, idList)
        .eq(MdmTerminalChannel::getTenantCode,TenantUtils.getTenantCode())
        .set(MdmTerminalChannel::getEnableStatus, enableCode)
        .update();
  }

  /**
   * 查询当前及全部下级渠道
   *
   * @param ruleCodes
   * @param enableCode 启禁用标记
   */
  public List<MdmTerminalChannel> findCurAndChildrenByRuleCodes(List<String> ruleCodes, String enableCode) {
    return this.mdmTerminalChannelMapper.findCurAndChildrenByRuleCodes(ruleCodes, enableCode, TenantUtils.getTenantCode());
  }


  /**
   * 查找所有上级为空的数据
   *
   * @param tenantCode
   * @return
   */
  public List<MdmTerminalChannel> findByParentCodeIsNull(String tenantCode) {
    return this.lambdaQuery()
        .isNull(MdmTerminalChannel::getParentChannelCode).or().eq(MdmTerminalChannel::getParentChannelCode, "")
        .eq(MdmTerminalChannel::getTenantCode, tenantCode)
        .eq(MdmTerminalChannel::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .list();
  }

  /**
   * 获取父级编码不为空但找不到对应的上级的数据的ID
   *
   * @param tenantCode
   * @return
   */
  public List<String> getIdByParentCodeNull(String tenantCode) {
    return this.mdmTerminalChannelMapper.getIdByParentCodeNull(tenantCode);
  }

  /**
   * 将其父级设成null
   *
   * @param ids
   */
  public void updateParentCodeNullById(List<String> ids) {
    this.lambdaUpdate()
        .in(MdmTerminalChannel::getId, ids)
        .eq(MdmTerminalChannel::getTenantCode,TenantUtils.getTenantCode())
        .set(MdmTerminalChannel::getParentChannelCode, null)
        .update();
  }

  /**
   * 根据 【EnableStatus】 orgName模糊 查询orgCode ruleCode
   *
   * @param enableStatus
   * @param name
   * @param tenantCode
   * @return
   */
  public List<MdmTerminalChannel> findOrgCodeAndRuleCodeByEnableStatusOptAndOrgNameLike(String enableStatus, String name, String tenantCode) {
    return this.lambdaQuery()
        .eq(StringUtils.isNotEmpty(enableStatus),MdmTerminalChannel::getEnableStatus,enableStatus)
        .like(MdmTerminalChannel::getTerminalChannelName,name)
        .eq(MdmTerminalChannel::getTenantCode,tenantCode)
        .select(MdmTerminalChannel::getTerminalChannelCode,MdmTerminalChannel::getRuleCode)
        .eq(MdmTerminalChannel::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
        .list();
  }

  /**
   * 查询懒加载数据
   *
   * @param enableStatus 启用状态
   * @param topOnly 传true只查第一层
   * @param parentCode 只查询该编码下一级
   * @param codeList 只查询这些编码
   * @param ruleCodeList 只查询这些降维编码
   * @param excludeRuleCode 排除这个降维编码的下级
   * @param tenantCode
   * @return
   */
  public List<LazyTreeVo> findOrgLazyTreeList(
      String enableStatus,
      Boolean topOnly,
      String parentCode,
      List<String> codeList,
      List<String> ruleCodeList,
      String excludeRuleCode,
      String tenantCode)  {
    return this.baseMapper.findOrgLazyTreeList(
        enableStatus,
        topOnly,
        parentCode,
        codeList,
        ruleCodeList,
        excludeRuleCode,
        tenantCode,
        DelFlagStatusEnum.NORMAL);
  }

  /**
   * 重构查询方法
   * @param id
   * @param tenantCode
   * @return
   */
  public MdmTerminalChannel findByIdAndTenantCode(String id, String tenantCode) {
    return  this.lambdaQuery()
        .eq(MdmTerminalChannel::getTenantCode,tenantCode)
        .in(MdmTerminalChannel::getId,id)
        .one();
  }

  /**
   * 重构修改方法
   * @param curAndChildrenByRuleCodes
   * @param tenantCode
   */
  public void updateBatchByIdAndTenantCode(List<MdmTerminalChannel> curAndChildrenByRuleCodes, String tenantCode) {
    curAndChildrenByRuleCodes.stream().forEach(mdmTerminalChannel -> {
      LambdaUpdateWrapper<MdmTerminalChannel>lambdaUpdateWrapper = Wrappers.lambdaUpdate();
      lambdaUpdateWrapper.eq(MdmTerminalChannel::getTenantCode,tenantCode);
      lambdaUpdateWrapper.in(MdmTerminalChannel::getId,mdmTerminalChannel.getId());
      this.baseMapper.update(mdmTerminalChannel,lambdaUpdateWrapper);
    });

  }
  /**
   * 终端渠道下拉
   *
   * @param dto 终端渠道查询参数
   */
  public List<CommonSelectVo> findTerminalChannelSelectList(TerminalChannelSelectDto dto) {
    if (StringUtils.isNotEmpty(dto.getSelectedCode())) {
      List<String> selectCodeList = Optional.ofNullable(dto.getSelectedCodeList()).orElse(Lists.newArrayList());
      selectCodeList.add(dto.getSelectedCode());
      dto.setSelectedCodeList(selectCodeList);
    }
    return mdmTerminalChannelMapper.findTerminalChannelSelectList(dto, CommonSelectConstant.COMMON_SELECT_DEFAULT_SIZE);
  }

    /**
     *
     * @author huojia
     * @date 2022/12/20 11:25
     * @param codes
     * @return java.util.List<com.biz.crm.mdm.business.terminal.channel.local.entity.MdmTerminalChannel>
     **/
    public List<MdmTerminalChannel> listByCodes(List<String> codes) {
      if (CollectionUtils.isEmpty(codes)) {
        return null;
      }
      return this.lambdaQuery()
              .in(MdmTerminalChannel::getTerminalChannelCode, codes)
              .eq(UuidFlagOpEntity::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
              .list();
    }

    /**
     * 编码查询名称
     * @param codes
     * @return {@link Map}<{@link String}, {@link String}>
     */
    public Map<String, String> findNameByCodes(List<String> codes) {
      Assert.notEmpty(codes, "参数编码不能为空");
      String tenantCode = TenantUtils.getTenantCode();
      Map<String, String> resultMap = Maps.newHashMap();
      List<MdmTerminalChannel> mdmTerminalChannels = this.mdmTerminalChannelMapper.findNameByCodes(codes, tenantCode);
      if(CollectionUtils.isEmpty(mdmTerminalChannels)){
        return resultMap;
      }
      mdmTerminalChannels.forEach(o -> {
        resultMap.put(o.getTerminalChannelCode(), o.getTerminalChannelName());
      });
      return resultMap;
    }
}

