package com.biz.crm.mdm.business.terminal.channel.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.dto.TreeDto;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.TreeRuleCodeStrategy;
import com.biz.crm.business.common.sdk.service.TreeRuleCodeStrategyHolder;
import com.biz.crm.mdm.business.terminal.channel.local.dto.TerminalChannelSelectDto;
import com.biz.crm.mdm.business.terminal.channel.local.entity.MdmTerminalChannel;
import com.biz.crm.mdm.business.terminal.channel.local.repository.MdmTerminalChannelRepository;
import com.biz.crm.mdm.business.terminal.channel.local.service.MdmTerminalChannelService;
import com.biz.crm.mdm.business.terminal.channel.sdk.contant.TerminalChannelConstant;
import com.biz.crm.mdm.business.terminal.channel.sdk.dto.TerminalChannelDto;
import com.biz.crm.mn.common.base.vo.CommonSelectVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.event.sdk.service.NebulaNetEventClient;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 终端渠道主表(MdmTerminalChannel)表服务实现类
 *
 * @author HanJiaJun
 * @since 2022-11-04 14:33:37
 */
@Service("mdmTerminalChannelService")
public class MdmTerminalChannelServiceImpl implements MdmTerminalChannelService {

  @Autowired
  private MdmTerminalChannelRepository mdmTerminalChannelRepository;

  @Autowired
  private GenerateCodeService generateCodeService;

  @Autowired(required = false)
  private TreeRuleCodeStrategyHolder treeRuleCodeStrategyHolder;

  @Autowired(required = false)
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private NebulaNetEventClient nebulaNetEventClient;

  /**
   * 分页查询数据
   *
   * @param pageable           分页对象
   * @param mdmTerminalChannel 实体对象
   * @return
   */
  @Override
  public Page<MdmTerminalChannel> findByConditions(Pageable pageable, MdmTerminalChannel mdmTerminalChannel) {
    ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
    if (Objects.isNull(mdmTerminalChannel)) {
      mdmTerminalChannel = new MdmTerminalChannel();
    }
    return this.mdmTerminalChannelRepository.findByConditions(pageable, mdmTerminalChannel);
  }

  /**
   * 下拉选择
   *
   * @param dto
   * @return
   */
  @Override
  public List<MdmTerminalChannel> findSelectByConditions(TerminalChannelSelectDto dto) {
    return this.mdmTerminalChannelRepository.findSelectByConditions(dto);
  }

  /**
   * 通过主键查询单条数据
   *
   * @param id 主键
   * @return 单条数据
   */
  @Override
  public MdmTerminalChannel findById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    //重构查询方法
    return this.mdmTerminalChannelRepository.findByIdAndTenantCode(id,TenantUtils.getTenantCode());
  }

  /**
   * 新增数据
   *
   * @param mdmTerminalChannel 实体对象
   * @return 新增结果
   */
  @Transactional
  @Override
  public MdmTerminalChannel create(MdmTerminalChannel mdmTerminalChannel) {
    this.createValidate(mdmTerminalChannel);
    MdmTerminalChannel form = this.createForm(mdmTerminalChannel);
    //新增租户编号
    form.setTenantCode(TenantUtils.getTenantCode());
    this.mdmTerminalChannelRepository.save(form);
    return mdmTerminalChannel;
  }

  /**
   * 新增补充数据
   *
   * @param entity
   */
  private MdmTerminalChannel createForm(MdmTerminalChannel entity) {
    //自动生成编码
    if (StringUtils.isBlank(entity.getTerminalChannelCode())) {
      String terminalChannelCode = this.generateCodeService.generateCode(TerminalChannelConstant.TERMINAL_CODE, 1).get(0);
      entity.setTerminalChannelCode(terminalChannelCode);
    }
    //验重
    MdmTerminalChannel exits = this.findByCode(entity.getTerminalChannelCode());
    Validate.isTrue(Objects.isNull(exits), "编码重复，包括已逻辑删除的数据");
    //判断层级
    int levelNum = 1;
    if (StringUtils.isNotBlank(entity.getParentChannelCode())) {
      //父节点存在则层级加1
      MdmTerminalChannel parent = this.findByCode(entity.getParentChannelCode());
      Validate.notNull(parent, "父节点不存在");
      levelNum = parent.getLevelNum() + 1;
    }
    entity.setParentChannelCode(Optional.ofNullable(entity.getParentChannelCode()).orElse(StringUtils.EMPTY));
    String parentChannelCode = entity.getParentChannelCode();
    String ruleCode = this.getRuleCodeByParentCode(parentChannelCode);
    entity.setRuleCode(ruleCode);
    entity.setLevelNum(levelNum);
    return entity;
  }

  /**
   * 产生规则编码
   *
   * @param parentChannelCode
   * @return
   */
  private String getRuleCodeByParentCode(String parentChannelCode) {
    String parentRuleCode = null;
    if (StringUtils.isNotEmpty(parentChannelCode)) {
      MdmTerminalChannel parent = this.findByCode(parentChannelCode);
      Validate.notNull(parent, "上级渠道不存在");
      parentRuleCode = parent.getRuleCode();
    }
    List<MdmTerminalChannel> childrenListByParentCode =
        this.findChildrenListByParentCode(parentChannelCode);
    List<TreeDto> childrenDto =
        Lists.newArrayList(
            nebulaToolkitService.copyCollectionByWhiteList(
                childrenListByParentCode,
                MdmTerminalChannel.class,
                TreeDto.class,
                HashSet.class,
                ArrayList.class));
    Validate.notNull(this.treeRuleCodeStrategyHolder, "系统未配置降维码策略控制器");
    TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
    Validate.notNull(treeRuleCodeStrategy, "系统未配置降维码策略");
    return treeRuleCodeStrategy.generate(
        TerminalChannelConstant.RULE_CODE_LENGTH, parentRuleCode, childrenDto);
  }

  /**
   * 修改新据
   *
   * @param mdmTerminalChannel 实体对象
   * @return 修改结果
   */
  @Transactional
  @Override
  public MdmTerminalChannel update(MdmTerminalChannel mdmTerminalChannel) {
    this.updateValidate(mdmTerminalChannel);
    this.updateForm(mdmTerminalChannel);

    this.mdmTerminalChannelRepository.saveOrUpdate(mdmTerminalChannel);
    return mdmTerminalChannel;
  }

  /**
   * 更新补充数据
   *
   * @param mdmTerminalChannel
   */
  private void updateForm(MdmTerminalChannel mdmTerminalChannel) {
    /*
    1. 基础校验
    2. 上级校验
    3. 保存基本信息
    4. 重置自身及子级降维编码
     */
    // 1.
    String currentId = mdmTerminalChannel.getId();
    //重构查询方法
    MdmTerminalChannel current = this.mdmTerminalChannelRepository.findByIdAndTenantCode(currentId,TenantUtils.getTenantCode());
    Validate.notNull(current, "不存在当前修改对象");
    int levelNum = 1;
    Validate.isTrue(mdmTerminalChannel.getTerminalChannelCode().equals(current.getTerminalChannelCode()), "渠道编码不能修改");
    String oldRuleCode = current.getRuleCode();
    //2.父级是否变动
    boolean parentChange = StringUtils.isNotBlank(mdmTerminalChannel.getParentChannelCode()) || StringUtils.isNotBlank(current.getParentChannelCode())
        && !Objects.equals(mdmTerminalChannel.getParentChannelCode(), current.getParentChannelCode());
    MdmTerminalChannel parent = null;
    //只有父级变动的时候才校验
    if (StringUtils.isNotBlank(mdmTerminalChannel.getParentChannelCode()) && parentChange) {
      Validate.isTrue(!mdmTerminalChannel.getParentChannelCode().equals(current.getTerminalChannelCode()), "上级渠道不能为当前渠道");
      parent = this.mdmTerminalChannelRepository.findByCode(mdmTerminalChannel.getParentChannelCode());
      Validate.notNull(parent, "上级渠道不存在");
      Validate.notBlank(parent.getRuleCode(), "上级渠道规则编码不能为空，请先重置降维编码进行修正");
      Validate.isTrue(!parent.getRuleCode().startsWith(oldRuleCode), "上级渠道不能是当前的下级渠道");
      levelNum = parent.getLevelNum() + 1;
    }
    //新增租户编号
    mdmTerminalChannel.setTenantCode(TenantUtils.getTenantCode());
    //3.更新基本信息
    this.mdmTerminalChannelRepository.saveOrUpdate(mdmTerminalChannel);
    //4.更新当前及下级降维编码
    if (parentChange) {
      String ruleCodeByParentCode = this.getRuleCodeByParentCode(mdmTerminalChannel.getParentChannelCode());
      updateRuleCodeAllChildren(mdmTerminalChannel.getTerminalChannelCode(), ruleCodeByParentCode, levelNum);
    }
  }

  /**
   * 更新自身及下级降维编码
   *
   * @param terminalChannelCode
   * @param ruleCodeByParentCode
   * @param levelNum
   */
  private void updateRuleCodeAllChildren(String terminalChannelCode, String ruleCodeByParentCode, int levelNum) {
    //更新自身
    this.mdmTerminalChannelRepository.updateRuleCodeAndLevelNumByChannelCode(terminalChannelCode, ruleCodeByParentCode, levelNum);
    //查找下级
    List<MdmTerminalChannel> childrenList = this.mdmTerminalChannelRepository.findChildrenListByParentCode(terminalChannelCode);
    if (CollectionUtils.isEmpty(childrenList)) {
      //无下级，递归结束
      return;
    }
    //遍历下级
    for (int i = 0; i < childrenList.size(); i++) {
      //递归调用
      MdmTerminalChannel children = childrenList.get(i);
      Validate.notNull(this.treeRuleCodeStrategyHolder, "系统未配置降为编码策略控制器");
      TreeRuleCodeStrategy strategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
      Validate.notNull(strategy, "系统未配置降为编码策略");
      //通过策略器拼接新的ruleCode
      String ruleCode = strategy.generateByNum(TerminalChannelConstant.RULE_CODE_LENGTH, i + 1);
      ruleCode = ruleCodeByParentCode + ruleCode;
      updateRuleCodeAllChildren(children.getTerminalChannelCode(), ruleCode, levelNum + 1);
    }
  }

  /**
   * 删除数据
   *
   * @param idList 主键结合
   * @return 删除结果
   */
  @Transactional
  @Override
  public void delete(List<String> idList) {
    Validate.isTrue(!CollectionUtils.isEmpty(idList), "删除数据时，主键集合不能为空！");
    List<MdmTerminalChannel> list = this.findByIds(idList);
    Validate.isTrue(list.size() == idList.size(), "删除个数与实际个数不匹配");
    Map<String, MdmTerminalChannel> parent = list.stream().collect(Collectors.toMap(MdmTerminalChannel::getTerminalChannelCode, t -> t));
    List<MdmTerminalChannel> parentList = this.mdmTerminalChannelRepository.findChildrenListByParentCodes(parent.keySet());
    Validate.isTrue(CollectionUtils.isEmpty(parentList), "当前ID集合存在绑定了下级渠道，请检查");
    this.mdmTerminalChannelRepository.updateDelFlagByids(idList);
  }

  /**
   * 通过编码查找
   *
   * @param terminalChannelCode
   * @return
   */
  @Override
  public MdmTerminalChannel findByCode(String terminalChannelCode) {
    if (StringUtils.isBlank(terminalChannelCode)) {
      return null;
    }
    return this.mdmTerminalChannelRepository.findByCode(terminalChannelCode);
  }

  @Override
  public List<MdmTerminalChannel> findChildrenListByParentCode(String parentChannelCode) {
    if (StringUtils.isBlank(parentChannelCode)) {
      return new ArrayList<>(0);
    }
    return this.mdmTerminalChannelRepository.findChildrenListByParentCode(parentChannelCode);
  }

  /**
   * 通过ID集合查询
   *
   * @param idList
   * @return
   */
  @Override
  public List<MdmTerminalChannel> findByIds(List<String> idList) {
    if (CollectionUtils.isEmpty(idList)) {
      return new ArrayList<>(0);
    }
    return this.mdmTerminalChannelRepository.findByIds(idList);
  }

  /**
   * 批量启用
   * @param idList
   */
  @Override
  public void enableBatch(List<String> idList) {
    /*
    启用时，需要校验其上级渠道是否都启用了
     */
    Validate.notEmpty(idList, "启用时，ID集合不能为空！");
    List<MdmTerminalChannel> list = this.mdmTerminalChannelRepository.findByIds(idList);
    Validate.isTrue(!CollectionUtils.isEmpty(list) && list.size() == idList.size(), "数据启用个数不匹配");
    //检查上级职位是否被禁用
    //获取规则编码
    List<String> ruleCodes = list.stream().map(MdmTerminalChannel::getRuleCode).filter(StringUtils::isNotBlank).collect(Collectors.toList());
    Validate.notNull(this.treeRuleCodeStrategyHolder, "系统未配置降为编码策略控制器");
    TreeRuleCodeStrategy strategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
    Validate.notNull(strategy, "系统未配置降为编码策略");
    //获取全部上级编码
    Set<String> parentRuleCode = strategy.findParentRuleCodeByRuleCodesExcludeAnySelf(TerminalChannelConstant.RULE_CODE_LENGTH, ruleCodes);
    if (!CollectionUtils.isEmpty(parentRuleCode)) {
      List<MdmTerminalChannel> parenList = this.mdmTerminalChannelRepository.findbyRuleCodesAndEnableStatus(parentRuleCode, EnableStatusEnum.DISABLE.getCode());
      Validate.isTrue(CollectionUtils.isEmpty(parenList), "当前启用的渠道存在未启用的上级渠道");
    }
    this.mdmTerminalChannelRepository.updateEnableStatusByIds(idList, EnableStatusEnum.ENABLE.getCode());
  }


  /**
   * 批量禁用
   * @param idList
   */
  @Override
  public void disableBatch(List<String> idList) {
    Validate.notEmpty(idList, "禁用时，ID集合不能为空！");
    /*
    禁用时会将其下级渠道全部禁用
     */
    List<MdmTerminalChannel> list = this.mdmTerminalChannelRepository.findByIds(idList);
    Validate.isTrue(!CollectionUtils.isEmpty(idList) && list.size()==idList.size(),"数据禁用个数不匹配");
    List<String> ruleCodes = list.stream().map(MdmTerminalChannel::getRuleCode).filter(StringUtils::isNotBlank).collect(Collectors.toList());
    //获取当前渠道和下级渠道
    List<MdmTerminalChannel> curAndChildrenByRuleCodes = this.mdmTerminalChannelRepository.findCurAndChildrenByRuleCodes(ruleCodes, EnableStatusEnum.ENABLE.getCode());
    //这个校验不该被抛出
    Validate.notEmpty(curAndChildrenByRuleCodes,"获取当前及下级渠道为空！");
    for (MdmTerminalChannel curAndChildrenByRuleCode : curAndChildrenByRuleCodes) {
      curAndChildrenByRuleCode.setEnableStatus(EnableStatusEnum.DISABLE.getCode());
    }
    //禁用   重构修改方法
    this.mdmTerminalChannelRepository.updateBatchByIdAndTenantCode(curAndChildrenByRuleCodes,TenantUtils.getTenantCode());
  }

  /**
   * 重置降维编码
   * <p>一般是在修改了规则编码长度或者是存有的规则编码出现了问题，导致业务上下级产生混乱，
   * 当前方法会将本业务的全部数据的规则编码及levelNum进行重新编码</p>
   */
  @Override
  public void updateRuleCode() {
    /*
    1. 将父级编码不为空但找不到对应的上级的数据，将其上级设为null
    2. 查找所有上级为空的数据
    3. 递归设置其ruleCode和其子级的ruleCode
     */
    //1.
    List<String> ids = this.mdmTerminalChannelRepository.getIdByParentCodeNull(TenantUtils.getTenantCode());
    if (!CollectionUtils.isEmpty(ids)){
      this.mdmTerminalChannelRepository.updateParentCodeNullById(ids);
    }
    //2.
    List<MdmTerminalChannel> parent = this.mdmTerminalChannelRepository.findByParentCodeIsNull(TenantUtils.getTenantCode());
    //3.
    for (int i = 0; i < parent.size(); i++) {
      //递归调用
      Validate.notNull(this.treeRuleCodeStrategyHolder, "系统未配置降维码策略控制器");
      TreeRuleCodeStrategy treeRuleCodeStrategy = this.treeRuleCodeStrategyHolder.getStrategy(null);
      Validate.notNull(treeRuleCodeStrategy, "系统未配置降维码策略");
      updateRuleCodeAllChildren(parent.get(i).getTerminalChannelCode(),
          treeRuleCodeStrategy.generateByNum(TerminalChannelConstant.RULE_CODE_LENGTH, i + 1),
          1);
    }

  }

  @Override
  public List<MdmTerminalChannel> findAll(TerminalChannelDto dto) {
    if (Objects.isNull(dto)){
      dto=new TerminalChannelDto();
    }
    dto.setTenantCode(TenantUtils.getTenantCode());
    return null;
  }

  /**
   * 创建验证
   *
   * @param mdmTerminalChannel
   */
  private void createValidate(MdmTerminalChannel mdmTerminalChannel) {
    Validate.notNull(mdmTerminalChannel, "新增时，对象信息不能为空！");
    mdmTerminalChannel.setId(null);
    mdmTerminalChannel.setTenantCode(TenantUtils.getTenantCode());
    mdmTerminalChannel.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    mdmTerminalChannel.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    Validate.notNull(mdmTerminalChannel.getTenantCode(), "新增数据时，租户编号不能为空！");
    Validate.notNull(mdmTerminalChannel.getTerminalChannelCode(), "新增数据时，终端渠道编码不能为空！");
    Validate.notNull(mdmTerminalChannel.getTerminalChannelName(), "新增数据时，终端渠道名称不能为空！");

  }

  /**
   * 修改验证
   *
   * @param mdmTerminalChannel
   */
  private void updateValidate(MdmTerminalChannel mdmTerminalChannel) {
    Validate.notNull(mdmTerminalChannel, "修改时，对象信息不能为空！");
    mdmTerminalChannel.setTenantCode(TenantUtils.getTenantCode());
    Validate.notBlank(mdmTerminalChannel.getId(), "修改数据时，ID不能为空！");
    Validate.notBlank(mdmTerminalChannel.getTenantCode(), "修改数据时，租户编号不能为空！");
    Validate.notBlank(mdmTerminalChannel.getTerminalChannelCode(), "修改数据时，终端渠道编码不能为空！");
    Validate.notBlank(mdmTerminalChannel.getTerminalChannelName(), "修改数据时，终端渠道名称不能为空！");

  }
  /**
   * 终端渠道下拉
   *
   * @param dto 终端渠道查询参数
   */
  @Override
  public List<CommonSelectVo> findTerminalChannelSelectList(TerminalChannelSelectDto dto) {
    return mdmTerminalChannelRepository.findTerminalChannelSelectList(dto);
  }

}

