package com.biz.crm.member.business.member.local.repository;

import cn.hutool.core.date.DateTime;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.member.business.member.sdk.vo.MemberInfoSignVo;
import com.biz.crm.member.business.member.local.entity.MemberInfoSign;
import com.biz.crm.member.business.member.local.mapper.MemberInfoSignMapper;
import com.biz.crm.member.business.member.sdk.dto.MemberInfoSignPaginationDto;
import com.biz.crm.member.business.member.sdk.vo.MemberSignCountVo;
import com.biz.crm.member.business.member.sdk.vo.MemberSignRecordVo;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

/**
 * 会员签到表(repository)
 *
 * @author zouhs
 * @date 2023-06-27 13:44:50
 */
@Component
public class MemberInfoSignRepository extends ServiceImpl<MemberInfoSignMapper, MemberInfoSign> {
  
  /**
   * 分页
   *
   * @param page
   * @param dto
   * @return
   */
  public Page<MemberInfoSignVo> findByConditions(Page<MemberInfoSignVo> page, MemberInfoSignPaginationDto dto) {
    return this.baseMapper.findByConditions(page, dto);
  }
  
  public MemberInfoSign findById(String id) {
    return this.baseMapper.selectById(id);
  }

  /**
   * 根据id集合获取详情集合
   *
   * @param ids
   * @return
   */
  public List<MemberInfoSign> findByIds(List<String> ids) {
    return this.lambdaQuery().in(MemberInfoSign::getId, ids).list();
  }

  /**
   * 根据id集合 更新组织启用/禁用状态
   *
   * @param ids
   * @param enable
   */
  public void updateEnableStatusByIds(List<String> ids, EnableStatusEnum enable) {
    this.lambdaUpdate()
        .in(MemberInfoSign::getId, ids)
        .set(MemberInfoSign::getEnableStatus, enable.getCode())
        .update();
  }

  /**
   * 逻辑删除
   *
   * @param ids
   */
  public void updateDelFlagByIds(List<String> ids) {
    this.lambdaUpdate()
        .in(MemberInfoSign::getId, ids)
        .set(MemberInfoSign::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .update();
  }

  public MemberSignCountVo signCountNum() {
    return this.baseMapper.signCountNum();
  }

  public MemberSignRecordVo querySignCountNum(MemberInfoSignPaginationDto memberInfoSignPaginationDto) {
    return this.baseMapper.querySignCountNum(memberInfoSignPaginationDto);
  }

  /**
   * 根据时间获取当前用户的本月签到信息，根据签到时间排序
   * @param dateTime
   * @param orderAsc
   * @return
   */
  public List<MemberInfoSign> listMemberInfoSign(String dateTime, String memberCode, Boolean orderAsc){
    LambdaQueryWrapper<MemberInfoSign> queryWrapper = new LambdaQueryWrapper<>();
    queryWrapper.eq(MemberInfoSign::getMemberCode, memberCode)
        .likeRight(MemberInfoSign::getSignTime, dateTime)
        .eq(MemberInfoSign::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(MemberInfoSign::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .orderByAsc(orderAsc, MemberInfoSign::getSignTime)
        .orderByDesc(! orderAsc, MemberInfoSign::getSignTime);
    List<MemberInfoSign> memberInfoSignList = this.list(queryWrapper);
    if (CollectionUtils.isEmpty(memberInfoSignList)){
      return Collections.EMPTY_LIST;
    }
    return memberInfoSignList;
  }

  /**
   * 获取用户本年度获取到的积分
   * @param memberCode
   * @return
   */
  public Integer countUserYearIntegral(String memberCode) {
    int currentYear = DateTime.of(System.currentTimeMillis()).year();
    return this.baseMapper.countUserYearIntegral(memberCode, currentYear);
  }

  /**
   * 获取指定月份的额外获取记录
   * @param monthTime
   * @return
   */
  public List<MemberInfoSign> listExtendIntegralSign(String monthTime, String memberCode) {
    LambdaQueryWrapper<MemberInfoSign> queryWrapper = new LambdaQueryWrapper<>();
    queryWrapper.eq(MemberInfoSign::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(MemberInfoSign::getMemberCode, memberCode)
        .isNotNull(MemberInfoSign::getExtendObtain)
        .isNotNull(MemberInfoSign::getContinueDay)
        .likeRight(MemberInfoSign::getSignTime, monthTime);
    return this.list(queryWrapper);
  }
}
