package com.biz.crm.member.business.member.local.repository;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.member.business.member.local.entity.Questionnaire;
import com.biz.crm.member.business.member.local.mapper.QuestionnaireMapper;
import com.biz.crm.member.business.member.sdk.dto.QuestionnairePaginationDto;
import com.biz.crm.member.business.member.sdk.enums.QuestionnaireStatusEnum;
import com.biz.crm.member.business.member.sdk.vo.QuestionnaireVo;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.springframework.stereotype.Component;

/**
 * @author 吴平
 * @version 1.0.0
 * @data 2023/6/20 14:47
 * @description 问卷管理(repository)
 */
@Component
public class QuestionnaireRepository extends ServiceImpl<QuestionnaireMapper, Questionnaire> {

  /**
   * 根据问卷编码获取问卷
   * @param code 问卷编码
   * @return
   */
  public Questionnaire getByCode(String code){
    LambdaQueryWrapper<Questionnaire> questionnaireQuery = new LambdaQueryWrapper<>();
    questionnaireQuery.eq(Questionnaire::getCode, code);
    Questionnaire questionnaire = this.getOne(questionnaireQuery);
    return questionnaire;
  }

  /**
   * 分页获取问卷信息
   * @param page
   * @param dto
   * @return
   */
  public Page<QuestionnaireVo> pageInfo(Page<QuestionnaireVo> page, QuestionnairePaginationDto dto) {
    Page<QuestionnaireVo> result = baseMapper
        .pageInfo(page, dto, DelFlagStatusEnum.NORMAL.getCode());
    return result;
  }

  /**
   * 小程序分页获取调查问卷
   * @param page
   * @param pubStatus
   * @return
   */
  public Page<QuestionnaireVo> appPageByPubStatusWithEnableStatus(Page<QuestionnaireVo> page, String pubStatus, List<String> submitQuestionnaireCodes){
    return baseMapper.appPageByPubStatusWithEnableStatus(page,
        pubStatus,
        EnableStatusEnum.ENABLE.getCode(),
        DelFlagStatusEnum.NORMAL.getCode(),
        QuestionnaireStatusEnum.NO_PUBLISH.getValue(),
        submitQuestionnaireCodes);
  }

  /**
   * 批量逻辑删除问卷信息
   * @param ids
   */
  public void logicDelete(List<String> ids) {
    this.lambdaUpdate()
        .set(Questionnaire::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .in(Questionnaire::getId, ids)
        .update();
  }

  /**
   * 修改状态 启用||禁用
   * @param ids
   * @param enable
   */
  public void modifyEnableStatus(List<String> ids, EnableStatusEnum enable) {
    this.lambdaUpdate()
        .set(Questionnaire::getEnableStatus, enable.getCode())
        .in(Questionnaire::getId, ids)
        .update();
  }

  /**
   * 根据问卷编码修改发布状态
   * @param questionnaireCode
   * @param statusEnum
   */
  public void modifyQuestionnaireStatus(String questionnaireCode, QuestionnaireStatusEnum statusEnum) {
    this.lambdaUpdate()
        .set(Questionnaire::getPubStatus, statusEnum.getValue())
        .eq(Questionnaire::getCode, questionnaireCode)
        .update();
  }

  /**
   * 根据用户账号信息查看用户
   * @param page
   * @param userAccount
   * @return
   */
  public Page<QuestionnaireVo> pageQuestionnaireByUserAccount(Page<QuestionnaireVo> page, String userAccount) {
    Page<QuestionnaireVo> result = baseMapper
        .pageQuestionnaireByUserAccount(page,
            userAccount,
            DelFlagStatusEnum.NORMAL.getCode());
    return result;
  }

  /**
   * 计算生成问卷的状态
   * @param records
   */
  public void generatePubStatus(List<QuestionnaireVo> records) {
    Date currentTime = new Date(System.currentTimeMillis());
    for (QuestionnaireVo questionnaire : records) {
      Date startTime = questionnaire.getStartTime();
      Date endTime = questionnaire.getEndTime();
      String pubStatus = questionnaire.getPubStatus();
      switch (pubStatus){
        case "no_started" :
          if (startTime.after(currentTime)){
            questionnaire.setPubStatus(QuestionnaireStatusEnum.RUNNING.getValue());
          }
          break;
        case "running" :
          if (endTime.before(currentTime)){
            questionnaire.setPubStatus(QuestionnaireStatusEnum.FINISH.getValue());
          }
          break;
      }
    }
  }

  /**
   * 根据问卷编码集合，批量修改状态
   * @param questionnaireCodes
   * @param statusEnum
   */
  public void batchModifyQuestionnairePubStatus(Collection<String> questionnaireCodes, QuestionnaireStatusEnum statusEnum){
    this.lambdaUpdate().set(Questionnaire::getPubStatus, statusEnum.getValue())
        .in(Questionnaire::getCode, questionnaireCodes)
        .update();
  }
}
