package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.identity.FacturerUserDetails;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.member.business.member.local.entity.ActivityApplyRecord;
import com.biz.crm.member.business.member.local.entity.ActivityManage;
import com.biz.crm.member.business.member.local.repository.ActivityApplyRecordRepository;
import com.biz.crm.member.business.member.local.repository.ActivityManageRepository;
import com.biz.crm.member.business.member.local.service.ActivityApplyRecordService;
import com.biz.crm.member.business.member.sdk.constants.ActivityConstants;
import com.biz.crm.member.business.member.sdk.dto.ActivityApplyRecordDto;
import com.biz.crm.member.business.member.sdk.enums.ApprovalStatusEnum;
import com.biz.crm.member.business.member.sdk.enums.CpsYesOrNoEnum;
import com.biz.crm.member.business.member.sdk.vo.ActivityApplyRecordVo;
import com.biz.crm.member.business.member.sdk.vo.ActivityManagePageVo;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBusinessMappingService;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * TODO
 *
 * @Description
 * @Author monroe
 * @Date 2023/6/19 19:12
 **/
@Service
public class ActivityApplyRecordServiceImpl implements ActivityApplyRecordService {

    @Autowired
    private ActivityApplyRecordRepository activityApplyRecordRepository;
    @Autowired(required = false)
    private ProcessBusinessService processBusinessService;
    @Autowired
    private NebulaToolkitService nebulaToolkitService;
    @Autowired
    private RedisMutexService redisMutexService;
    @Autowired
    private LoginUserService loginUserService;
    @Autowired
    private ActivityManageRepository activityManageRepository;
    @Value("${crm.business.realorder.process-key:}")
    private String defaultProcessKey;
    @Autowired(required = false)
    private ProcessBusinessMappingService processBusinessMappingService;
    @Autowired(required = false)
    private GenerateCodeService generateCodeService;

    /**
     * 报名记录编码redis key
     * 参数：租户
     */
    public static final String MMS_ACTIVITY_MANAGE_CODE_REDIS_KEY = "mms:activity:manage:code:index:%s";

    /**
     * 报名记录编码前缀
     */
    public static final String MMS_ACTIVITY_MANAGE_CODE_PREFIX = "HD";

    @Override
    public ActivityApplyRecordVo create(ActivityApplyRecordDto activityApplyRecordDto) {
        return null;

    }

    @Override
    @Transactional
    public void join(ActivityApplyRecordDto dto) {
      //报名基本校验及编码生成
      String recordCode = baseValidate(dto);
      //查询报名此活动是否需要审批
      ActivityManage activity = this.activityManageRepository.lambdaQuery().eq(ActivityManage::getCode, dto.getActivityCode()).one();
      Validate.isTrue(ObjectUtils.isNotEmpty(activity)&&activity.getApplyEndDate().compareTo(new Date())>0,"活动编码或报名时间不正确！");
      boolean needApproval = activity.getNeedApproval().equals("1");
      boolean locked = false;
      try {
        if (!needApproval){
          locked = redisMutexService.tryLock(dto.getActivityCode(), TimeUnit.SECONDS, 5);
          Validate.isTrue(locked, "业务繁忙，请稍候再试。");
        }
        //校验是否超出报名最大额度
        ActivityManage act = this.activityManageRepository.lambdaQuery().eq(ActivityManage::getCode, dto.getActivityCode()).one();
        Validate.isTrue(ObjectUtils.defaultIfNull(act.getHasJoin(), 0) < act.getApplyCeilingNum(), "抱歉，报名名额已用完！");
        //报名记录表保存
        ActivityApplyRecord applyRecord = this.nebulaToolkitService
                .copyObjectByBlankList(dto, ActivityApplyRecord.class, HashSet.class, LinkedList.class);
        applyRecord.setTenantCode(TenantUtils.getTenantCode());
        applyRecord.setApplyDate(new Date());
        applyRecord.setRecordCode(recordCode);
        this.activityApplyRecordRepository.save(applyRecord);
        dto.setId(applyRecord.getId());
        //活动：1需要审批,2不需要审批
        if (needApproval) {
          applyRecord.setProcessNumber(this.commitProcess(dto));
          applyRecord.setProcessStatus(ProcessStatusEnum.COMMIT.getDictCode());
        }else {
          //更新报名人数
          act.setHasJoin(ObjectUtils.defaultIfNull(act.getHasJoin(), 0) + 1);
          this.activityManageRepository.saveOrUpdate(act);
          applyRecord.setProcessStatus(ProcessStatusEnum.PASS.getDictCode());
          applyRecord.setApprovalDate(new Date());
        }
        this.activityApplyRecordRepository.saveOrUpdate(applyRecord);
      } catch (Exception e) {
        throw new RuntimeException(e);
      } finally {
        if (locked) {
          redisMutexService.unlock(dto.getActivityCode());
        }
      }
    }

  //获取报名操作标识(防重复点击)
  @Override
  public String getId() {
    return UUID.randomUUID().toString().replace("-", "");
  }

    @Override
    public void sign(ActivityApplyRecordDto dto) {
      Validate.isTrue(StringUtils.isNoneEmpty(dto.getActivityCode(),dto.getUserName(),"签到时，活动编码以及用户名均不能为空！"));
      ActivityManage activity = this.activityManageRepository.lambdaQuery().eq(ActivityManage::getCode, dto.getActivityCode()).one();
      Validate.isTrue(activity.getSignStartDate().compareTo(new Date())<0&&activity.getSignEndDate().compareTo(new Date())>0,"请确保当前时间在签到时间范围内！");
      this.activityApplyRecordRepository.lambdaUpdate().eq(ActivityApplyRecord::getActivityCode,dto.getActivityCode())
              .eq(ActivityApplyRecord::getUserName,dto.getUserName()).set(ActivityApplyRecord::getHasAttend, CpsYesOrNoEnum.YES.getValue())
              .set(ActivityApplyRecord::getMyAttendTime,new Date()) .update();
    }

    @Override
    public ActivityApplyRecordVo updateForm(ActivityApplyRecordDto activityApplyRecordDto) {
        return null;
    }

    @Override
    public void enableBatch(List<String> ids) {

        if(CollectionUtils.isEmpty(ids)) {
            return;
        }

        this.activityApplyRecordRepository.lambdaUpdate().set(ActivityApplyRecord::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                .in(ActivityApplyRecord::getId,ids).update();

    }

    @Override
    public ActivityApplyRecordVo findDetailById(String id) {
        if(StringUtils.isBlank(id)) {
            return null;
        }

        ActivityApplyRecord activityApplyRecord = this.activityApplyRecordRepository.getById(id);
        Validate.notNull(activityApplyRecord,"报名信息不存在！");
        ActivityApplyRecordVo activityApplyRecordVo = this.nebulaToolkitService.copyObjectByWhiteList(activityApplyRecord, ActivityApplyRecordVo.class, HashSet.class, LinkedList.class);
        return activityApplyRecordVo;
    }

    @Override
    public Page<ActivityApplyRecordVo> findByConditions(Pageable pageable, ActivityApplyRecordDto activityApplyRecordDto) {
        if (Objects.isNull(activityApplyRecordDto)) {
            activityApplyRecordDto = new ActivityApplyRecordDto();
        }
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        Page<ActivityApplyRecordVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return activityApplyRecordRepository.findByConditions(page,activityApplyRecordDto);    }

    @Override
    public void disableBatch(List<String> ids) {

        if(CollectionUtils.isEmpty(ids)) {
            return;
        }

        this.activityApplyRecordRepository.lambdaUpdate().set(ActivityApplyRecord::getEnableStatus, EnableStatusEnum.DISABLE.getCode())
                .in(ActivityApplyRecord::getId,ids).update();

    }

    @Override
    public void deleteBatch(List<String> ids) {

        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.activityApplyRecordRepository.lambdaUpdate().set(ActivityApplyRecord::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .in(ActivityApplyRecord::getId,ids).update();
    }

    @Override
    public void approvalPass(List<String> ids) {

        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.activityApplyRecordRepository.lambdaUpdate().set(ActivityApplyRecord::getDelFlag, ApprovalStatusEnum.PASS.getCode())
                .in(ActivityApplyRecord::getId,ids).update();

    }

    @Override
    public void approvalReturn(List<String> ids) {

        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.activityApplyRecordRepository.lambdaUpdate().set(ActivityApplyRecord::getDelFlag, ApprovalStatusEnum.BACK.getCode())
                .in(ActivityApplyRecord::getId,ids).update();
    }



    /**
     * 提交工作流进行审批
     * @param dto
     */
    private String commitProcess(ActivityApplyRecordDto dto) {
        ProcessBusinessDto processBusiness = Optional.ofNullable(dto.getProcessBusiness())
                .orElse(new ProcessBusinessDto());
        String processKey = processBusiness.getProcessKey();
        if (StringUtils.isBlank(processKey)) {
            processBusiness.setProcessKey(defaultProcessKey);
        }
        processBusiness.setBusinessNo(dto.getId());
        processBusiness.setBusinessFormJson(JsonUtils.obj2JsonString(dto));
        processBusiness.setBusinessCode(ActivityConstants.APPLET_ACT_WORK_FLOW);
        processBusiness.setProcessTitle("活动编码" + dto.getActivityCode() + "用户名" + dto.getUserName());
        this.processBusinessMappingService.deleteByBusinessNoAndBusinessCodeAndProcessKey(
                dto.getId(),
                ActivityConstants.APPLET_ACT_WORK_FLOW,
                processBusiness.getProcessKey());
        ProcessBusinessVo processBusinessVo = processBusinessService.processStart(processBusiness);
        return processBusinessVo.getProcessNo();
    }

    /*
    * 活动报名校验
    */
    private String baseValidate(ActivityApplyRecordDto recordDto){
      Validate.isTrue(StringUtils.isNoneBlank(recordDto.getApplyUserName(),recordDto.getSex(),recordDto.getHobby()
              ,recordDto.getActivityCode()),"姓名、性别、爱好、活动编码均不能为空！");
      Validate.isTrue(ObjectUtils.isNotEmpty(recordDto.getBirthday()),"生日不能为空！");
      FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
      recordDto.setUserName(loginDetails.getAccount());
      //查询该用户该活动历史报名记录
      List<ActivityApplyRecord> hasExist = this.activityApplyRecordRepository.lambdaQuery().eq(ActivityApplyRecord::getActivityCode, recordDto.getActivityCode())
              .eq(ActivityApplyRecord::getUserName, loginDetails.getAccount()).list();
      if (!"1".equals(recordDto.getOperateAgain())){
        //首次报名
        Validate.isTrue(CollectionUtils.isEmpty(hasExist),"请勿重复报名！");
      }else {
        //重新报名
        Validate.isTrue(hasExist.size()==1&&!ProcessStatusEnum.PASS.getDictCode().equals(hasExist.get(0).getProcessStatus())&&!ProcessStatusEnum.COMMIT.getDictCode().equals(hasExist.get(0).getProcessStatus()) ,"存在审批中或审批通过的报名记录，请勿重复报名！");
        this.activityApplyRecordRepository.lambdaUpdate().eq(ActivityApplyRecord::getRecordCode,hasExist.get(0).getRecordCode()).remove();
      }
      //报名记录编码生成
      return this.generateCodeService.generateCode(ActivityConstants.APPLET_ACT_JOIN_PRIF + "-", 1, 5, 2, TimeUnit.DAYS).get(0);
    }
}
