package com.biz.crm.member.business.member.local.service.internal;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.identity.FacturerUserDetails;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.member.business.member.local.entity.ActivityApplyConfig;
import com.biz.crm.member.business.member.local.entity.ActivityManage;
import com.biz.crm.member.business.member.local.repository.ActivityApplyConfigRepository;
import com.biz.crm.member.business.member.local.repository.ActivityApplyRecordRepository;
import com.biz.crm.member.business.member.local.repository.ActivityManageRepository;
import com.biz.crm.member.business.member.local.service.ActivityManageService;
import com.biz.crm.member.business.member.sdk.dto.ActivityApplyConfigDto;
import com.biz.crm.member.business.member.sdk.dto.ActivityManageDto;
import com.biz.crm.member.business.member.sdk.dto.ActivityManagePageDto;
import com.biz.crm.member.business.member.sdk.enums.ActivityForUserEnum;
import com.biz.crm.member.business.member.sdk.enums.ActivityStatusEnum;
import com.biz.crm.member.business.member.sdk.enums.CpsYesOrNoEnum;
import com.biz.crm.member.business.member.sdk.utils.QRBarCodeUtil;
import com.biz.crm.member.business.member.sdk.vo.ActivityApplyConfigVo;
import com.biz.crm.member.business.member.sdk.vo.ActivityManagePageVo;
import com.biz.crm.member.business.member.sdk.vo.ActivityManageVo;
import com.biz.crm.member.business.member.sdk.vo.BenefitsManageOrgVo;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.text.SimpleDateFormat;
import java.util.*;
import org.springframework.transaction.annotation.Transactional;

/**
 * 活动管理实现类
 *
 * @Description
 * @Author monroe
 * @Date 2023/6/16 14:30
 **/
@Service
public class ActivityManageServiceImpl implements ActivityManageService {

  @Autowired
  private ActivityManageRepository activityManageRepository;
  @Autowired
  private ActivityApplyConfigRepository activityApplyConfigRepository;

  @Autowired
  private ActivityApplyRecordRepository activityApplyRecordRepository;

  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private RedisMutexService redisMutexService;
  @Autowired
  private LoginUserService loginUserService;

  /**
   * 活动管理编码redis key 参数：租户
   */
  public static final String MMS_ACTIVITY_MANAGE_CODE_REDIS_KEY = "mms:activity:manage:code:index:%s";

  /**
   * 活动管理编码前缀
   */
  public static final String MMS_ACTIVITY_MANAGE_CODE_PREFIX = "HD";


  /**
   * 生成编码
   */
  private String generateCode(String tenantCode) {
    String redisKey = String.format(MMS_ACTIVITY_MANAGE_CODE_REDIS_KEY, tenantCode);
    String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
    SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
    return StringUtils.join(MMS_ACTIVITY_MANAGE_CODE_PREFIX, format.format(new Date()), index);
  }

  @Override
  public Page<ActivityManagePageVo> findByConditions(Pageable pageable,
      ActivityManagePageDto activityManageDto) {
    if (Objects.isNull(activityManageDto)) {
      activityManageDto = new ActivityManagePageDto();
    }
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<ActivityManagePageVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    //赋值签到人数
    page.getRecords().forEach(x->{
      if (StringUtils.isEmpty(x.getAttendNum()))
        x.setAttendNum("0");
    } );
    // 活动状态筛选 参考字典编码类型 mms_activity_status
    return activityManageRepository.findByConditions(page, activityManageDto);
  }

  @Override
  public Page<ActivityManagePageVo> findByConditionsByApplet(Pageable pageable, ActivityManagePageDto dto) {
    Validate.isTrue(StringUtils.isNotEmpty(dto.getActivityStatus()),"活动类型编码不能为空，字典编码类型mms_activity_status");
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<ActivityManagePageVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
    String currentAccount = loginDetails.getAccount();
    dto.setUserName(currentAccount);
    Page<ActivityManagePageVo> activities = activityManageRepository.findByConditionsByApplet(page, dto);
    //赋值活动对当前用户的状态
    List<ActivityManagePageVo> records = activities.getRecords();
    for (ActivityManagePageVo vo : records) {
      String activityStatus = vo.getActivityStatus();
      String processStatus = vo.getProcessStatus();
      if (activityStatus.equals(ActivityStatusEnum.APPLY_ING.getDes())) {
        /*报名中的活动*/
        if (ObjectUtils.defaultIfNull(vo.getHasJoin(),0)>= vo.getApplyCeilingNum()&&!ObjectUtils.defaultIfNull(vo.getUserName(),"").equals(currentAccount)) {
          //满员
          vo.setStatusForUser(ActivityForUserEnum.FULL_JOIN.getDes());
        } else if (StringUtils.isEmpty(vo.getUserName()) || !ObjectUtils.defaultIfNull(vo.getUserName(),"").equals(currentAccount)) {
          //立即报名
          vo.setStatusForUser(ActivityForUserEnum.JOIN_NOW.getDes());
        } else if (ProcessStatusEnum.REJECT.getDictCode().equals(processStatus)) {
          //再次报名
          vo.setActivityStatus("报名被驳回");
          vo.setStatusForUser(ActivityForUserEnum.JOIN_AGAIN.getDes());
        } else
          //已经报名
          vo.setStatusForUser(ActivityForUserEnum.HAS_JOIN.getDes());
      } else if (activityStatus.equals(ActivityStatusEnum.SIGN_ING.getDes())) {
        /*签到中状态的活动（报名未通过审批的活动不与展示）*/
        if (StringUtils.isEmpty(vo.getUserName()) || !ObjectUtils.defaultIfNull(vo.getUserName(),"").equals(currentAccount)) {
          //未报名
          vo.setStatusForUser(ActivityForUserEnum.NO_JOIN.getDes());
        } else if (ActivityStatusEnum.WAIT_SIGN.getDes().equals(activityStatus)) {
          //等待签到
          vo.setStatusForUser(ActivityForUserEnum.WAIT_SIGN.getDes());
        } else if ("1".equals(vo.getHasAttend())) {
          //已经签到
          vo.setStatusForUser(ActivityForUserEnum.HAS_SIGN.getDes());
        } else if (ActivityStatusEnum.SIGN_ING.getDes().equals(activityStatus)) {
          //扫码签到
          vo.setStatusForUser(ActivityForUserEnum.SCANNING_SIGN.getDes());
        }
      }else if (activityStatus.equals(ActivityStatusEnum.END.getDes())) {
        /*已结束状态的活动*/
        if (CpsYesOrNoEnum.YES.getValue().equals(vo.getHasAttend())) {
          //已参与
          vo.setStatusForUser(ActivityForUserEnum.COMPLETE.getDes());
        } else if (ProcessStatusEnum.PASS.getDictCode().equals(vo.getProcessStatus())&&!CpsYesOrNoEnum.YES.getValue().equals(vo.getHasAttend())) {
          //未签到
          vo.setStatusForUser(ActivityForUserEnum.NO_SIGN.getDes());
        } else if (StringUtils.isEmpty(vo.getUserName())) {
          //未报名
          vo.setStatusForUser(ActivityForUserEnum.NO_JOIN.getDes());
        }
      }
    }
    return activities;
  }

  @Override
  public ActivityManageVo findDetailById(String id,String fromPhone) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    FacturerUserDetails loginDetails = this.loginUserService.getLoginDetails(FacturerUserDetails.class);
    String currentAccount = loginDetails.getAccount();
    ActivityManagePageDto dto=new ActivityManagePageDto();
    dto.setId(id);
    dto.setUserName(currentAccount);
    //小程序标记
    dto.setFromPhone(fromPhone);
    ActivityManage act = this.activityManageRepository.getActById(dto);

    Validate.notNull(act, "活动不存在");
    //来自小程序的详情查询
    if ("1".equals(fromPhone)){
      //查询用户身份  TODO
      String actStatus = act.getActivityStatus();
      String processStatus = act.getProcessStatus();
      //未开始
      if (ActivityStatusEnum.NOT_START.getDes().equals(actStatus)){
        act.setStatusForUser(ActivityForUserEnum.NO_START.getDes());
      }//已结束
      if (ActivityStatusEnum.END.getDes().equals(actStatus)){
        act.setStatusForUser(ActivityForUserEnum.THE_END.getDes());
      }//名额已满
      else if (ObjectUtils.defaultIfNull(act.getHasJoin(),0)>=act.getApplyCeilingNum()&&StringUtils.isEmpty(act.getProcessStatus())){
        act.setStatusForUser(ActivityForUserEnum.FULL_JOIN.getDes());
      }//限黑金会员
//      else if (act.getApplyRequireName().equals("黑金会员")&&1!=2){
//        act.setStatusForUser(ActivityForUserEnum.ONLY_FOR_GOLD.getDes());
//      }//立即报名
      else if (ActivityStatusEnum.APPLY_ING.getDes().equals(actStatus)&&StringUtils.isEmpty(processStatus)){
        act.setStatusForUser(ActivityForUserEnum.JOIN_NOW.getDes());
      }//审批中
      else if (ProcessStatusEnum.COMMIT.getDictCode().equals(processStatus)){
        act.setStatusForUser(ActivityForUserEnum.COMMITTING.getDes());
      }//待签到
      else if ((ActivityStatusEnum.WAIT_SIGN.getDes().equals(actStatus)||ActivityStatusEnum.APPLY_ING.getDes().equals(actStatus))
              &&ProcessStatusEnum.PASS.getDictCode().equals(processStatus)){
        act.setStatusForUser(ActivityForUserEnum.WAIT_SIGN.getDes());
      }//已签到
      else if (CpsYesOrNoEnum.YES.getValue().equals(act.getHasAttend())){
        act.setStatusForUser(ActivityForUserEnum.HAS_SIGN.getDes());
      }//签到中
      else if (ActivityStatusEnum.SIGN_ING.getDes().equals(actStatus)&&ProcessStatusEnum.PASS.getDictCode().equals(processStatus)
              &&!CpsYesOrNoEnum.YES.getValue().equals(act.getHasAttend())){
        act.setStatusForUser(ActivityForUserEnum.SCANNING_SIGN.getDes());
      }//重新报名
      else if (StringUtils.isNotEmpty(processStatus)&&!ProcessStatusEnum.PASS.getDictCode().equals(processStatus)&&!ProcessStatusEnum.COMMIT.getDictCode().equals(processStatus)){
        act.setStatusForUser(ActivityForUserEnum.JOIN_AGAIN.getDes());
      }//错过报名机会
      else if ((ActivityStatusEnum.WAIT_SIGN.getDes().equals(actStatus)||ActivityStatusEnum.SIGN_ING.getDes().equals(actStatus))&&StringUtils.isEmpty(processStatus)){
        act.setStatusForUser(ActivityForUserEnum.MISS.getDes());
      }
    }
    ActivityManageVo activityManageVo = this.nebulaToolkitService
        .copyObjectByWhiteList(act, ActivityManageVo.class, HashSet.class,
            LinkedList.class);
    List<ActivityApplyConfig> list = this.activityApplyConfigRepository.lambdaQuery()
        .eq(ActivityApplyConfig::getActivityCode, act.getCode())
        .orderByAsc(ActivityApplyConfig::getSortIndex).list();
    if (CollectionUtils.isNotEmpty(list)) {
      activityManageVo.setActivityApplyConfigVo(
          (List<ActivityApplyConfigVo>) this.nebulaToolkitService
              .copyCollectionByWhiteList(list, ActivityApplyConfig.class,
                  ActivityApplyConfigVo.class, HashSet.class, ArrayList.class));
    }
    //当前时间活动状态
    if (!"1".equals(fromPhone)){
      this.currentActivityStatus(activityManageVo);
    }
    return activityManageVo;
  }

  /**
   * 计算当前活动状态
   *
   * @param activityManageVo
   */
  public void currentActivityStatus(ActivityManageVo activityManageVo) {
    Date now = new Date();
    Date applyStartDate = activityManageVo.getApplyStartDate();
    Date applyEndDate = activityManageVo.getApplyEndDate();
    Date signStartDate = activityManageVo.getSignStartDate();
    Date signEndDate = activityManageVo.getSignEndDate();

    boolean before = now.before(applyStartDate);
    boolean after = now.after(signEndDate);
    boolean applyIn = DateUtil.isIn(now, applyStartDate, applyEndDate);
    boolean signIn = DateUtil.isIn(now, signStartDate, signEndDate);
    boolean wait = DateUtil.isIn(now, applyEndDate, signStartDate);

    String activityStatus = null;
    if (after) {
      activityStatus = ActivityStatusEnum.END.getCode();
    } else if (signIn) {
      activityStatus = ActivityStatusEnum.SIGN_ING.getCode();
    } else if (wait) {
      activityStatus = ActivityStatusEnum.WAIT_SIGN.getCode();
    } else if (applyIn) {
      activityStatus = ActivityStatusEnum.APPLY_ING.getCode();
    } else {
      activityStatus = ActivityStatusEnum.NOT_START.getCode();
    }

    activityManageVo.setActivityStatus(activityStatus);

  }

  @Override
  @Transactional
  public ActivityManageVo create(ActivityManageDto activityManageDto) {
    this.createValidate(activityManageDto);
    activityManageDto.setTenantCode(TenantUtils.getTenantCode());
    activityManageDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    activityManageDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    activityManageDto.setCode(this.generateCode(TenantUtils.getTenantCode()));
    ActivityManage activityManage = this.nebulaToolkitService
        .copyObjectByWhiteList(activityManageDto, ActivityManage.class, HashSet.class,
            LinkedList.class);
    activityManage.setFileId(activityManageDto.getFileId());
    this.activityManageRepository.saveOrUpdate(activityManage);

    //报名配置
    if (CollectionUtils.isNotEmpty(activityManageDto.getActivityApplyConfigDto())) {
      for (ActivityApplyConfigDto activityApplyConfigDto : activityManageDto
          .getActivityApplyConfigDto()) {
        ActivityApplyConfig activityApplyConfig = this.nebulaToolkitService
            .copyObjectByWhiteList(activityApplyConfigDto, ActivityApplyConfig.class, HashSet.class,
                LinkedList.class);
        activityApplyConfig.setActivityCode(activityManage.getCode());
        activityApplyConfig.setTenantCode(TenantUtils.getTenantCode());
        activityApplyConfig.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        activityApplyConfig.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        this.activityApplyConfigRepository.saveOrUpdate(activityApplyConfig);
      }
    }

    return this.nebulaToolkitService
        .copyObjectByWhiteList(activityManage, ActivityManageVo.class, HashSet.class,
            LinkedList.class);
  }

  private void createValidate(ActivityManageDto activityManageDto) {
    Validate.notNull(activityManageDto, "新增参数实体不能为空");
    Validate.notNull(activityManageDto.getSignEndDate(), "签约时间不能为空");
    Validate.notNull(activityManageDto.getSignStartDate(), "签约时间不能为空");
    Validate.notNull(activityManageDto.getApplyStartDate(), "报名时间不能为空");
    Validate.notNull(activityManageDto.getApplyEndDate(), "报名时间不能为空");
    Validate.notNull(activityManageDto.getApplyCeilingNum(), "报名人数上限不能为空");
    Validate.notBlank(activityManageDto.getName(), "活动名称不能为空");
    Validate.notBlank(activityManageDto.getAddress(), "活动地址不能为空");
    Validate.notBlank(activityManageDto.getActivityDsc(), "活动说明不能为空");
    Validate.notBlank(activityManageDto.getNeedApproval(), "报名审批不能为空");
    Validate.notBlank(activityManageDto.getApplyRequire(), "等级限制不能为空");
    Validate.isTrue(activityManageDto.getSignStartDate().compareTo(activityManageDto.getApplyEndDate())>=0,"签到开始时间必须大于等于报名截止时间！");
    if (CollectionUtils.isNotEmpty(activityManageDto.getActivityApplyConfigDto())) {
      for (ActivityApplyConfigDto applyConfigDto : activityManageDto
          .getActivityApplyConfigDto()) {
        Validate.notNull(applyConfigDto.getTitle(), "标头名称不能为空");
        Validate.notBlank(applyConfigDto.getField(), "组件编码不能为空");
      }
    }

    //签到开始时间不能小于报名开始时间
    if(activityManageDto.getSignStartDate().before(activityManageDto.getApplyEndDate())) {
      throw new RuntimeException("签到开始时间不能小于报名结束时间");
    }
  }

  @Override
  @Transactional
  public ActivityManageVo updateForm(ActivityManageDto activityManageDto) {
    this.updateValidate(activityManageDto);
    ActivityManageVo activityManageVo = this.findDetailById(activityManageDto.getId(),null);
    Validate.notNull(activityManageVo, "活动配置不存在");

    activityManageDto.setTenantCode(TenantUtils.getTenantCode());

    //未开始不校验,已结束不保存
    switch (ActivityStatusEnum.codeToEnum(activityManageVo.getActivityStatus())) {
      case APPLY_ING:
      case WAIT_SIGN:
        //报名中or待签到 只保存基础信息，不能编辑报名表单信息
        this.updateBaseInfo(activityManageDto);
      case SIGN_ING:
        //签到中 不能编辑
        this.validateSignIng(activityManageDto, activityManageVo);
        this.updateBaseInfo(activityManageDto);
        break;
      case NOT_START:
        this.update(activityManageDto);
        break;
      default:
    }

    return null;
  }

  private ActivityManageVo update(ActivityManageDto activityManageDto) {
    ActivityManageVo activityManageVo = this.updateBaseInfo(activityManageDto);

    // 删除历史的
    QueryWrapper<ActivityApplyConfig> wrapper = new QueryWrapper<>();
    wrapper.eq("activity_code", activityManageDto.getCode());
    this.activityApplyConfigRepository.remove(wrapper);
    //修改报名配置
    if (CollectionUtils.isNotEmpty(activityManageDto.getActivityApplyConfigDto())) {
      for (ActivityApplyConfigDto activityApplyConfigDto : activityManageDto
          .getActivityApplyConfigDto()) {
        ActivityApplyConfig activityApplyConfig = this.nebulaToolkitService
            .copyObjectByWhiteList(activityApplyConfigDto, ActivityApplyConfig.class, HashSet.class,
                LinkedList.class);
        activityApplyConfig.setId(null);
        activityApplyConfig.setActivityCode(activityManageDto.getCode());
        activityApplyConfig.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        activityApplyConfig.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        activityApplyConfig.setTenantCode(TenantUtils.getTenantCode());
        this.activityApplyConfigRepository.saveOrUpdate(activityApplyConfig);
      }
    }
    return activityManageVo;
  }

  private void validateSignIng(ActivityManageDto activityManageDto,
      ActivityManageVo activityManageVo) {
    int compare = DateUtil
        .compare(activityManageDto.getApplyEndDate(), activityManageVo.getApplyEndDate());
    Validate.isTrue(compare == 0, "签到中,不能编辑报名时间");
    int compare1 = DateUtil
        .compare(activityManageDto.getApplyStartDate(), activityManageVo.getApplyStartDate());
    Validate.isTrue(compare1 == 0, "签到中,不能编辑报名时间");
    Validate.isTrue(
        StrUtil.equals(activityManageDto.getNeedApproval(), activityManageVo.getNeedApproval()),
        "签到中,不能编辑是否审批");
    Validate.isTrue(
        StrUtil.equals(activityManageDto.getNeedApproval(), activityManageVo.getNeedApproval()),
        "签到中,不能编辑是否审批");
    Validate.isTrue(activityManageDto.getApplyCeilingNum() == activityManageVo.getApplyCeilingNum(),
        "签到中,不能编辑报名人数上限");
  }

  private ActivityManageVo updateBaseInfo(ActivityManageDto activityManageDto) {
    ActivityManage activityManage = this.nebulaToolkitService
        .copyObjectByWhiteList(activityManageDto, ActivityManage.class, HashSet.class,
            LinkedList.class);
    this.activityManageRepository.saveOrUpdate(activityManage);
    return this.nebulaToolkitService
        .copyObjectByWhiteList(activityManage, ActivityManageVo.class, HashSet.class,
            LinkedList.class);
  }

  private void updateValidate(ActivityManageDto activityManageDto) {

    Validate.notNull(activityManageDto, "编辑参数实体不能为空");
    Validate.notNull(activityManageDto.getSignEndDate(), "签约时间不能为空");
    Validate.notNull(activityManageDto.getSignStartDate(), "签约时间不能为空");
    Validate.notNull(activityManageDto.getApplyStartDate(), "报名时间不能为空");
    Validate.notNull(activityManageDto.getApplyEndDate(), "报名时间不能为空");
    Validate.notNull(activityManageDto.getApplyCeilingNum(), "报名人数上限不能为空");
    Validate.notBlank(activityManageDto.getName(), "活动名称不能为空");
    Validate.notBlank(activityManageDto.getAddress(), "活动地址不能为空");
    Validate.notBlank(activityManageDto.getActivityDsc(), "活动说明不能为空");
    Validate.notBlank(activityManageDto.getNeedApproval(), "报名审批不能为空");
    Validate.notBlank(activityManageDto.getApplyRequire(), "等级限制不能为空");
    Validate.notBlank(activityManageDto.getId(), "编辑时，主键id不能为空");
    Validate.notBlank(activityManageDto.getCode(), "编辑时，活动编码不能为空");

    //签到开始时间不能小于报名开始时间
    if(activityManageDto.getSignStartDate().before(activityManageDto.getApplyEndDate())) {
      throw new RuntimeException("签到开始时间不能小于报名结束时间");
    }

    if (CollectionUtils.isNotEmpty(activityManageDto.getActivityApplyConfigDto())) {
      for (ActivityApplyConfigDto activityApplyConfigDto : activityManageDto
          .getActivityApplyConfigDto()) {
        Validate.notNull(activityApplyConfigDto.getTitle(), "标头名称不能为空");
        Validate.notBlank(activityApplyConfigDto.getField(), "组件编码不能为空");
      }
    }
  }

  @Override
  public void enableBatch(List<String> ids) {

    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    this.activityManageRepository.lambdaUpdate()
        .set(ActivityManage::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .in(ActivityManage::getId, ids).update();
  }

  @Override
  public void disableBatch(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    this.activityManageRepository.lambdaUpdate()
        .set(ActivityManage::getEnableStatus, EnableStatusEnum.DISABLE.getCode())
        .in(ActivityManage::getId, ids).update();
  }

  @Override
  public void deleteBatch(List<String> ids) {
    if (CollectionUtils.isEmpty(ids)) {
      return;
    }
    this.activityManageRepository.lambdaUpdate()
        .set(ActivityManage::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .in(ActivityManage::getId, ids).update();
  }

  @Override
  public void buildBarCode(String code) {
    if (StringUtils.isBlank(code)) {
      return;
    }
    QRBarCodeUtil.createCodeToFile(code, null, "活动二维码" + code);
  }

  /**
   * 获取状态为启用,且根据签到结束时间倒叙排序的第一条活动信息
   *
   * @return
   * @Author 吴平
   * @Date 2023/06/27 10:43
   */
  @Override
  public ActivityManageVo getOneByEnableOrderBySignEndTime() {
    LambdaQueryWrapper<ActivityManage> queryWrapper = new LambdaQueryWrapper<>();
    queryWrapper
        .eq(ActivityManage::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(ActivityManage::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .orderByDesc(ActivityManage::getSignEndDate)
        .last("limit 1");
    ActivityManage activityManage = this.activityManageRepository.getOne(queryWrapper);
    ActivityManageVo result = this.nebulaToolkitService
        .copyObjectByWhiteList(activityManage, ActivityManageVo.class, HashSet.class,
            LinkedList.class);
    return result;
  }

}
