package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.member.business.member.local.entity.CardProduct;
import com.biz.crm.member.business.member.local.entity.HolidayEntity;
import com.biz.crm.member.business.member.local.repository.CardProductRepository;
import com.biz.crm.member.business.member.local.service.CardProductService;
import com.biz.crm.member.business.member.sdk.dto.CardProductDto;
import com.biz.crm.member.business.member.sdk.dto.HolidayDto;
import com.biz.crm.member.business.member.sdk.vo.CardProductVo;
import com.biz.crm.member.business.member.sdk.vo.HolidayVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import javax.smartcardio.Card;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 名片产品实现类
 *
 * @Description
 * @Author monroe
 * @Date 2023/6/26 11:04
 **/
@Service
public class CardProductServiceImpl implements CardProductService {

    @Autowired
    private CardProductRepository cardProductRepository;

    @Autowired
    private NebulaToolkitService nebulaToolkitService;
    @Autowired
    private RedisMutexService redisMutexService;
    @Autowired
    private LoginUserService loginUserService;

    /**
     * 行业编码redis key
     * 参数：租户
     */
    public static final String MMS_CARD_PRODUCT_CODE_REDIS_KEY = "mms:card:product:code:index:%s";

    /**
     * 行业编码前缀
     */
    public static final String MMS_CARD_PRODUCT_CODE_PREFIX = "CP";

    /**
     * 生成编码
     */
    private String generateCode(String tenantCode) {
        String redisKey = String.format(MMS_CARD_PRODUCT_CODE_REDIS_KEY, tenantCode);
        String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        return StringUtils.join(MMS_CARD_PRODUCT_CODE_PREFIX, format.format(new Date()), index);
    }
    @Override
    public Page<CardProductVo> findByConditions(Pageable pageable, CardProductDto cardProductDto) {
        if (Objects.isNull(cardProductDto)) {
            cardProductDto = new CardProductDto();
        }
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        Page<CardProductVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.cardProductRepository.findByConditions(page, cardProductDto);
    }

    @Override
    public CardProductVo findDetailById(String id) {
        if(StringUtils.isBlank(id)) {
            return null;
        }
        CardProduct cardProduct = this.cardProductRepository.getById(id);
        Validate.notNull(cardProduct,"名片产品不存在！");
        return this.nebulaToolkitService.copyObjectByWhiteList(cardProduct,CardProductVo.class, HashSet.class,LinkedList.class);
    }

    @Override
    public CardProductVo create(CardProductDto cardProductDto) {

        Validate.notNull(cardProductDto,"新增实体不能为空");
        Validate.notBlank(cardProductDto.getName(),"产品名称不能为空");

        String code = this.generateCode(TenantUtils.getTenantCode());
        cardProductDto.setCode(code);
        cardProductDto.setTenantCode(TenantUtils.getTenantCode());
        cardProductDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        cardProductDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());

        CardProduct cardProduct = this.nebulaToolkitService.copyObjectByWhiteList(cardProductDto, CardProduct.class, HashSet.class, LinkedList.class);
        this.cardProductRepository.saveOrUpdate(cardProduct);
        return this.nebulaToolkitService.copyObjectByWhiteList(cardProduct, CardProductVo.class, HashSet.class, LinkedList.class);
    }

    @Override
    public CardProductVo update(CardProductDto cardProductDto) {
        Validate.notNull(cardProductDto,"编辑实体不能为空");
        Validate.notBlank(cardProductDto.getName(),"产品名称不能为空");

        cardProductDto.setTenantCode(TenantUtils.getTenantCode());
        cardProductDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        cardProductDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());

        CardProduct cardProduct = this.nebulaToolkitService.copyObjectByWhiteList(cardProductDto, CardProduct.class, HashSet.class, LinkedList.class);
        this.cardProductRepository.saveOrUpdate(cardProduct);
        return this.nebulaToolkitService.copyObjectByWhiteList(cardProduct, CardProductVo.class, HashSet.class, LinkedList.class);
    }

    @Override
    public void enableBatch(List<String> ids) {

        if(CollectionUtils.isEmpty(ids)) {
            return;
        }

        this.cardProductRepository.lambdaUpdate().set(CardProduct::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                .in(CardProduct::getId,ids).update();
    }

    @Override
    public void disableBatch(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)) {
            return;
        }

        this.cardProductRepository.lambdaUpdate().set(CardProduct::getEnableStatus, EnableStatusEnum.DISABLE.getCode())
                .in(CardProduct::getId,ids).update();

    }

    @Override
    public void deleteBatch(List<String> ids) {

        if(CollectionUtils.isEmpty(ids)) {
            return;
        }

        this.cardProductRepository.lambdaUpdate().set(CardProduct::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .in(CardProduct::getId,ids).update();

    }
}
