package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.member.business.member.local.entity.ChoiceNessCategory;
import com.biz.crm.member.business.member.local.repository.ChoiceNessCategoryRepository;
import com.biz.crm.member.business.member.local.service.ChoiceNessCategoryService;
import com.biz.crm.member.business.member.sdk.dto.ChoiceNessCategoryDto;
import com.biz.crm.member.business.member.sdk.dto.ChoiceNessCategoryPaginationDto;
import com.biz.crm.member.business.member.sdk.dto.InfoCategoryDto;
import com.biz.crm.member.business.member.sdk.dto.InfoCategoryPaginationDto;
import com.biz.crm.member.business.member.sdk.vo.ChoiceNessCategoryVo;
import com.biz.crm.member.business.member.sdk.vo.InfoCategoryVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author 吴平
 * @version 1.0.0
 * @data 2023/6/16 17:06
 * @description 精选分类管理
 */
@Slf4j
@Service
public class ChoiceNessCategoryServiceImpl implements ChoiceNessCategoryService {

  @Autowired
  private ChoiceNessCategoryRepository choiceNessCategoryRepository;

  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  /**
   * 新增精选分类
   * @param dto
   * @return
   */
  @Override
  public ChoiceNessCategoryVo createChoiceNessCategory(ChoiceNessCategoryDto dto) {
    this.createValidation(dto);
    ChoiceNessCategory choiceNessCategory = this.nebulaToolkitService
        .copyObjectByWhiteList(dto, ChoiceNessCategory.class, HashSet.class,
            LinkedList.class);
    choiceNessCategory.setTenantCode(TenantUtils.getTenantCode());
    choiceNessCategory.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    choiceNessCategory.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    choiceNessCategory.setCode(this.generateCodeService.generateCode("JX", 5).get(0));
    this.choiceNessCategoryRepository.save(choiceNessCategory);
    ChoiceNessCategoryVo result = this.nebulaToolkitService
        .copyObjectByWhiteList(choiceNessCategory, ChoiceNessCategoryVo.class, HashSet.class,
            LinkedList.class);
    return result;
  }

  /**
   * 修改精选分类
   * @param dto
   * @return
   */
  @Override
  public ChoiceNessCategoryVo updateChoiceNessCategory(ChoiceNessCategoryDto dto) {
    String id = dto.getId();
    Optional.ofNullable(dto.getChoiceSort()).ifPresent(sort->{
      Validate.isTrue(sort>0, "排序不能小于0");
    });
    Validate.notBlank(id, "修改精选分类id不能为空");
    ChoiceNessCategory category = this.choiceNessCategoryRepository.getById(id);
    Validate.notNull(category, "修改信息不存在");
    ChoiceNessCategory choiceNessCategory = this.nebulaToolkitService
        .copyObjectByWhiteList(dto, ChoiceNessCategory.class, HashSet.class, LinkedList.class);
    this.choiceNessCategoryRepository.saveOrUpdate(choiceNessCategory);
    ChoiceNessCategoryVo result = this.nebulaToolkitService
        .copyObjectByWhiteList(choiceNessCategory, ChoiceNessCategoryVo.class, HashSet.class,
            LinkedList.class);
    return result;
  }

  /**
   * 根据id获取单个精选分类信息
   * @param id
   * @return
   */
  @Override
  public ChoiceNessCategoryVo getChoiceNessCategoryById(String id) {
    Validate.notBlank(id, "id不能为空");
    ChoiceNessCategory category = this.choiceNessCategoryRepository.getById(id);
    Validate.notNull(category, "查询信息不存在");
    ChoiceNessCategoryVo result = this.nebulaToolkitService
        .copyObjectByWhiteList(category, ChoiceNessCategoryVo.class, HashSet.class,
            LinkedList.class);
    return result;
  }

  /**
   * 分页获取精选分类信息
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<ChoiceNessCategoryVo> pageChoiceNessCategory(Pageable pageable,
      ChoiceNessCategoryPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new ChoiceNessCategoryPaginationDto());
    Page<ChoiceNessCategoryVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<ChoiceNessCategoryVo> result = this.choiceNessCategoryRepository.pageChoiceNessCategory(page, dto);
    return result;
  }

  /**
   * 根据id集合批量修改状态为启用
   * @param ids
   */
  @Transactional
  @Override
  public void turnOnCategory(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.choiceNessCategoryRepository.batchUpdateStatus(ids, EnableStatusEnum.ENABLE);
  }

  /**
   * 根据id集合批量修改状态为禁用
   * @param ids
   */
  @Transactional
  @Override
  public void turnOffCategory(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.choiceNessCategoryRepository.batchUpdateStatus(ids, EnableStatusEnum.DISABLE);
  }

  /**
   * 根据id批量逻辑删除精选分类
   * @param ids
   */
  @Transactional
  @Override
  public void logicDeleteCategory(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.choiceNessCategoryRepository.batchLogicDeleteCategory(ids);
  }

  /**
   * 小程序端获取有效的精选分类
   * @return
   */
  @Override
  public List<ChoiceNessCategoryVo> appListEnableChoiceNessCategory() {
    // 获取当前有效的精选分类
    LambdaQueryWrapper<ChoiceNessCategory> queryWrapper = new LambdaQueryWrapper<>();
    queryWrapper.eq(ChoiceNessCategory::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(ChoiceNessCategory::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .orderByAsc(ChoiceNessCategory::getChoiceSort)
        .orderByDesc(ChoiceNessCategory::getModifyTime)
        .orderByDesc(ChoiceNessCategory::getCreateTime)
        .last("limit 3");
    List<ChoiceNessCategory> categoryList = this.choiceNessCategoryRepository.list(queryWrapper);
    Collection<ChoiceNessCategoryVo> result = this.nebulaToolkitService
        .copyCollectionByWhiteList(categoryList, ChoiceNessCategory.class,
            ChoiceNessCategoryVo.class, HashSet.class, LinkedList.class);
    return (List<ChoiceNessCategoryVo>) result;
  }

  /**
   * 新增资讯分类校验
   *
   * @param dto
   */
  private void createValidation(ChoiceNessCategoryDto dto) {
    this.validation(dto);
  }

  private void validation(ChoiceNessCategoryDto dto) {
    Validate.notBlank(dto.getChoiceNessName(), "精选分类不能为空");
    Validate.notNull(dto.getChoiceSort(), "排序不能为空");
    Validate.notBlank(dto.getCategoryPic(), "分类图片不能为空");
    Validate.isTrue(dto.getChoiceSort()>0, "序号不能小于0");
  }
}
