package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.member.business.member.local.entity.Discuss;
import com.biz.crm.member.business.member.local.entity.MemberInfo;
import com.biz.crm.member.business.member.local.entity.MemberInfoAssets;
import com.biz.crm.member.business.member.local.entity.TastingConsultant;
import com.biz.crm.member.business.member.local.helper.UserSearchHelper;
import com.biz.crm.member.business.member.local.repository.DiscussRepository;
import com.biz.crm.member.business.member.local.repository.MemberInfoAssetsRepository;
import com.biz.crm.member.business.member.local.repository.MemberInfoRepository;
import com.biz.crm.member.business.member.local.repository.TastingConsultantRepository;
import com.biz.crm.member.business.member.local.service.DiscussService;
import com.biz.crm.member.business.member.sdk.dto.DiscussDto;
import com.biz.crm.member.business.member.sdk.dto.DiscussPaginationDto;
import com.biz.crm.member.business.member.sdk.enums.DiscussCheckStatusEnum;
import com.biz.crm.member.business.member.sdk.vo.DiscussVo;
import com.biz.crm.member.business.member.sdk.vo.InfoCountDto;
import com.biz.crm.member.business.member.sdk.vo.login.MemberUserDetails;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author 吴平
 * @version 1.0.0
 * @data 2023/6/19 14:08
 * @description 评论管理
 */
@Service
@Slf4j
public class DiscussServiceImpl implements DiscussService {

  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private DiscussRepository discussRepository;

  @Autowired
  private GenerateCodeService generateCodeService;

  @Autowired
  private UserSearchHelper userSearchHelper;

  @Autowired
  private MemberInfoRepository memberInfoRepository;

  @Autowired
  private MemberInfoAssetsRepository memberInfoAssetsRepository;

  @Autowired
  private TastingConsultantRepository tastingConsultantRepository;

  /**
   * 新增一个评论
   * @param dto
   * @return
   */
  @Override
  public DiscussVo createDiscuss(DiscussDto dto) {
    Boolean enableDiscuss = this.enableDiscuss();
    Validate.isTrue(enableDiscuss, "当前用户权限不足，不能评论");
    String memberPhone = userSearchHelper.getMemberLogin().getMemberPhone();
    String memberCode = userSearchHelper.getMemberLogin().getMemberCode();
    dto.setDiscussTel(memberPhone);

    this.createValidate(dto);
    Discuss discuss = this.nebulaToolkitService
        .copyObjectByWhiteList(dto, Discuss.class, HashSet.class, LinkedList.class);
    discuss.setChoiceDiscuss(0);
    discuss.setMemberCode(memberCode);
    discuss.setCheckStatus(DiscussCheckStatusEnum.WAITING.getValue());
    discuss.setCode(this.generateCodeService.generateCode("PL", 5).get(0));
    discuss.setTenantCode(TenantUtils.getTenantCode());
    discuss.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    discuss.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.discussRepository.save(discuss);
    DiscussVo result = this.nebulaToolkitService
        .copyObjectByWhiteList(discuss, DiscussVo.class, HashSet.class, LinkedList.class);
    return result;
  }

  /**
   * 批量逻辑删除评论
   * @param ids
   */
  @Transactional
  @Override
  public void logicDeleteByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id不能为空");
    this.discussRepository.logicDeleteByIds(ids);
  }

  /**
   * 批量操作审核状态
   * @param ids 所需修改的id集合
   * @param checkStatus 审核状态编码
   */
  @Transactional
  @Override
  public void batchModifyCheckStatus(List<String> ids, String checkStatus) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id不能为空");
    Validate.notBlank(checkStatus, "审批状态不能为空");
    List<Discuss> discussList = this.discussRepository.listByIds(ids);
    if (DiscussCheckStatusEnum.APPROVE.getValue().equals(checkStatus)) {
      List<Discuss> discusses = discussList.stream().filter(discuss -> {
        String refuseValue = DiscussCheckStatusEnum.REFUSE.getValue();
        return discuss.getCheckStatus().equals(refuseValue);
      }).collect(Collectors.toList());
      Validate.isTrue(CollectionUtils.isEmpty(discusses), "非待审批状态的评论不能通过审批！");
    }
    if (DiscussCheckStatusEnum.REFUSE.getValue().equals(checkStatus)){
      List<Discuss> discusses = discussList.stream().filter(discuss -> {
        String approveValue = DiscussCheckStatusEnum.APPROVE.getValue();
        return discuss.getCheckStatus().equals(approveValue);
      }).collect(Collectors.toList());
      Validate.isTrue(CollectionUtils.isEmpty(discusses), "非待审批状态的评论不能直接驳回！");
    }
    this.discussRepository.modifyCheckStatusByIds(ids, checkStatus);
  }

  /**
   * 批量启用状态
   * @param ids
   */
  @Transactional
  @Override
  public void turnOnDiscuss(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id不能为空");
    this.discussRepository.modifyEnableStatus(ids, EnableStatusEnum.ENABLE);
  }

  /**
   * 批量禁用
   * @param ids
   */
  @Transactional
  @Override
  public void turnOffDiscuss(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id不能为空");
    this.discussRepository.modifyEnableStatus(ids, EnableStatusEnum.DISABLE);
  }

  /**
   * 批量精选评论
   * @param ids
   */
  @Transactional
  @Override
  public void batchModifyChoiceDiscuss(List<String> ids, Integer choiceDiscuss) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id不能为空");
    this.discussRepository.modifyChoiceDiscuss(ids, choiceDiscuss);
  }

  /**
   * 分页获取评论信息
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<DiscussVo> pageDiscuss(Pageable pageable, DiscussPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new DiscussPaginationDto());
    Page<DiscussVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<DiscussVo> result = this.discussRepository.pageDiscussInfo(page, dto);
    return result;
  }

  /**
   * 分组统计评论数量
   * @param infoCodes
   * @return
   */
  @Override
  public List<InfoCountDto> countByInfoCodes(List<String> infoCodes, String checkStatus, String enableStatus) {
    if (CollectionUtils.isEmpty(infoCodes)){
      return Collections.EMPTY_LIST;
    }
    List<InfoCountDto> result = this.discussRepository.countByInfoCodes(infoCodes, checkStatus, enableStatus);
    return result;
  }

  /**
   * 小程序分页获取资讯评论信息
   * @param pageable
   * @param infoCode
   * @return
   */
  @Override
  public Page<DiscussVo> appPageDiscussByInfoCode(Pageable pageable, String infoCode) {
    Validate.notBlank(infoCode, "资讯编码不能为空");
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    Page<DiscussVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    String memberCode = this.userSearchHelper.getMemberLogin().getMemberCode();
    Page<DiscussVo> result = this.discussRepository.appPageDiscussByInfoCode(page, infoCode, memberCode);
//    this.populateBelongOwner(result.getRecords());
    return result;
  }

  /**
   * 判断当前用户是否可以评论
   * @return
   */
  @Override
  public Boolean enableDiscuss() {
    MemberUserDetails memberLogin = this.userSearchHelper.getMemberLogin();
    String memberCode = memberLogin.getMemberCode();
    String memberPhone = memberLogin.getMemberPhone();
    /**
     * 有等级信息或者是品鉴顾问
     * */
    MemberInfoAssets memberInfoAssets = memberInfoAssetsRepository.lambdaQuery()
        .eq(MemberInfoAssets::getMemberCode, memberCode)
        .eq(MemberInfoAssets::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
        .one();
    TastingConsultant tastingConsultant = tastingConsultantRepository.lambdaQuery()
        .eq(TastingConsultant::getPhone, memberPhone)
        .eq(TastingConsultant::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .last("limit 1")
        .orderByDesc(TastingConsultant::getCreateTime)
        .one();
    return ObjectUtils.isNotEmpty(memberCode) || ObjectUtils.isNotEmpty(tastingConsultant);
  }

  /**
   * 小程序获取属于自己的前3条已通过审批的评论
   * @param infoCode
   * @return
   */
  @Override
  public List<DiscussVo> appListOwnerDiscussLimit(String infoCode) {
    String memberCode = this.userSearchHelper.getMemberLogin().getMemberCode();
    List<Discuss> discussList = this.discussRepository.lambdaQuery()
        .eq(Discuss::getMemberCode, memberCode)
        .eq(Discuss::getInfoCode, infoCode)
        .eq(Discuss::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(Discuss::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(Discuss::getCheckStatus, DiscussCheckStatusEnum.APPROVE.getValue())
        .orderByDesc(Discuss::getCreateTime)
        .last("limit 3")
        .list();
    if (CollectionUtils.isEmpty(discussList)) {
      discussList = Collections.EMPTY_LIST;
    }
    List<String> memberCodes = discussList.stream().map(Discuss::getMemberCode).distinct()
        .collect(Collectors.toList());
    Collection<DiscussVo> result = this.nebulaToolkitService
        .copyCollectionByWhiteList(discussList, Discuss.class, DiscussVo.class, HashSet.class,
            LinkedList.class);
    // 填充头像信息
    if (CollectionUtils.isNotEmpty(memberCodes)) {
      Map<String, String> memberInfoMap = this.memberInfoRepository.lambdaQuery()
          .in(MemberInfo::getMemberCode, memberCodes)
          .list()
          .stream()
          .collect(Collectors.toMap(MemberInfo::getMemberCode, MemberInfo::getMemberAvatar));
      result.forEach(discussVo -> discussVo.setAvatar(memberInfoMap.get(discussVo.getMemberCode())));
    }
    return (List<DiscussVo>) result;
  }

  /**
   * 填充owner属性
   * @param records
   */
  private void populateBelongOwner(List<DiscussVo> records) {
    String memberCode = this.userSearchHelper.getMemberLogin().getMemberCode();
    for (DiscussVo record : records) {
      String discussMemberCode = record.getMemberCode();
      record.setOwner(memberCode.equals(discussMemberCode));
    }
  }

  /**
   * 新增校验
   * @param dto
   */
  private void createValidate(DiscussDto dto){
    this.validate(dto);
  }

  /**
   * 校验
   * @param dto
   */
  private void validate(DiscussDto dto){
    Validate.notBlank(dto.getInfoCode(), "资讯编码不能为空");
    Validate.notBlank(dto.getInfoTitle(), "资讯标题不能为空");
    Validate.notBlank(dto.getDiscussContent(), "评论内容不能为空");
  }


}
