package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.mdm.business.org.sdk.service.OrgVoService;
import com.biz.crm.mdm.business.org.sdk.vo.OrgVo;
import com.biz.crm.member.business.member.local.entity.DummyOrder;
import com.biz.crm.member.business.member.local.entity.VerificationUser;
import com.biz.crm.member.business.member.local.helper.UserSearchHelper;
import com.biz.crm.member.business.member.local.repository.DummyOrderRepository;
import com.biz.crm.member.business.member.local.repository.VerificationUserRepository;
import com.biz.crm.member.business.member.local.service.DummyOrderService;
import com.biz.crm.member.business.member.local.service.DummyOrderUserService;
import com.biz.crm.member.business.member.local.service.MemberInfoArchivesService;
import com.biz.crm.member.business.member.local.service.MemberInfoService;
import com.biz.crm.member.business.member.sdk.constants.ProcessConstant;
import com.biz.crm.member.business.member.sdk.dto.AppOrderPaginationDto;
import com.biz.crm.member.business.member.sdk.dto.DummyOrderPaginationDto;
import com.biz.crm.member.business.member.sdk.enums.AppDummyOrderStatusEnum;
import com.biz.crm.member.business.member.sdk.enums.DummyOrderStatusEnum;
import com.biz.crm.member.business.member.sdk.enums.OrderSettleStatusEnum;
import com.biz.crm.member.business.member.sdk.enums.VerificationUserTypeEnum;
import com.biz.crm.member.business.member.sdk.vo.DummyOrderUserVo;
import com.biz.crm.member.business.member.sdk.vo.DummyOrderVo;
import com.biz.crm.member.business.member.sdk.vo.MemberInfoArchivesVo;
import com.biz.crm.member.business.member.sdk.vo.MemberInfoVo;
import com.biz.crm.member.business.member.sdk.vo.login.MemberUserDetails;
import com.biz.crm.workflow.sdk.dto.ProcessBusinessDto;
import com.biz.crm.workflow.sdk.enums.ProcessStatusEnum;
import com.biz.crm.workflow.sdk.service.ProcessBusinessMappingService;
import com.biz.crm.workflow.sdk.service.ProcessBusinessService;
import com.biz.crm.workflow.sdk.vo.ProcessBusinessVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.JsonUtils;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Sets;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 虚拟订单(DummyOrder)表服务实现类
 *
 * @author zouhs
 * @date 2023-06-14 15:01:00
 */
@Slf4j
@Service("dummyOrderService")
public class DummyOrderServiceImpl implements DummyOrderService {
  
  @Autowired(required = false)
  private DummyOrderRepository dummyOrderRepository;

  @Autowired(required = false)
  private DummyOrderUserService dummyOrderUserService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private UserSearchHelper userSearchHelper;

  @Autowired(required = false)
  private OrgVoService orgVoService;

  @Autowired
  private MemberInfoArchivesService memberInfoArchivesService;

  @Autowired
  private MemberInfoService memberInfoService;

  @Autowired
  private VerificationUserRepository verificationUserRepository;

  @Override
  public Page<DummyOrderVo> findByConditions(Pageable pageable, DummyOrderPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new DummyOrderPaginationDto());
    Page<DummyOrderVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.dummyOrderRepository.findByConditions(page, dto);
  }

  @Override
  public DummyOrderVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    DummyOrder dummyOrder = this.dummyOrderRepository.findById(id);
    DummyOrderVo dummyOrderVo = this.nebulaToolkitService.copyObjectByWhiteList(dummyOrder, DummyOrderVo.class, HashSet.class, LinkedList.class);
    return dummyOrderVo;
  }
  
  @Override
  @Transactional
  public DummyOrderVo create(DummyOrderVo dummyOrderVo) {
    this.createValidation(dummyOrderVo);
    DummyOrder dummyOrder = this.nebulaToolkitService.copyObjectByWhiteList(dummyOrderVo, DummyOrder.class, HashSet.class, LinkedList.class);
    dummyOrder.setTenantCode(TenantUtils.getTenantCode());
    dummyOrder.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    dummyOrder.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    MemberInfoVo memberInfoVo = memberInfoService
        .findDetailByMemberCode(dummyOrder.getUserCode());
    Validate.notNull(memberInfoVo,"会员信息异常");
    MemberInfoArchivesVo memberInfoArchivesVo = memberInfoArchivesService
        .findDetailByMemberCode(dummyOrder.getUserCode());
    Validate.notNull(memberInfoArchivesVo,"会员档案信息异常");
    // 查询会员组织是否有配送权限 null和否 = 总部，是=市场配送订单
    List<OrgVo> orgVos = orgVoService.findByOrgCodes(Arrays.asList(memberInfoArchivesVo.getOrgCode()));
    if(CollectionUtils.isNotEmpty(orgVos)){
      OrgVo orgVo = orgVos.get(0);
      if(orgVo!=null){
        dummyOrder.setOrgCode(orgVo.getOrgCode());
        dummyOrder.setOrgName(orgVo.getOrgName());
      }
    }
    dummyOrder.setOrderCode(generateCodeService.generateCode("XN",5).get(0));
    dummyOrder.setUserName(memberInfoVo.getMemberName());
    dummyOrder.setOrderTime(new Date());
    dummyOrder.setOrderName(memberInfoVo.getMemberName());
    dummyOrder.setOrderPhone(memberInfoVo.getMemberPhone());
    this.dummyOrderRepository.saveOrUpdate(dummyOrder);
    for (DummyOrderUserVo dummyOrderUserVo : dummyOrderVo.getList()) {
      dummyOrderUserVo.setOrderCode(dummyOrder.getOrderCode());
    }
    // 存储报名人信息
    dummyOrderUserService.batchSave(dummyOrderVo.getList());
    dummyOrderVo.setOrderCode(dummyOrder.getOrderCode());
    dummyOrderVo.setId(dummyOrder.getId());
    return dummyOrderVo;
  }

  @Override
  @Transactional
  public DummyOrderVo update(DummyOrderVo dummyOrderVo) {
    this.updateValidation(dummyOrderVo);
    String currentId = dummyOrderVo.getId();
    DummyOrder current = dummyOrderRepository.findById(currentId);
    current = Validate.notNull(current, "修改信息不存在");
    current = this.nebulaToolkitService.copyObjectByWhiteList(dummyOrderVo, DummyOrder.class, HashSet.class, LinkedList.class);
    this.dummyOrderRepository.saveOrUpdate(current);
    return dummyOrderVo;
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.dummyOrderRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.dummyOrderRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
  }
  
  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.dummyOrderRepository.updateDelFlagByIds(ids);
  }

  @Override
  public List<DummyOrderUserVo> userInfoList(String orderCode) {
    Validate.notBlank(orderCode, "订单号不能为空");
    List<DummyOrderUserVo> list = dummyOrderUserService.userInfoList(orderCode);
    return list;
  }

  @Override
  public void batchWriteOff(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.dummyOrderRepository.updateWriteOffByIds(ids);
  }

  @Override
  @Transactional
  public void settleSubmit(DummyOrderVo dummyOrderVo) {
    Validate.notBlank(dummyOrderVo.getId(), "请选择订单");
    DummyOrder dummyOrder = this.dummyOrderRepository.findById(dummyOrderVo.getId());
    Validate.notNull(dummyOrder, "虚拟订单不存在!");
    Validate
        .isTrue(Objects.equals(OrderSettleStatusEnum.STAY_SETTLE.getValue(), dummyOrder.getSettleStatus()),
            "请选择未结算订单!");
    dummyOrder.setProcessNumber(this.commitProcess(dummyOrder));
    dummyOrder.setProcessStatus(ProcessStatusEnum.COMMIT.getDictCode());
    dummyOrder.setSettleStatus(OrderSettleStatusEnum.LOAD_SETTLE.getValue());
    this.dummyOrderRepository.updateById(dummyOrder);
  }

  /**
   * 小程序分页获取虚拟订单
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<DummyOrderVo> appPageDummyOrder(Pageable pageable, AppOrderPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new AppOrderPaginationDto());
    this.appPageValidate(dto);
    String orderStatus = dto.getOrderStatus();
    String orderStatusAll = AppDummyOrderStatusEnum.ALL.getValue();
    if (orderStatusAll.equals(orderStatus)){
      dto.setOrderStatus(StringUtils.EMPTY);
    }
    Page<DummyOrderVo> page = new Page<>(pageable.getPageNumber(),
        pageable.getPageSize());
    String memberCode = userSearchHelper.getMemberLogin().getMemberCode();
    Page<DummyOrderVo> result = this.dummyOrderRepository.appPageDummyOrder(page, dto, memberCode);
    return result;
  }

  @Override
  public DummyOrderVo findDetailByOrderCode(String orderCode) {
    if (StringUtils.isBlank(orderCode)) {
      return null;
    }
    DummyOrder dummyOrder = this.dummyOrderRepository.lambdaQuery().eq(DummyOrder::getOrderCode,orderCode).one();
    if(dummyOrder == null){
      return null;
    }
    DummyOrderVo dummyOrderVo = this.nebulaToolkitService.copyObjectByWhiteList(dummyOrder, DummyOrderVo.class, HashSet.class, LinkedList.class);
    return dummyOrderVo;
  }

  /**
   * 根据虚拟订单Id核销虚拟订单
   * @param dummyOrderId
   */
  @Override
  public void writeDummyOrderById(String dummyOrderId) {
    Validate.notBlank(dummyOrderId, "虚拟订单Id不能为空");
    DummyOrder dummyOrder = this.dummyOrderRepository.lambdaQuery()
        .eq(DummyOrder::getId, dummyOrderId)
        .eq(DummyOrder::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(DummyOrder::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .one();
    Validate.notNull(dummyOrder, "核销订单不存在");
    String orderStatus = dummyOrder.getOrderStatus();
    String stayUseValue = DummyOrderStatusEnum.STAY_USE.getValue();
    Validate.isTrue(stayUseValue.equals(orderStatus), "核销失败,订单状态不是待核销");
    String writeOffValue = DummyOrderStatusEnum.HAS_WRITE_OFF.getValue();
    MemberUserDetails memberLogin = this.userSearchHelper.getMemberLogin();
    String memberPhone = memberLogin.getMemberPhone();
    VerificationUser verificationUser = this.verificationUserRepository.lambdaQuery()
        .eq(VerificationUser::getPhone, memberPhone)
        .eq(VerificationUser::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(VerificationUser::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(VerificationUser::getUserType, VerificationUserTypeEnum.VIRTUAL.getCode())
        .one();
    Validate.notNull(verificationUser, "当前登录用户不是虚拟产品核销人员或已被禁用，无权核销该订单");
    String userName = verificationUser.getUserName();
    String userCode = verificationUser.getUserCode();
    dummyOrder.setOrderStatus(writeOffValue);
    dummyOrder.setWriteOffTime(new Date(System.currentTimeMillis()));
    dummyOrder.setWriteOffCode(userCode);
    dummyOrder.setWriteOffName(userName);
    this.dummyOrderRepository.saveOrUpdate(dummyOrder);
  }

  /**
   * 小程序分页查询参数校验
   * @param dto
   */
  private void appPageValidate(AppOrderPaginationDto dto) {
    Validate.notNull(dto, "查询参数不能为空");
    Validate.notBlank(dto.getOrderSource(), "数据来源不能为空");
    Validate.notBlank(dto.getOrderStatus(), "订单状态不能为空");
  }

  private void createValidation(DummyOrderVo dummyOrderVo) {
    this.validation(dummyOrderVo);
  }

  private void updateValidation(DummyOrderVo dummyOrderVo) {
    this.validation(dummyOrderVo);
  }

  private void validation(DummyOrderVo dummyOrderVo) {
    Validate.notBlank(dummyOrderVo.getOrderStatus(),"订单状态不能为空");
    Validate.notBlank(dummyOrderVo.getOrderSource(),"订单来源不能为空");
    Validate.notBlank(dummyOrderVo.getUserCode(),"关联会员用户不能为空");
    Validate.notBlank(dummyOrderVo.getProductImg(),"商品缩略图不能为空");
    Validate.notBlank(dummyOrderVo.getProductName(),"商品名称不能为空");
    Validate.notNull(dummyOrderVo.getProductNum(),"商品数量不能为空");
    Validate.isTrue(CollectionUtils.isNotEmpty(dummyOrderVo.getList()),"报名人信息不能为空");
  }

  @Autowired
  private ProcessBusinessService processBusinessService;

  @Autowired(required = false)
  private ProcessBusinessMappingService processBusinessMappingService;

  @Value("${crm.business.dummyorder.process-key:}")
  private String defaultProcessKey;

  private static final Set<String> STATUS_SET =
      Sets.newHashSet(
          ProcessStatusEnum.PREPARE.getDictCode(),
          ProcessStatusEnum.PASS.getDictCode(),
          ProcessStatusEnum.REJECT.getDictCode(),
          ProcessStatusEnum.RECOVER.getDictCode());

  /**
   * 经销商新增提交工作流进行审批，提交成功返回流程实例ID，提交失败则抛出异常
   *
   * @param dto 授信请求DTO
   */
  private String commitProcess(DummyOrder dto) {
    ProcessBusinessDto processBusiness = Optional.ofNullable(dto.getProcessBusiness())
        .orElse(new ProcessBusinessDto());
    String processKey = processBusiness.getProcessKey();
    if (StringUtils.isBlank(processKey)) {
      processBusiness.setProcessKey(defaultProcessKey);
    }
    processBusiness.setBusinessNo(dto.getId());
    processBusiness.setBusinessFormJson(JsonUtils.obj2JsonString(dto));
    processBusiness.setBusinessCode(ProcessConstant.MMS_DUMMY_ORDER_CREATE);
    processBusiness.setProcessTitle(dto.getOrderName()+dto.getOrderCode());
    this.processBusinessMappingService.deleteByBusinessNoAndBusinessCodeAndProcessKey(
        dto.getId(),
        ProcessConstant.MMS_DUMMY_ORDER_CREATE,
        processBusiness.getProcessKey());
    ProcessBusinessVo processBusinessVo = processBusinessService.processStart(processBusiness);
    return processBusinessVo.getProcessNo();
  }
}
