package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.member.business.member.local.entity.HolidayEntity;
import com.biz.crm.member.business.member.local.entity.HolidayIntegral;
import com.biz.crm.member.business.member.local.helper.UserSearchHelper;
import com.biz.crm.member.business.member.local.repository.HolidayIntegralRepository;
import com.biz.crm.member.business.member.local.repository.HolidayRepository;
import com.biz.crm.member.business.member.local.service.AppletPopupNotifyService;
import com.biz.crm.member.business.member.local.service.HolidayService;
import com.biz.crm.member.business.member.local.utils.AssertUtils;
import com.biz.crm.member.business.member.sdk.dto.HolidayDto;
import com.biz.crm.member.business.member.sdk.dto.HolidayIntegralDto;
import com.biz.crm.member.business.member.sdk.enums.HolidayTypeEnum;
import com.biz.crm.member.business.member.sdk.enums.TriggerConditionEnum;
import com.biz.crm.member.business.member.sdk.enums.TriggerTypeEnum;
import com.biz.crm.member.business.member.sdk.vo.HolidayIntegralVo;
import com.biz.crm.member.business.member.sdk.vo.HolidayVo;
import com.biz.crm.member.business.member.sdk.vo.applet.AppletPopupNotifyVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import java.text.ParseException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;

import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 节日实现类
 *
 * @Description
 * @Author monroe
 * @Date 2023/6/20 17:47
 **/
@Service
public class HolidayServiceImpl implements HolidayService {

    @Autowired
    private HolidayRepository holidayRepository;
    @Autowired
    private HolidayIntegralRepository holidayIntegralRepository;

    @Autowired
    private NebulaToolkitService nebulaToolkitService;
    @Autowired
    private RedisMutexService redisMutexService;
    @Autowired
    private LoginUserService loginUserService;
    @Autowired
    private AppletPopupNotifyService appletPopupNotifyService;
    @Autowired
    private UserSearchHelper userSearchHelper;

    /**
     * 行业编码redis key
     * 参数：租户
     */
    public static final String MMS_HOLIDAY_CODE_REDIS_KEY = "mms:holiday:code:index:%s";

    /**
     * 行业编码前缀
     */
    public static final String MMS_HOLIDAY_CODE_PREFIX = "JR";

    /**
     * 生成编码
     */
    private String generateCode(String tenantCode) {
        String redisKey = String.format(MMS_HOLIDAY_CODE_REDIS_KEY, tenantCode);
        String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        return StringUtils.join(MMS_HOLIDAY_CODE_PREFIX, format.format(new Date()), index);
    }

    @Override
    public Page<HolidayVo> findByConditions(Pageable pageable, HolidayDto holidayDto) {
        if (Objects.isNull(holidayDto)) {
            holidayDto = new HolidayDto();
        }
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        Page<HolidayVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.holidayRepository.findByConditions(page, holidayDto);
    }

    @Override
    public HolidayVo findDetailById(String id) {

        if(StringUtils.isBlank(id)) {
            return null;
        }

        HolidayEntity holidayEntity = this.holidayRepository.getById(id);
        Validate.notNull(holidayEntity,"节日不存在");

        HolidayVo holidayVo = this.nebulaToolkitService.copyObjectByWhiteList(holidayEntity, HolidayVo.class, HashSet.class, LinkedList.class);

        List<HolidayIntegral> integrals = this.holidayIntegralRepository.lambdaQuery().eq(HolidayIntegral::getHolidayCode, holidayVo.getCode())
                .eq(HolidayIntegral::getTenantCode, TenantUtils.getTenantCode()).list();

        if(CollectionUtils.isNotEmpty(integrals)) {
            List<HolidayIntegralVo> holidayIntegralVos = (List<HolidayIntegralVo>) this.nebulaToolkitService.copyCollectionByWhiteList(integrals, HolidayIntegral.class, HolidayIntegralVo.class, HashSet.class, LinkedList.class);
            holidayVo.setIntegrals(holidayIntegralVos);
        }

        return holidayVo;
    }

    @Override
    public HolidayVo create(HolidayDto holidayDto) {

        this.createValidate(holidayDto);
        holidayDto.setCode(this.generateCode(TenantUtils.getTenantCode()));
        holidayDto.setTenantCode(TenantUtils.getTenantCode());
        holidayDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        holidayDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());

        HolidayEntity holidayEntity = this.nebulaToolkitService.copyObjectByWhiteList(holidayDto, HolidayEntity.class, HashSet.class, LinkedList.class);
        this.holidayRepository.saveOrUpdate(holidayEntity);

        //节日积分倍数
        List<HolidayIntegralDto> integrals = holidayDto.getIntegrals();
        if(CollectionUtils.isNotEmpty(integrals)) {
            List<HolidayIntegral> holidayIntegrals = (List<HolidayIntegral>) this.nebulaToolkitService.copyCollectionByWhiteList(integrals, HolidayIntegralDto.class, HolidayIntegral.class, HashSet.class, LinkedList.class);
            for (HolidayIntegral holidayIntegral : holidayIntegrals) {
                holidayIntegral.setHolidayCode(holidayEntity.getCode());
                holidayIntegral.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                holidayIntegral.setTenantCode(TenantUtils.getTenantCode());
                holidayIntegral.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            }
            this.holidayIntegralRepository.saveBatch(holidayIntegrals);
        }

        return this.nebulaToolkitService.copyObjectByWhiteList(holidayEntity,HolidayVo.class,HashSet.class,LinkedList.class);
    }

    private void createValidate(HolidayDto holidayDto) {

        Validate.notNull(holidayDto,"新增实体不能为空");
        Validate.notBlank(holidayDto.getName(),"节日名称不能为空");
        Validate.notBlank(holidayDto.getType(),"节日类型不能为空");
        Validate.notNull(holidayDto.getExtraIntegralTop(),"积分上限不能为空");

        //节假日需要选择日期
        if(HolidayTypeEnum.FESTIVAL.getCode().equals(holidayDto.getType())) {
            Validate.notNull(holidayDto.getHolidayEndDate(),"节假日日期不能为空");
            Validate.notNull(holidayDto.getHolidayStartDate(),"节假日日期不能为空");
        }

        //启用的生日唯一校验
        if(HolidayTypeEnum.BIRTHDAY.getCode().equals(holidayDto.getType())) {
            HolidayEntity entity = this.holidayRepository.lambdaQuery().eq(HolidayEntity::getType, HolidayTypeEnum.BIRTHDAY.getCode())
                    .eq(HolidayEntity::getEnableStatus, EnableStatusEnum.ENABLE.getCode()).one();
            AssertUtils.isTrue(Objects.isNull(entity),"已经存在生日的配置,请勿重复配置");
        }


        //校验积分倍数
        List<HolidayIntegralDto> integrals = holidayDto.getIntegrals();

        if(CollectionUtils.isNotEmpty(integrals)) {
            List<String> channels = new ArrayList<>();
            for (HolidayIntegralDto integral : integrals) {
                Validate.notBlank(integral.getChannel(),"渠道不能为空");
                Validate.notNull(integral.getMultiple(),"积分倍数不能为空");
                channels.add(integral.getChannel());
            }

            HashSet<String> set = new HashSet<>();
            set.addAll(channels);

            Validate.isTrue(set.size() == channels.size(),"渠道重复");
        }


    }

    @Override
    public HolidayVo update(HolidayDto holidayDto) {
        this.updateValidate(holidayDto);

        holidayDto.setTenantCode(TenantUtils.getTenantCode());
        holidayDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        holidayDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());

        HolidayEntity oldHoliday = this.holidayRepository.getById(holidayDto.getId());
        Validate.notNull(oldHoliday,"编辑节日不存在");

        HolidayEntity holidayEntity = this.nebulaToolkitService.copyObjectByWhiteList(holidayDto, HolidayEntity.class, HashSet.class, LinkedList.class);
        this.holidayRepository.saveOrUpdate(holidayEntity);
        //删除
        QueryWrapper<HolidayIntegral> wrapper = new QueryWrapper<>();
        wrapper.eq("holiday_code",oldHoliday.getCode());
        wrapper.eq("tenant_code",oldHoliday.getTenantCode());
        this.holidayIntegralRepository.remove(wrapper);

        //节日积分倍数
        List<HolidayIntegralDto> integrals = holidayDto.getIntegrals();
        if(CollectionUtils.isNotEmpty(integrals)) {
            List<HolidayIntegral> holidayIntegrals = (List<HolidayIntegral>) this.nebulaToolkitService.copyCollectionByWhiteList(integrals, HolidayIntegralDto.class, HolidayIntegral.class, HashSet.class, LinkedList.class);
            for (HolidayIntegral holidayIntegral : holidayIntegrals) {
                holidayIntegral.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
                holidayIntegral.setTenantCode(TenantUtils.getTenantCode());
                holidayIntegral.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
                holidayIntegral.setHolidayCode(holidayEntity.getCode());
            }
            this.holidayIntegralRepository.saveBatch(holidayIntegrals);
        }
        return this.nebulaToolkitService.copyObjectByWhiteList(holidayEntity,HolidayVo.class,HashSet.class,LinkedList.class);
    }

    private void updateValidate(HolidayDto holidayDto) {
        Validate.notNull(holidayDto,"编辑实体不能为空");
        Validate.notBlank(holidayDto.getName(),"节日名称不能为空");
        Validate.notBlank(holidayDto.getType(),"节日类型不能为空");
        Validate.notNull(holidayDto.getExtraIntegralTop(),"积分上限不能为空");
        Validate.notBlank(holidayDto.getCode(),"编辑时，编码不能为空");
        Validate.notBlank(holidayDto.getId(),"编辑时，主键id不能为空");

        //节假日需要选择日期
        if(HolidayTypeEnum.FESTIVAL.getCode().equals(holidayDto.getType())) {
            Validate.notNull(holidayDto.getHolidayEndDate(),"节假日日期不能为空");
            Validate.notNull(holidayDto.getHolidayStartDate(),"节假日日期不能为空");
        }

        //生日唯一校验
        if(HolidayTypeEnum.BIRTHDAY.getCode().equals(holidayDto.getType())) {
            List<HolidayEntity> list = this.holidayRepository.lambdaQuery().ne(HolidayEntity::getCode, holidayDto.getCode())
                    .eq(HolidayEntity::getType, HolidayTypeEnum.BIRTHDAY.getCode())
                    .eq(HolidayEntity::getEnableStatus, EnableStatusEnum.ENABLE.getCode()).list();

            AssertUtils.isTrue(list.size() == 0,"已经存在生日的配置,请勿重复配置");

        }


        //校验积分倍数
        List<HolidayIntegralDto> integrals = holidayDto.getIntegrals();

        if(CollectionUtils.isNotEmpty(integrals)) {
            List<String> channels = new ArrayList<>();
            for (HolidayIntegralDto integral : integrals) {
                Validate.notBlank(integral.getChannel(),"渠道不能为空");
                Validate.notNull(integral.getMultiple(),"积分倍数不能为空");
                channels.add(integral.getChannel());
            }

            HashSet<String> set = new HashSet<>();
            set.addAll(channels);

            Validate.isTrue(set.size() == channels.size(),"渠道重复");
        }
    }

    @Override
    public void enableBatch(List<String> ids) {

        if(CollectionUtils.isEmpty(ids)) {
            return;
        }

        this.holidayRepository.lambdaUpdate().set(HolidayEntity::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                .in(HolidayEntity::getId,ids).update();

        //生日唯一
        List<HolidayEntity> list = this.holidayRepository.lambdaQuery()
                .eq(HolidayEntity::getType, HolidayTypeEnum.BIRTHDAY.getCode())
                .eq(HolidayEntity::getEnableStatus, EnableStatusEnum.ENABLE.getCode()).list();

        AssertUtils.isTrue(list.size() > 1,"不能同时启用两个以上的生日配置");

    }

    @Override
    public void disableBatch(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)) {
            return;
        }

        this.holidayRepository.lambdaUpdate().set(HolidayEntity::getEnableStatus, EnableStatusEnum.DISABLE.getCode())
                .in(HolidayEntity::getId,ids).update();
    }

    @Override
    public void deleteBatch(List<String> ids) {

        if(CollectionUtils.isEmpty(ids)) {
            return;
        }

        this.holidayRepository.lambdaUpdate().set(HolidayEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .in(HolidayEntity::getId,ids).update();
    }

    @Override
    @DynamicTaskService(cornExpression = "0 1 0 1/1 * ?", taskDesc = "每天8点55向终端推送一次合计奖励")
    public void addOnePopup() throws ParseException {
        String memberCode = userSearchHelper.getMemberLogin().getMemberCode();
        //获取当前日期
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
        Date date = new Date();
        String format = simpleDateFormat.format(date);
        Date currentDate = simpleDateFormat.parse(format);
        //查询当前日期所在的节日列表
        List<HolidayEntity> list = holidayRepository.lambdaQuery()
            .eq(HolidayEntity::getHolidayStartDate, currentDate)
            .list();
        if (CollectionUtils.isNotEmpty(list)){
            for (HolidayEntity holidayEntity : list) {
                AppletPopupNotifyVo appletPopupNotifyVo = new AppletPopupNotifyVo();
                appletPopupNotifyVo.setTriggerType(TriggerTypeEnum.HOLIDAY.getCode());
                appletPopupNotifyVo.setTriggerConditions(TriggerConditionEnum.HOLIDAY_CONF.getCode());
                appletPopupNotifyVo.setMemberCode(memberCode);
                //获取结束时间,需要带有时间的
                Date holidayEndDate = holidayEntity.getHolidayEndDate();
                // 创建 Calendar 对象并将其设置为给定日期
                Calendar calendar = Calendar.getInstance();
                calendar.setTime(holidayEndDate);
                // 设置时间为晚上 11:59:00
                calendar.set(Calendar.HOUR_OF_DAY, 23);
                calendar.set(Calendar.MINUTE, 59);
                calendar.set(Calendar.SECOND, 0);
                calendar.set(Calendar.MILLISECOND, 0);
                // 获取最终的日期和时间
                Date endTime = calendar.getTime();
                appletPopupNotifyVo.setEndTime(endTime);
                appletPopupNotifyService.addOne(appletPopupNotifyVo);
            }
        }
    }
}
