package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.member.business.member.local.entity.IndustryEntity;
import com.biz.crm.member.business.member.local.repository.IndustryRepository;
import com.biz.crm.member.business.member.local.service.IndustryService;
import com.biz.crm.member.business.member.sdk.dto.IndustryDto;
import com.biz.crm.member.business.member.sdk.dto.IndustryPageDto;
import com.biz.crm.member.business.member.sdk.vo.IndustryPageVo;
import com.biz.crm.member.business.member.sdk.vo.IndustryVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.security.sdk.login.UserIdentity;
import lombok.extern.slf4j.Slf4j;
import org.apache.catalina.security.SecurityUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 行业管理实现类
 *
 * @Description
 * @Author monroe
 * @Date 2023/6/12 16:19
 **/
@Slf4j
@Service
public class IndustryServiceImpl implements IndustryService {

    @Autowired
    private IndustryRepository industryRepository;
    @Autowired
    private NebulaToolkitService nebulaToolkitService;
    @Autowired
    private RedisMutexService redisMutexService;
    @Autowired
    private LoginUserService loginUserService;

    /**
     * 行业编码redis key
     * 参数：租户
     */
    public static final String MMS_INDUSTRY_CODE_REDIS_KEY = "mms:industry:code:index:%s";

    /**
     * 行业编码前缀
     */
    public static final String MMS_INDUSTRY_CODE_PREFIX = "HY";

    @Override
    public Page<IndustryPageVo> findByConditions(Pageable pageable, IndustryPageDto industryPageDto) {
        if (Objects.isNull(industryPageDto)) {
            industryPageDto = new IndustryPageDto();
        }
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        Page<IndustryPageVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.industryRepository.findByConditions(page, industryPageDto);
    }

    @Override
    public IndustryVo findDetailById(String id) {
        if (StringUtils.isBlank(id)) {
            return null;
        }
        IndustryEntity industryEntity = this.industryRepository.lambdaQuery().eq(IndustryEntity::getId, id).one();
        return this.nebulaToolkitService.copyObjectByWhiteList(industryEntity, IndustryVo.class, HashSet.class, LinkedList.class);
    }

    @Override
    public IndustryVo create(IndustryDto industryDto) {
        //创建参数校验
        this.createValidate(industryDto);

        IndustryEntity industryEntity = this.nebulaToolkitService.copyObjectByWhiteList(industryDto, IndustryEntity.class, HashSet.class, LinkedList.class);
        this.industryRepository.save(industryEntity);
        return this.nebulaToolkitService.copyObjectByWhiteList(industryEntity, IndustryVo.class, HashSet.class, LinkedList.class);
    }

    private void createValidate(IndustryDto industryDto) {
        Validate.notNull(industryDto, "新增实体不能为空");
        Validate.notBlank(industryDto.getName(), "行业名称不能为空！");
        //生成行业编码
        String code = this.generateCode(TenantUtils.getTenantCode());
        industryDto.setCode(code);

        //初始化信息
        Date now = new Date();
        industryDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        industryDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        industryDto.setTenantCode(TenantUtils.getTenantCode());
//        UserIdentity loginUser = loginUserService.getLoginUser();
//        industryDto.setCreateAccount(loginUser.getAccount());
//        industryDto.setCreateName(loginUserService.getLoginAccountName());
//        industryDto.setModifyAccount(loginUser.getAccount());
//        industryDto.setModifyName(loginUserService.getLoginAccountName());
//        industryDto.setCreateTime(now);
//        industryDto.setModifyTime(now);

    }

    /**
     * 生成编码
     */
    private String generateCode(String tenantCode) {
        String redisKey = String.format(MMS_INDUSTRY_CODE_REDIS_KEY, tenantCode);
        String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        return StringUtils.join(MMS_INDUSTRY_CODE_PREFIX, format.format(new Date()), index);
    }

    @Override
    public IndustryVo update(IndustryDto industryDto) {
        //修改参数校验
        this.updateValidate(industryDto);
        IndustryEntity oldEntity = this.industryRepository.lambdaQuery().eq(IndustryEntity::getId, industryDto.getId()).one();
        Validate.notNull(oldEntity, "编辑行业信息不存在！");

        IndustryEntity industryEntity = this.nebulaToolkitService.copyObjectByWhiteList(industryDto, IndustryEntity.class, HashSet.class, LinkedList.class);
        this.industryRepository.saveOrUpdate(industryEntity);
        return this.nebulaToolkitService.copyObjectByWhiteList(industryEntity, IndustryVo.class, HashSet.class, LinkedList.class);
    }

    public void updateValidate(IndustryDto industryDto) {

        Validate.notNull(industryDto, "编辑实体不能为空");
        Validate.notBlank(industryDto.getId(), "修改信息时主键id不能为空！");
        Validate.notBlank(industryDto.getCode(), "修改信息时，行业编码不能为空");
        Validate.notBlank(industryDto.getName(), "修改信息时，行业名称不能为空");

        //初始化信息
        Date now = new Date();
        industryDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        industryDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        industryDto.setTenantCode(TenantUtils.getTenantCode());
//        UserIdentity loginUser = loginUserService.getLoginUser();
//        industryDto.setModifyAccount(loginUser.getAccount());
//        industryDto.setModifyName(loginUserService.getLoginAccountName());
//        industryDto.setModifyTime(now);

    }

    @Override
    public void enableBatch(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        List<IndustryEntity> industryEntities = this.industryRepository.lambdaQuery().in(IndustryEntity::getId, ids).list();
        if (CollectionUtils.isEmpty(industryEntities)) {
            return;
        }
        industryEntities.stream().forEach(industryEntity -> industryEntity.setEnableStatus(EnableStatusEnum.ENABLE.getCode()));
        this.industryRepository.updateBatchById(industryEntities);
    }

    @Override
    public void disableBatch(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }
        List<IndustryEntity> industryEntities = this.industryRepository.lambdaQuery().in(IndustryEntity::getId, ids).list();
        if (CollectionUtils.isEmpty(industryEntities)) {
            return;
        }
        industryEntities.stream().forEach(industryEntity -> industryEntity.setEnableStatus(EnableStatusEnum.DISABLE.getCode()));
        this.industryRepository.updateBatchById(industryEntities);
    }

    @Override
    public void deleteBatch(List<String> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            return;
        }

        this.industryRepository.lambdaUpdate().set(IndustryEntity::getDelFlag,DelFlagStatusEnum.DELETE)
                .in(IndustryEntity::getId,ids).update();
    }
}
