package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.member.business.member.local.entity.InfoCategory;
import com.biz.crm.member.business.member.local.mapper.InfoCategoryMapper;
import com.biz.crm.member.business.member.local.repository.InfoCategoryRepository;
import com.biz.crm.member.business.member.local.service.InfoCategoryService;
import com.biz.crm.member.business.member.sdk.dto.InfoCategoryDto;
import com.biz.crm.member.business.member.sdk.dto.InfoCategoryPaginationDto;
import com.biz.crm.member.business.member.sdk.vo.InfoCategoryVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author 吴平
 * @version 1.0.0
 * @data 2023/6/16 10:59
 * @description 资讯分类管理
 */
@Slf4j
@Service
public class InfoCategoryServiceImpl implements InfoCategoryService {

  @Autowired
  private InfoCategoryRepository infoCategoryRepository;

  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  private InfoCategoryMapper infoCategoryMapper;

  /**
   * 新增一个资讯分类
   *
   * @param infoCategoryDto
   * @return
   */
  @Override
  public InfoCategoryVo createInfoCategory(InfoCategoryDto infoCategoryDto) {
    this.createValidation(infoCategoryDto);
    InfoCategory infoCategory = this.nebulaToolkitService
        .copyObjectByWhiteList(infoCategoryDto, InfoCategory.class, HashSet.class,
            LinkedList.class);
    infoCategory.setTenantCode(TenantUtils.getTenantCode());
    infoCategory.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    infoCategory.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    infoCategory.setCode(this.generateCodeService.generateCode("ZX_CATEGORY", 5).get(0));
    this.infoCategoryRepository.save(infoCategory);
    InfoCategoryVo infoCategoryVo = this.nebulaToolkitService
        .copyObjectByWhiteList(infoCategory, InfoCategoryVo.class, HashSet.class,
            LinkedList.class);
    return infoCategoryVo;
  }

  /**
   * 根据id修改资讯分类信息
   * @param infoCategoryDto
   * @return
   */
  @Override
  public InfoCategoryVo updateInfoCategory(InfoCategoryDto infoCategoryDto) {
    Optional.ofNullable(infoCategoryDto.getInfoSort()).ifPresent(infoSort->{
      Validate.isTrue(infoSort>0, "序号不能小于0");
    });
    String infoCategoryId = infoCategoryDto.getId();
    Validate.notBlank(infoCategoryId,"修改资讯分类信息id不能为空");
    InfoCategory current = this.infoCategoryRepository.getById(infoCategoryId);
    Validate.notNull(current, "修改信息不存在");
    InfoCategory infoCategory = this.nebulaToolkitService
        .copyObjectByWhiteList(infoCategoryDto, InfoCategory.class, HashSet.class,
            LinkedList.class);
    this.infoCategoryRepository.saveOrUpdate(infoCategory);
    InfoCategoryVo result = this.nebulaToolkitService
        .copyObjectByWhiteList(infoCategory, InfoCategoryVo.class, HashSet.class, LinkedList.class);
    return result;
  }

  /**
   * 分页获取资讯分类信息
   * @param paginationDto
   * @return
   */
  @Override
  public Page<InfoCategoryVo> pageInfoCategory(Pageable pageable, InfoCategoryPaginationDto paginationDto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    paginationDto = Optional.ofNullable(paginationDto).orElse(new InfoCategoryPaginationDto());
    Page<InfoCategoryVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<InfoCategoryVo> result = this.infoCategoryMapper.pageInfoCategory(paginationDto, page ,
        DelFlagStatusEnum.NORMAL.getCode());
    return result;
  }

  /**
   * 根据id获取资讯分类信息
   * @param id 资讯分类id
   * @return
   */
  @Override
  public InfoCategoryVo getInfoCategoryById(String id) {
    Validate.notBlank(id, "获取单个资讯分类信息时，id不能为空");
    InfoCategory infoCategory = this.infoCategoryRepository.getById(id);
    Validate.notNull(infoCategory, "查询的信息不存在");
    InfoCategoryVo result = this.nebulaToolkitService
        .copyObjectByWhiteList(infoCategory, InfoCategoryVo.class, HashSet.class, LinkedList.class);
    return result;
  }

  /**
   * 根据id批量启用资讯分类
   * @param ids
   */
  @Transactional
  @Override
  public void turnOnInfoCategory(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "ids不能为空");
    this.infoCategoryRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
  }

  /**
   * 根据id批量禁用资讯分类
   * @param ids
   */
  @Transactional
  @Override
  public void turnOffInfoCategory(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "ids不能为空");
    this.infoCategoryRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
  }

  /**
   * 根据id批量逻辑删除资讯分类
   * @param ids
   */
  @Transactional
  @Override
  public void logicDeleteInfoCategory(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.infoCategoryRepository.logicDeleteInfoCategoryByIds(ids);
  }

  /**
   * 小程序端根据应用范围获取有效的资讯分类
   * @param applyRange 应用范围,数据字典:mms_info_category
   * @return
   */
  @Override
  public List<InfoCategoryVo> appListEnableInfoCategoryByApplyRange(String applyRange) {
    Validate.notBlank(applyRange, "应用范围不能为空");
    LambdaQueryWrapper<InfoCategory> queryWrapper = new LambdaQueryWrapper<>();
    queryWrapper.eq(InfoCategory::getApplyRange, applyRange)
        .eq(InfoCategory::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(InfoCategory::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .orderByAsc(InfoCategory::getInfoSort);
    List<InfoCategory> infoCategories = this.infoCategoryRepository.list(queryWrapper);
    Collection<InfoCategoryVo> result = this.nebulaToolkitService
        .copyCollectionByWhiteList(infoCategories, InfoCategory.class, InfoCategoryVo.class,
            HashSet.class, LinkedList.class);
    return (List<InfoCategoryVo>) result;
  }


  /**
   * 新增资讯分类校验
   *
   * @param infoCategoryDto
   */
  private void createValidation(InfoCategoryDto infoCategoryDto) {
    this.validation(infoCategoryDto);
  }

  private void validation(InfoCategoryDto infoCategoryDto) {
    Validate.notBlank(infoCategoryDto.getInfoName(), "资讯分类不能为空");
    Validate.notNull(infoCategoryDto.getInfoSort(), "排序不能为空");
    Validate.notBlank(infoCategoryDto.getApplyRange(), "应用范围不能为空");
    Validate.isTrue(infoCategoryDto.getInfoSort()>0, "序号不能小于0");
  }
}
