package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.member.business.member.local.entity.InfoManage;
import com.biz.crm.member.business.member.local.entity.InfoUpvote;
import com.biz.crm.member.business.member.local.helper.UserSearchHelper;
import com.biz.crm.member.business.member.local.repository.InfoManageRepository;
import com.biz.crm.member.business.member.local.service.DiscussService;
import com.biz.crm.member.business.member.local.service.InfoManageService;
import com.biz.crm.member.business.member.local.service.InfoUpvoteService;
import com.biz.crm.member.business.member.sdk.dto.InfoManageDto;
import com.biz.crm.member.business.member.sdk.dto.InfoManagePaginationDto;
import com.biz.crm.member.business.member.sdk.dto.InfoManagePubDto;
import com.biz.crm.member.business.member.sdk.enums.DiscussCheckStatusEnum;
import com.biz.crm.member.business.member.sdk.enums.InfoApplyModelEnum;
import com.biz.crm.member.business.member.sdk.enums.InfoTypeEnum;
import com.biz.crm.member.business.member.sdk.vo.InfoCountDto;
import com.biz.crm.member.business.member.sdk.vo.InfoManageVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author 吴平
 * @version 1.0.0
 * @data 2023/6/19 10:11
 * @description 资讯管理
 */
@Service
@Slf4j
public class InfoManageServiceImpl implements InfoManageService {

  @Autowired
  private InfoManageRepository infoManageRepository;

  @Autowired
  private NebulaToolkitService nebulaToolkitService;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  private InfoUpvoteService infoUpvoteService;

  @Autowired
  private DiscussService discussService;

  @Autowired
  private UserSearchHelper userSearchHelper;

  /**
   * 新增富文本类型的资讯管理
   * @param dto
   * @return
   */
  @Override
  public InfoManageVo createRichText(InfoManageDto dto) {
    this.createRichTextValidate(dto);
    InfoManage infoManage = this.nebulaToolkitService
        .copyObjectByWhiteList(dto, InfoManage.class, HashSet.class,
            LinkedList.class);
    InfoManageVo result = this.create(infoManage);
    return result;
  }

  /**
   * 新增公众号类型的资讯管理
   * @param dto
   * @return
   */
  @Override
  public InfoManageVo createPubAccount(InfoManagePubDto dto) {
    this.createPubValidate(dto);
    InfoManage infoManage = this.nebulaToolkitService
        .copyObjectByWhiteList(dto, InfoManage.class, HashSet.class,
            LinkedList.class);
    InfoManageVo result = this.create(infoManage);
    return result;
  }

  /**
   * 根据id集合批量逻辑删除资讯
   * @param ids
   */
  @Transactional
  @Override
  public void logicDeleteInfoByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "ids不能为空");
    this.infoManageRepository.logicDeleteByIds(ids);
  }

  /**
   * 修改资讯信息
   * @param dto
   * @return
   */
  @Override
  public InfoManageVo updateInfo(InfoManageDto dto) {
    Optional.ofNullable(dto.getSerialNum()).ifPresent(serialNum->{
      Validate.isTrue(serialNum>0, "序号不能小于0");
    });
    String infoId = dto.getId();
    Validate.notBlank(infoId, "id不能为空");
    InfoManage infoManage = this.nebulaToolkitService
        .copyObjectByWhiteList(dto, InfoManage.class, HashSet.class, LinkedList.class);
    this.infoManageRepository.saveOrUpdate(infoManage);
    InfoManageVo result = this.nebulaToolkitService
        .copyObjectByWhiteList(infoManage, InfoManageVo.class, HashSet.class, LinkedList.class);
    return result;
  }

  /**
   * 根据id集合修改资讯状态为启用
   * @param ids
   */
  @Transactional
  @Override
  public void turnOnInfoByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id不能为空");
    this.infoManageRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
  }

  /**
   * 根据id集合禁用资讯
   * @param ids
   */
  @Transactional
  @Override
  public void turnOffByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id不能为空");
    this.infoManageRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
  }

  /**
   * 分页获取资讯
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<InfoManageVo> pageInfo(Pageable pageable, InfoManagePaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new InfoManagePaginationDto());
    Page<InfoManageVo> page = new Page<>(pageable.getPageNumber(),
        pageable.getPageSize());
    Page<InfoManageVo> result = this.infoManageRepository.pageInfo(page, dto);
    this.populateCount(result.getRecords(), StringUtils.EMPTY, StringUtils.EMPTY);
    return result;
  }

  /**
   * 小程序获取资讯精选模块的最新三个资讯信息
   * @return
   */
  @Override
  public List<InfoManageVo> getAppIndexInfoLimitThree() {
    LambdaQueryWrapper<InfoManage> queryWrapper = new LambdaQueryWrapper<>();
    queryWrapper
        .eq(InfoManage::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(InfoManage::getApplyModel, InfoApplyModelEnum.INFO_CHOICE_NESS)
        .eq(InfoManage::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .orderByDesc(InfoManage::getCreateTime)
        .last("limit 3");
    List<InfoManage> infoManages = this.infoManageRepository.list(queryWrapper);
    Collection<InfoManageVo> result = this.nebulaToolkitService
        .copyCollectionByWhiteList(infoManages, InfoManage.class, InfoManageVo.class, HashSet.class,
            LinkedList.class);
    return (List<InfoManageVo>) result;
  }

  /**
   * 小程序获取资讯列表，根据应用模块查询
   * 应用模块对应的数据字典是:mms_info_apply_model,枚举:InfoApplyModelEnum
   * @param pageable
   * @return
   */
  @Override
  public Page<InfoManageVo> pageInfoManageApp(Pageable pageable, String infoCategoryCode, String applyModel) {
    Validate.notBlank(applyModel, "应用模块不能为空");
    Validate.notBlank(infoCategoryCode, "资讯分类编码不能为空");
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    InfoManagePaginationDto paginationDto = new InfoManagePaginationDto();
    paginationDto.setApplyModel(applyModel);
    paginationDto.setInfoCategoryCode(infoCategoryCode);
    Page<InfoManageVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    Page<InfoManageVo> result = this.infoManageRepository.pageInfo(page, paginationDto);
    // 填充统计评论数量和点赞数量
    this.populateCount(result.getRecords(),
        DiscussCheckStatusEnum.APPROVE.getValue(),
        EnableStatusEnum.ENABLE.getCode());
    // 填充当前用户是否已点赞
    this.populateUpvote(result.getRecords());
    return result;
  }


  /**
   * 为指定资讯增加一个阅读量
   * @param infoCode
   */
  @Override
  public void addInfoReadNum(String infoCode, String modifyAccount, String modifyName, Date modifyTime) {
    Integer step = 1;
    this.infoManageRepository.incrementReadNumByStep(infoCode, step, modifyAccount, modifyName, modifyTime);
  }

  /**
   * 小程序获取资讯详情
   * 在此接口中增加资讯的阅读数量
   * @param infoCode
   * @return
   */
  @Override
  public InfoManageVo appGetInfoDetailByCode(String infoCode) {
    LambdaQueryWrapper<InfoManage> queryWrapper = new LambdaQueryWrapper<>();
    queryWrapper.eq(InfoManage::getCode, infoCode);
    InfoManage infoManage = this.infoManageRepository.getOne(queryWrapper);
    // 阅读量+1
    String modifyAccount = infoManage.getModifyAccount();
    String modifyName = infoManage.getModifyName();
    Date modifyTime = infoManage.getModifyTime();
    this.addInfoReadNum(infoCode, modifyAccount, modifyName, modifyTime);
    InfoManageVo result = this.nebulaToolkitService
        .copyObjectByWhiteList(infoManage, InfoManageVo.class, HashSet.class, LinkedList.class);
    // 填充详情的点赞量和评论量
    this.populateCount(Collections.singletonList(result),
        DiscussCheckStatusEnum.APPROVE.getValue(),
        EnableStatusEnum.ENABLE.getCode());
    // 填充当前用户是否已点赞
    this.populateUpvote(Collections.singletonList(result));
    return result;
  }

  /**
   * 小程序分享资讯
   * @param infoCode
   * @return
   */
  @Override
  public InfoManageVo appShareInfoByCode(String infoCode) {
    Validate.notBlank(infoCode, "资讯编码不能为空");
    LambdaQueryWrapper<InfoManage> queryWrapper = new LambdaQueryWrapper<>();
    queryWrapper.eq(InfoManage::getCode, infoCode);
    InfoManage infoManage = this.infoManageRepository.getOne(queryWrapper);
    InfoManageVo result = this.nebulaToolkitService
        .copyObjectByWhiteList(infoManage, InfoManageVo.class, HashSet.class, LinkedList.class);
    return result;
  }

  /**
   * 获取所有，有效的，且分类是精选分类的，资讯信息
   * @return
   */
  @Override
  public List<InfoManageVo> listEnableChoiceNessCategoryInfo() {
    LambdaQueryWrapper<InfoManage> queryWrapper = new LambdaQueryWrapper<>();
    queryWrapper.eq(InfoManage::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .eq(InfoManage::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .isNotNull(InfoManage::getChoiceNessCategory)
        .orderByDesc(InfoManage::getCreateTime);
    List<InfoManage> infoManageList = this.infoManageRepository.list(queryWrapper);
    if (CollectionUtils.isEmpty(infoManageList)){
      return Collections.EMPTY_LIST;
    }
    Collection<InfoManageVo> result = this.nebulaToolkitService
        .copyCollectionByWhiteList(infoManageList, InfoManage.class, InfoManageVo.class,
            HashSet.class, LinkedList.class);
    return (List<InfoManageVo>) result;
  }

  /**
   * 小程序端根据精选分类编码获取最新的一个资讯
   * @param choiceCategoryCode
   * @return
   */
  @Override
  public InfoManageVo getInfoByChoiceCodeLast(String choiceCategoryCode) {
    InfoManage infoManage = this.infoManageRepository.lambdaQuery()
        .eq(InfoManage::getChoiceNessCategory, choiceCategoryCode)
        .eq(InfoManage::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(InfoManage::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
        .orderByDesc(InfoManage::getCreateTime)
        .last("limit 1")
        .one();
    InfoManageVo result = this.nebulaToolkitService.copyObjectByWhiteList(infoManage,
        InfoManageVo.class,
        HashSet.class,
        LinkedList.class);
    // 填充详情的点赞量和评论量
    this.populateCount(Collections.singletonList(result),
        DiscussCheckStatusEnum.APPROVE.getValue(),
        EnableStatusEnum.ENABLE.getCode());
    // 填充当前用户是否已点赞
    this.populateUpvote(Collections.singletonList(result));
    return result;
  }

  /**
   * 填充当前用户是否已点赞
   * @param records
   */
  private void populateUpvote(List<InfoManageVo> records) {
    // 获取资讯编码
    List<String> infoCodes = records
        .stream()
        .map(InfoManageVo::getCode)
        .distinct()
        .collect(Collectors.toList());
    // 获取当前登录用户的memberCode
    String memberCode = this.userSearchHelper.getMemberLogin().getMemberCode();
    // 获取当前登录用户的点赞信息
    List<InfoUpvote> upvoteList = this.infoUpvoteService
        .listUpvoteInfoByMemberCodeAndInfoCodes(infoCodes, memberCode);
    Set<String> upvoteInfoCodes = upvoteList
        .stream()
        .map(InfoUpvote::getInfoCode)
        .collect(Collectors.toSet());
    for (InfoManageVo record : records) {
      String infoCode = record.getCode();
      record.setAlreadyUpvote(upvoteInfoCodes.contains(infoCode));
    }
  }

  /**
   * 填充阅读量、点赞量、评论量
   * @param infoManageVoList
   */
  private void populateCount(List<InfoManageVo> infoManageVoList, String checkStatus, String enableStatus){
    // 点赞量资讯编码集合
    List<String> infoCodes = infoManageVoList.stream()
        .map(InfoManageVo::getCode)
        .distinct()
        .collect(Collectors.toList());
    // 评论量资讯编码集合
    List<String> discussInfoCodes = infoManageVoList.stream()
        .filter(obj -> (obj.getInfoType().equals(InfoTypeEnum.RICH_TEXT.getValue())))
        .map(InfoManageVo::getCode)
        .distinct()
        .collect(Collectors.toList());
    // 点赞量统计
    Map<String, Integer> upvoteCountMap = this.infoUpvoteService.countUpvoteByInfoCodes(infoCodes)
        .stream()
        .collect(Collectors.toMap(InfoCountDto::getInfoCode, InfoCountDto::getCountNum));
    // 评论统计
    Map<String, Integer> discussCountMap = this.discussService.countByInfoCodes(discussInfoCodes, checkStatus, enableStatus)
        .stream()
        .collect(Collectors.toMap(InfoCountDto::getInfoCode, InfoCountDto::getCountNum));
    // 赋值
    for (InfoManageVo infoManageVo : infoManageVoList) {
      String infoCode = infoManageVo.getCode();

      Integer upvoteNum = upvoteCountMap.getOrDefault(infoCode, 0);
      infoManageVo.setUpvoteNum(upvoteNum);

      Integer discussNum = discussCountMap.getOrDefault(infoCode, 0);
      infoManageVo.setDiscussNum(discussNum);
    }
  }

  /**
   * 统一新增资讯管理
   * @param infoManage
   * @return
   */
  private InfoManageVo create(InfoManage infoManage){
    infoManage.setCode(this.generateCodeService.generateCode("ZX", 5).get(0));
    infoManage.setTenantCode(TenantUtils.getTenantCode());
    infoManage.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    infoManage.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    infoManage.setReadNum(0);
    this.infoManageRepository.save(infoManage);
    InfoManageVo result = this.nebulaToolkitService
        .copyObjectByWhiteList(infoManage, InfoManageVo.class, HashSet.class, LinkedList.class);
    return result;
  }

  /**
   * 新增公众号类型的资讯管理校验
   * @param dto
   */
  private void createPubValidate(InfoManagePubDto dto){
    Validate.notNull(dto.getSerialNum(), "序号不能为空");
    Validate.notBlank(dto.getInfoTitle(), "资讯标题不能为空");
    Validate.notNull(dto.getStartTime(), "开始时间不能为空");
    Validate.notNull(dto.getEndTime(), "结束时间不能为空");
    Validate.notBlank(dto.getInfoType(), "资讯类型不能为空");
    Validate.notBlank(dto.getTargetUrl(), "跳转链接不能为空");
    Validate.isTrue(dto.getSerialNum()>0, "序号不能小于0");
  }

  /**
   * 新增富文本类型的资讯管理校验
   * @param dto
   */
  private void createRichTextValidate(InfoManageDto dto){
    this.validation(dto);
    Validate.notBlank(dto.getApplyModel(), "应用模块不能为空");
    Validate.notBlank(dto.getInfoCategory(), "资讯分类不能为空");
    Validate.notBlank(dto.getInfoType(), "资讯类型不能为空");
    Validate.notBlank(dto.getOverPic(), "封面图不能为空");
  }

  /**
   * 校验规则
   * @param dto
   */
  private void validation(InfoManageDto dto){
    Validate.notNull(dto.getSerialNum(), "序号不能为空");
    Validate.notBlank(dto.getInfoTitle(), "资讯标题不能为空");
    Validate.notNull(dto.getStartTime(), "开始时间不能为空");
    Validate.notNull(dto.getEndTime(), "结束时间不能为空");
    Validate.isTrue(dto.getSerialNum()>0, "序号不能小于0");
  }
}
