package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.member.business.member.local.entity.InfoUpvote;
import com.biz.crm.member.business.member.local.helper.UserSearchHelper;
import com.biz.crm.member.business.member.local.repository.InfoUpvoteRepository;
import com.biz.crm.member.business.member.local.service.InfoUpvoteService;
import com.biz.crm.member.business.member.sdk.vo.InfoCountDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.Collections;
import java.util.List;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * @author 吴平
 * @version 1.0.0
 * @data 2023/6/19 16:19
 * @description 资讯点赞量管理
 */
@Service
@Slf4j
public class InfoUpvoteServiceImpl implements InfoUpvoteService {

  @Autowired
  private LoginUserService loginUserService;

  @Autowired
  private InfoUpvoteRepository infoUpvoteRepository;

  @Autowired
  private UserSearchHelper userSearchHelper;

  /**
   * 为指定资讯添加一条点信息
   * @param infoCode
   */
  @Override
  public void addInfoUpvote(String infoCode) {
    Validate.notBlank(infoCode, "资讯编码不能为空");
    String memberCode = userSearchHelper.getMemberLogin().getMemberCode();

    LambdaQueryWrapper<InfoUpvote> queryWrapper = new LambdaQueryWrapper<>();
    queryWrapper.eq(InfoUpvote::getInfoCode, infoCode)
        .eq(InfoUpvote::getMemberCode, memberCode)
        .eq(InfoUpvote::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
    InfoUpvote current = this.infoUpvoteRepository.getOne(queryWrapper);
    Validate.isTrue(ObjectUtils.isEmpty(current), "不可重复点赞");
    InfoUpvote infoUpvote = new InfoUpvote();
    infoUpvote.setMemberCode(memberCode);
    infoUpvote.setInfoCode(infoCode);
    infoUpvote.setTenantCode(TenantUtils.getTenantCode());
    infoUpvote.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    infoUpvote.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.infoUpvoteRepository.save(infoUpvote);
  }

  /**
   * 取消点赞
   * @param infoCode 资讯编码
   */
  @Override
  public void logicDeleteUpvote(String infoCode) {
    Validate.notBlank(infoCode, "资讯编码不能为空");
    String memberCode = userSearchHelper.getMemberLogin().getMemberCode();
    this.infoUpvoteRepository.cancelUpvoteByInfoCode(infoCode, memberCode);
  }

  /**
   * 根据资讯编码统计点赞量
   * @param infoCode
   * @return
   */
  @Override
  public Integer countUpvoteByInfoCode(String infoCode) {
    LambdaQueryWrapper<InfoUpvote> queryWrapper = new LambdaQueryWrapper<>();
    queryWrapper.eq(InfoUpvote::getInfoCode, infoCode)
        .eq(InfoUpvote::getDelFlag, DelFlagStatusEnum.NORMAL.getCode());
    int result = this.infoUpvoteRepository.count(queryWrapper);
    return result;
  }

  /**
   * 批量统计资讯点赞数
   * @param infoCodes
   * @return
   */
  @Override
  public List<InfoCountDto> countUpvoteByInfoCodes(List<String> infoCodes) {
    if (CollectionUtils.isEmpty(infoCodes)){
      return Collections.EMPTY_LIST;
    }
    List<InfoCountDto> result = this.infoUpvoteRepository.getBaseMapper()
        .countByInfoCode(infoCodes, DelFlagStatusEnum.NORMAL.getCode());
    return result;
  }

  /**
   * 根据资讯编码和会员编码获取点赞信息
   * @param infoCodes
   * @param memberCode
   * @return
   */
  @Override
  public List<InfoUpvote> listUpvoteInfoByMemberCodeAndInfoCodes(List<String> infoCodes,
      String memberCode) {
    if (CollectionUtils.isEmpty(infoCodes)){
      return Collections.EMPTY_LIST;
    }
    Validate.notBlank(memberCode, "获取不到当前登录用户的会员编码");
    LambdaQueryWrapper<InfoUpvote> queryWrapper = new LambdaQueryWrapper<>();
    queryWrapper.in(InfoUpvote::getInfoCode, infoCodes)
        .eq(InfoUpvote::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .eq(InfoUpvote::getMemberCode, memberCode);
    return this.infoUpvoteRepository.list(queryWrapper);
  }
}
