package com.biz.crm.member.business.member.local.service.internal;


import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.local.entity.UuidEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.member.business.member.local.entity.Label;
import com.biz.crm.member.business.member.local.entity.LabelDetail;
import com.biz.crm.member.business.member.local.repository.LabelDetailRepository;
import com.biz.crm.member.business.member.local.repository.LabelRepository;
import com.biz.crm.member.business.member.local.service.LabelService;
import com.biz.crm.member.business.member.sdk.dto.LabelPaginationDto;
import com.biz.crm.member.business.member.sdk.enums.LabelDetailEnum;
import com.biz.crm.member.business.member.sdk.vo.LabelDetailVo;
import com.biz.crm.member.business.member.sdk.vo.LabelVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author hupan
 * @date 2023/6/13 17:35
 */
@Service("labelService")
public class LabelServiceImpl implements LabelService {

  @Autowired
  private NebulaToolkitService nebulaToolkitService;
  @Autowired
  private LabelRepository labelRepository;
  @Autowired
  private LabelDetailRepository labelDetailRepository;
  @Autowired
  private GenerateCodeService generateCodeService;

  @Override
  public Page<LabelVo> findByConditions(Pageable pageable, LabelPaginationDto labelPaginationDto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    labelPaginationDto = Optional.ofNullable(labelPaginationDto).orElse(new LabelPaginationDto());
    Page<LabelVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return labelRepository.findByConditions(page, labelPaginationDto);
  }

  @Override
  @Transactional
  public void create(LabelVo labelVo) {
    this.createValidation(labelVo);
    Label label = nebulaToolkitService
        .copyObjectByWhiteList(labelVo, Label.class, HashSet.class,
            LinkedList.class);
    //设置租户编号,生成标签编码和默认值
    String labelCode = generateCodeService.generateCode("LM", 5).get(0);
    label.setCode(labelCode);
    label.setTenantCode(TenantUtils.getTenantCode());
    label.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    label.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    labelRepository.saveOrUpdate(label);
    if (CollectionUtils.isNotEmpty(labelVo.getConditions())) {
      for (LabelDetailVo condition : labelVo.getConditions()) {
        LabelDetail labelDetail = nebulaToolkitService
            .copyObjectByWhiteList(condition, LabelDetail.class, HashSet.class,
                LinkedList.class);
        labelDetail.setId(null);
        labelDetail.setLabelCode(labelCode);
        //设置租户编号和默认值
        labelDetail.setTenantCode(TenantUtils.getTenantCode());
        labelDetail.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        labelDetail.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        labelDetailRepository.insert(labelDetail);
      }
    }
  }

  @Override
  @Transactional
  public void update(LabelVo labelVo) {
    this.updateValidation(labelVo);
    String currentId = labelVo.getId();
    Label current = this.labelRepository.selectById(currentId);
    current = Validate.notNull(current, "修改信息不存在");
    current = this.nebulaToolkitService
        .copyObjectByWhiteList(labelVo, Label.class, HashSet.class, LinkedList.class);
    this.labelRepository.saveOrUpdate(current);
    // 处理明细,先查询老的数据删除
    List<LabelDetail> list = this.labelDetailRepository.lambdaQuery()
        .eq(LabelDetail::getLabelCode, current.getCode()).list();
    if (CollectionUtils.isNotEmpty(list)) {
      Set<String> ids = list.stream().map(UuidEntity::getId).collect(Collectors.toSet());
      this.labelDetailRepository.removeByIds(ids);
    }
    if (CollectionUtils.isNotEmpty(labelVo.getConditions())) {
      for (LabelDetailVo condition : labelVo.getConditions()) {
        LabelDetail labelDetail = nebulaToolkitService
            .copyObjectByWhiteList(condition, LabelDetail.class, HashSet.class,
                LinkedList.class);
        labelDetail.setId(null);
        labelDetail.setLabelCode(current.getCode());
        //设置租户编号和默认值
        labelDetail.setTenantCode(TenantUtils.getTenantCode());
        labelDetail.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        labelDetail.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        labelDetailRepository.insert(labelDetail);
      }
    }
  }

  @Override
  public LabelVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    Label label = labelRepository.selectById(id);
    Validate.notNull(label, "标签未找到");
    LabelVo labelVo = nebulaToolkitService
        .copyObjectByWhiteList(label, LabelVo.class, HashSet.class, LinkedList.class);
    // 查询集合
    List<LabelDetail> list = this.labelDetailRepository.lambdaQuery()
        .eq(LabelDetail::getLabelCode, label.getCode()).list();
    if (CollectionUtils.isNotEmpty(list)) {
      labelVo.setConditions((List<LabelDetailVo>) this.nebulaToolkitService
          .copyCollectionByWhiteList(list, LabelDetail.class, LabelDetailVo.class, HashSet.class,
              ArrayList.class));
    } else {
      labelVo.setConditions(Lists.newArrayList());
    }
    return labelVo;
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.labelRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.labelRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.labelRepository.updateDelFlagByIds(ids);
  }

  private void createValidation(LabelVo labelVo) {
    this.validation(labelVo);
  }

  private void updateValidation(LabelVo labelVo) {
    this.validation(labelVo);
  }

  private void validation(LabelVo labelVo) {
    Validate.notBlank(labelVo.getName(), "标签名称不能为空");
    Validate.notBlank(labelVo.getType(), "标签类型不能为空");
    Validate.notBlank(labelVo.getCategoryCode(), "标签分类不能为空");
    Validate.notBlank(labelVo.getCategoryName(), "标签分类不能为空");
    if (Objects.equals("0",labelVo.getType())) {
    } else if (Objects.equals("1",labelVo.getType())) {
      Validate.isTrue(CollectionUtils.isNotEmpty(labelVo.getConditions()), "自动标签条件不能为空");
      for (LabelDetailVo condition : labelVo.getConditions()) {
        Validate.notBlank(condition.getConditionType(), "条件类型不能为空");
        if (Objects
            .equals(LabelDetailEnum.LAST_SIGN_DAY.getValue(), condition.getConditionType())) {
          Validate.notNull(condition.getMin(), "最近签收时间X天不能为空");
        } else if (Objects
            .equals(LabelDetailEnum.CONTINUOUS_YEAR.getValue(), condition.getConditionType())) {
          Validate.notNull(condition.getMin(), "会员等级连续X年不能为空");
          Validate.notBlank(condition.getLevelCode(), "请选择等级");
          Validate.notBlank(condition.getLevelName(), "请选择等级");
        } else {
          Validate.notNull(condition.getMin(), "最小区间值不能为空");
          Validate.notNull(condition.getMax(), "最大区间值不能为空");
          Validate.isTrue(condition.getMin().intValue() < condition.getMax().intValue(),"最小区间值不能大于最大区间值");
        }
      }
    } else {
      Validate.isTrue(false, "标签类型异常");
    }
  }
}
