package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.member.business.member.local.entity.AppletPopupNotify;
import com.biz.crm.member.business.member.local.entity.SystemSettingsDetail;
import com.biz.crm.member.business.member.local.helper.UserSearchHelper;
import com.biz.crm.member.business.member.local.service.AppletPopupNotifyService;
import com.biz.crm.member.business.member.local.service.MemberInfoAssetsService;
import com.biz.crm.member.business.member.sdk.dto.MemberInfoIntegralRecordPaginationAPIDto;
import com.biz.crm.member.business.member.sdk.enums.IntegralSourceEnum;
import com.biz.crm.member.business.member.sdk.enums.MemberInfoIntegralUseStatusEnum;
import com.biz.crm.member.business.member.sdk.enums.PointsViewDetailAPIEnum;
import com.biz.crm.member.business.member.sdk.enums.SystemSettingsDetailEnum;
import com.biz.crm.member.business.member.sdk.enums.TriggerConditionEnum;
import com.biz.crm.member.business.member.sdk.enums.TriggerTypeEnum;
import com.biz.crm.member.business.member.sdk.vo.MemberInfoAssetsVo;
import com.biz.crm.member.business.member.sdk.vo.MemberInfoIntegralRecordAPIVo;
import com.biz.crm.member.business.member.sdk.vo.MemberInfoIntegralRecordVo;
import com.biz.crm.member.business.member.local.entity.MemberInfoIntegralRecord;
import com.biz.crm.member.business.member.local.repository.MemberInfoIntegralRecordRepository;
import com.biz.crm.member.business.member.local.service.MemberInfoIntegralRecordService;
import com.biz.crm.member.business.member.sdk.dto.MemberInfoIntegralRecordPaginationDto;
import com.biz.crm.member.business.member.sdk.event.MemberInfoIntegralRecordEventListener;
import com.biz.crm.member.business.member.sdk.vo.MemberIntegralCountVo;
import com.biz.crm.member.business.member.sdk.vo.applet.AppletPopupNotifyVo;
import com.biz.crm.member.business.member.sdk.vo.login.MemberUserDetails;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.bizunited.nebula.task.annotations.DynamicTaskService;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.YearMonth;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 会员积分变动记录(MemberInfoIntegralRecord)表服务实现类
 *
 * @author zouhs
 * @date 2023-06-19 14:34:02
 */
@Slf4j
@Service("memberInfoIntegralRecordService")
public class MemberInfoIntegralRecordServiceImpl implements MemberInfoIntegralRecordService {

  @Autowired(required = false)
  private MemberInfoIntegralRecordRepository memberInfoIntegralRecordRepository;
  @Autowired(required = false)
  private SystemSettingsDetailServiceImpl systemSettingsDetailService;

  @Autowired(required = false)
  private List<MemberInfoIntegralRecordEventListener> eventListeners;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private MemberInfoAssetsService memberInfoAssetsService;

  @Autowired
  private UserSearchHelper userSearchHelper;

  @Autowired
  private AppletPopupNotifyService appletPopupNotifyService;

  /**
   * 后台条件分页获取积分记录
   *
   * @param pageable
   * @param dto
   * @return
   */
  @Override
  public Page<MemberInfoIntegralRecordVo> findByConditions(Pageable pageable,
      MemberInfoIntegralRecordPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new MemberInfoIntegralRecordPaginationDto());
    Page<MemberInfoIntegralRecordVo> page = new Page<>(pageable.getPageNumber(),
        pageable.getPageSize());
    return this.memberInfoIntegralRecordRepository.findByConditions(page, dto);
  }

  @Override
  public Page<MemberInfoIntegralRecordAPIVo> findByConditionsAPI(Pageable pageable,
      MemberInfoIntegralRecordPaginationAPIDto dto) {
    // 获取用户
    MemberUserDetails memberLogin = userSearchHelper.getMemberLogin();
    dto.setMemberCode(memberLogin.getMemberCode());
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new MemberInfoIntegralRecordPaginationAPIDto());
    Page<MemberInfoIntegralRecordAPIVo> page = new Page<>(pageable.getPageNumber(),
        pageable.getPageSize());
    //用于拼接条件的集合
    List<String> sources = new ArrayList<>();
    //如果source的值为空,则按照type来拼接条件
    if (StringUtils.isBlank(dto.getSource())) {
      //查看收入的全部信息
      if (PointsViewDetailAPIEnum.INCOME.getValue().equals(dto.getType())) {
        sources.add(IntegralSourceEnum.SYS_ADD.getValue());
        sources.add(IntegralSourceEnum.QUESTIONNAIRE.getValue());
        sources.add(IntegralSourceEnum.CULTURAL_TOURISM_TOPIC.getValue());
        sources.add(IntegralSourceEnum.BIG_TURNTABLE.getValue());
        sources.add(IntegralSourceEnum.GROUP_BUY_RECEIPT.getValue());
        sources.add(IntegralSourceEnum.DAY_SIGN.getValue());
        dto.setSources(sources);
      }
      //查看支出的全部信息
      if (PointsViewDetailAPIEnum.EXPENDITURE.getValue().equals(dto.getType())) {
        sources.add(IntegralSourceEnum.PRODUCT_EXCHANGE.getValue());
        sources.add(IntegralSourceEnum.BIG_TURNTABLE_PAY.getValue());
        dto.setSources(sources);
      }
      //查看扣减的全部信息
      if (PointsViewDetailAPIEnum.DEDUCTION.getValue().equals(dto.getType())) {
        sources.add(IntegralSourceEnum.SYS_DEDUCTION.getValue());
        sources.add(IntegralSourceEnum.INTEGRAL_EXPIRED.getValue());
        sources.add(IntegralSourceEnum.GROUP_BUY_RETREAT.getValue());
        dto.setSources(sources);
      }
      //查看全部信息
      if (PointsViewDetailAPIEnum.ALL_TYPE.getValue().equals(dto.getType())) {
        dto.setSources(null);
      }
    } else {
      //如果source的值不为空,则按照source的值来进行查询
      sources.add(dto.getSource());
    }
    dto.setSources(sources);
    return memberInfoIntegralRecordRepository.findByConditionsAPI(page, dto);
  }

  @Override
  public MemberInfoIntegralRecordVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    MemberInfoIntegralRecord memberInfoIntegralRecord = this.memberInfoIntegralRecordRepository
        .findById(id);
    MemberInfoIntegralRecordVo memberInfoIntegralRecordVo = this.nebulaToolkitService
        .copyObjectByWhiteList(memberInfoIntegralRecord, MemberInfoIntegralRecordVo.class,
            HashSet.class, LinkedList.class);
    return memberInfoIntegralRecordVo;
  }

  @Override
  public MemberInfoIntegralRecordVo findDetailByCode(String code) {
    if (StringUtils.isBlank(code)) {
      return null;
    }
    MemberInfoIntegralRecord memberInfoIntegralRecord = this.memberInfoIntegralRecordRepository
        .lambdaQuery().eq(MemberInfoIntegralRecord::getMemberCode, code).one();
    MemberInfoIntegralRecordVo memberInfoIntegralRecordVo = this.nebulaToolkitService
        .copyObjectByWhiteList(memberInfoIntegralRecord, MemberInfoIntegralRecordVo.class,
            HashSet.class, LinkedList.class);
    return memberInfoIntegralRecordVo;
  }

  @Override
  @Transactional
  public MemberInfoIntegralRecordVo create(MemberInfoIntegralRecordVo memberInfoIntegralRecordVo) {
    this.createValidation(memberInfoIntegralRecordVo);
    MemberInfoIntegralRecord memberInfoIntegralRecord = this.nebulaToolkitService
        .copyObjectByWhiteList(memberInfoIntegralRecordVo, MemberInfoIntegralRecord.class,
            HashSet.class, LinkedList.class);
    memberInfoIntegralRecord.setTenantCode(TenantUtils.getTenantCode());
    memberInfoIntegralRecord.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    memberInfoIntegralRecord.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.memberInfoIntegralRecordRepository.saveOrUpdate(memberInfoIntegralRecord);
    return memberInfoIntegralRecordVo;
  }

  @Override
  @Transactional
  public MemberInfoIntegralRecordVo update(MemberInfoIntegralRecordVo memberInfoIntegralRecordVo) {
    this.updateValidation(memberInfoIntegralRecordVo);
    String currentId = memberInfoIntegralRecordVo.getId();
    MemberInfoIntegralRecord current = memberInfoIntegralRecordRepository.findById(currentId);
    current = Validate.notNull(current, "修改信息不存在");
    current = this.nebulaToolkitService
        .copyObjectByWhiteList(memberInfoIntegralRecordVo, MemberInfoIntegralRecord.class,
            HashSet.class, LinkedList.class);
    this.memberInfoIntegralRecordRepository.saveOrUpdate(current);
    // 发送修改通知
    // TODO 发送通知
    return memberInfoIntegralRecordVo;
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.memberInfoIntegralRecordRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.memberInfoIntegralRecordRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.memberInfoIntegralRecordRepository.updateDelFlagByIds(ids);
  }

  @Override
  @Transactional
  public void commonInternal(MemberInfoIntegralRecordVo memberInfoIntegralRecordVo) {
    Validate.notBlank(memberInfoIntegralRecordVo.getMemberCode(), "会员编码不能为空");
    Validate.notBlank(memberInfoIntegralRecordVo.getSource(), "来源不能为空");
    Validate.notBlank(memberInfoIntegralRecordVo.getType(), "积分类型不能为空");
    Validate.notNull(memberInfoIntegralRecordVo.getIntegral(), "操作积分不能为空");
    Validate.isTrue(IntegralSourceEnum.isExist(memberInfoIntegralRecordVo.getSource()), "来源异常");
    Validate.isTrue(IntegralSourceEnum.isExistType(memberInfoIntegralRecordVo.getType()), "积分类型异常");
    MemberInfoAssetsVo memberInfoArchivesVo = memberInfoAssetsService
        .findDetailByMemberCode(memberInfoIntegralRecordVo.getMemberCode());
    Validate.notNull(memberInfoArchivesVo, "会员资产异常");
    MemberInfoIntegralRecord memberInfoIntegralRecord = new MemberInfoIntegralRecord();
    memberInfoIntegralRecord.setId(null);
    memberInfoIntegralRecord.setTenantCode(TenantUtils.getTenantCode());
    memberInfoIntegralRecord.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    memberInfoIntegralRecord.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    memberInfoIntegralRecord.setIntegralCode(generateCodeService.generateCode("JF", 5).get(0));
    memberInfoIntegralRecord.setMemberCode(memberInfoIntegralRecordVo.getMemberCode());
    memberInfoIntegralRecord.setOptTime(new Date());
    memberInfoIntegralRecord.setBeforeIntegral(memberInfoArchivesVo.getTotalIntegral());
    memberInfoIntegralRecord.setAfterIntegral(
        memberInfoArchivesVo.getTotalIntegral() + memberInfoIntegralRecordVo.getIntegral());
    memberInfoIntegralRecord.setSource(memberInfoIntegralRecordVo.getSource());
    memberInfoIntegralRecord.setType(memberInfoIntegralRecordVo.getType());
    memberInfoIntegralRecord.setIntegral(memberInfoIntegralRecordVo.getIntegral());
    this.memberInfoIntegralRecordRepository.save(memberInfoIntegralRecord);
    // 变更主表 - 将就用着吧这里会存在积分唯一性问题。以流水记录为准
    memberInfoArchivesVo.setTotalIntegral(
        memberInfoArchivesVo.getTotalIntegral() + memberInfoIntegralRecordVo.getIntegral());
    memberInfoArchivesVo.setCurrentIntegral(
        memberInfoArchivesVo.getCurrentIntegral() + memberInfoIntegralRecordVo.getIntegral());
    this.memberInfoAssetsService.update(memberInfoArchivesVo);
  }

  @Override
  public MemberInfoIntegralRecordAPIVo initData(String memberCode) {
    MemberInfoIntegralRecordAPIVo vo = new MemberInfoIntegralRecordAPIVo();
    MemberInfoAssetsVo memberInfoArchivesVo = memberInfoAssetsService
        .findDetailByMemberCode(memberCode);
    Validate.notNull(memberInfoArchivesVo, "会员资产异常");
    //获取当前剩余积分
    vo.setCurrentIntegral(memberInfoArchivesVo.getCurrentIntegral());
    List<MemberInfoIntegralRecord> list = memberInfoIntegralRecordRepository.lambdaQuery()
        .eq(MemberInfoIntegralRecord::getMemberCode, memberCode)
        .orderByDesc(MemberInfoIntegralRecord::getOptTime)
        .list();
    if (list.isEmpty()) {
      return vo;
    }
    //查询系统设置的积分有效期
    SystemSettingsDetail valueByKey = systemSettingsDetailService
        .findValueByKey(SystemSettingsDetailEnum.VALIDITY_PERIOD_OF_POINTS.getValue());
    if (StringUtils.isBlank(valueByKey.getTypeValue())) {
      valueByKey.setTypeValue("10000");
    }
    String periodOfValidity = valueByKey.getTypeValue();
    int i = Integer.parseInt(periodOfValidity);
    SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    Calendar calendar = Calendar.getInstance();
    //获取当前时间
    Date currentTime = calendar.getTime();
    //获取即将过期的日期区间
    calendar.setTime(currentTime);
    calendar.add(Calendar.MONTH, -i);
    Date timeOpen = calendar.getTime();
    calendar.add(Calendar.DATE, 30);
    Date timeClose = calendar.getTime();
    //获取近30天即将过期的积分总和
    int sum = memberInfoIntegralRecordRepository
        .sumResidualIntegral(memberCode, timeOpen, timeClose);
    vo.setExpiringSoon(sum);
    return vo;
  }

  @Override
  @DynamicTaskService(cornExpression = "0 0 0 1 * ?", taskDesc = "每月1号查询当前会员本月有没有过期的积分")
  public void expiringSoon() {
    String memberCode = userSearchHelper.getMemberLogin().getMemberCode();
    //查询系统设置的积分有效期
    SystemSettingsDetail valueByKey = systemSettingsDetailService
        .findValueByKey(SystemSettingsDetailEnum.VALIDITY_PERIOD_OF_POINTS.getValue());
    if (StringUtils.isBlank(valueByKey.getTypeValue())) {
      valueByKey.setTypeValue("10000");
    }
    String periodOfValidity = valueByKey.getTypeValue();
    int i = Integer.parseInt(periodOfValidity);
    // 获取当前日期
    Calendar calendar = Calendar.getInstance();
    // 获取本月的年份和月份
    int year = calendar.get(Calendar.YEAR);
    int month = calendar.get(Calendar.MONTH);
    // 设置日期为本月的第一天
    calendar.set(year, month, 1);
    Date firstDayOfMonth = calendar.getTime();
    // 设置日期为本月的最后一天
    calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
    Date lastDayOfMonth = calendar.getTime();
    //获取本月即将过期的积分总和
    int sum = memberInfoIntegralRecordRepository
        .sumResidualIntegral(memberCode, firstDayOfMonth, lastDayOfMonth);
    if (sum>0){
      AppletPopupNotifyVo appletPopupNotifyVo = new AppletPopupNotifyVo();
      appletPopupNotifyVo.setMemberCode(memberCode);
      appletPopupNotifyVo.setTriggerType(TriggerTypeEnum.INTEGRAL.getCode());
      appletPopupNotifyVo.setTriggerConditions(TriggerConditionEnum.INTEGRAL_EXPIRED_MOTH.getCode());
      appletPopupNotifyService.addOne(appletPopupNotifyVo);
    }
  }

  /**
   * 后台统计积分
   *
   * @return
   */
  @Override
  public MemberIntegralCountVo countIntegralRecord() {
    SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM");
    String currentTime = dateFormat.format(new Date(System.currentTimeMillis()));
    List<String> addType = Arrays.asList("add");
    List<String> deductionType = Arrays.asList("pay", "deduction");
    int totalAddIntegral = this.memberInfoIntegralRecordRepository
        .sumIntegralByTypeAndTime(addType, null);
    int currentMonthAddIntegral = this.memberInfoIntegralRecordRepository
        .sumIntegralByTypeAndTime(addType, currentTime);
    int totalDeductionIntegral = this.memberInfoIntegralRecordRepository
        .sumIntegralByTypeAndTime(deductionType, null);
    int currentMonthDeductionIntegral = this.memberInfoIntegralRecordRepository
        .sumIntegralByTypeAndTime(deductionType, currentTime);
    MemberIntegralCountVo result = new MemberIntegralCountVo();
    result.setTotalAddIntegral(totalAddIntegral);
    result.setCurrentMonthAddIntegral(currentMonthAddIntegral);
    result.setTotalDeductionIntegral(totalDeductionIntegral);
    result.setCurrentMonthDeductionIntegral(currentMonthDeductionIntegral);
    return result;
  }

  private void createValidation(MemberInfoIntegralRecordVo memberInfoIntegralRecordVo) {
    this.validation(memberInfoIntegralRecordVo);
  }

  private void updateValidation(MemberInfoIntegralRecordVo memberInfoIntegralRecordVo) {
    this.validation(memberInfoIntegralRecordVo);
  }

  private void validation(MemberInfoIntegralRecordVo memberInfoIntegralRecordVo) {
    Validate.notBlank(memberInfoIntegralRecordVo.getMemberCode(), "会员编码不能为空");
    Validate.notBlank(memberInfoIntegralRecordVo.getIntegralCode(), "流水编码不能为空");
    Validate.notBlank(memberInfoIntegralRecordVo.getSource(), "来源不能为空");
    Validate.notNull(memberInfoIntegralRecordVo.getIntegral(), "操作积分不能为空");
  }

}
