package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.member.business.member.local.entity.MemberInfoReceiptRecordDetail;
import com.biz.crm.member.business.member.local.repository.MemberInfoReceiptRecordDetailRepository;
import com.biz.crm.member.business.member.local.service.MemberInfoReceiptRecordDetailService;
import com.biz.crm.member.business.member.sdk.vo.GrowRecordAPIVo;
import com.biz.crm.member.business.member.sdk.vo.MemberInfoReceiptRecordDetailVo;
import com.biz.crm.member.business.member.sdk.vo.MemberInfoReceiptRecordVo;
import com.biz.crm.member.business.member.local.entity.MemberInfoReceiptRecord;
import com.biz.crm.member.business.member.local.repository.MemberInfoReceiptRecordRepository;
import com.biz.crm.member.business.member.local.service.MemberInfoReceiptRecordService;
import com.biz.crm.member.business.member.sdk.dto.MemberInfoReceiptRecordPaginationDto;
import com.biz.crm.member.business.member.sdk.event.MemberInfoReceiptRecordEventListener;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 会员签收记录(MemberInfoReceiptRecord)表服务实现类
 *
 * @author zouhs
 * @date 2023-06-28 16:55:37
 */
@Slf4j
@Service("memberInfoReceiptRecordService")
public class MemberInfoReceiptRecordServiceImpl implements MemberInfoReceiptRecordService {

  @Autowired(required = false)
  private MemberInfoReceiptRecordRepository memberInfoReceiptRecordRepository;

  @Autowired(required = false)
  private List<MemberInfoReceiptRecordEventListener> eventListeners;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  private MemberInfoReceiptRecordDetailRepository memberInfoReceiptRecordDetailRepository;

  @Autowired
  private MemberInfoReceiptRecordDetailService memberInfoReceiptRecordDetailService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<MemberInfoReceiptRecordVo> findByConditions(Pageable pageable,
      MemberInfoReceiptRecordPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new MemberInfoReceiptRecordPaginationDto());
    Page<MemberInfoReceiptRecordVo> page = new Page<>(pageable.getPageNumber(),
        pageable.getPageSize());
    return this.memberInfoReceiptRecordRepository.findByConditions(page, dto);
  }

  @Override
  public MemberInfoReceiptRecordVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    MemberInfoReceiptRecord memberInfoReceiptRecord = this.memberInfoReceiptRecordRepository
        .findById(id);
    MemberInfoReceiptRecordVo memberInfoReceiptRecordVo = this.nebulaToolkitService
        .copyObjectByWhiteList(memberInfoReceiptRecord, MemberInfoReceiptRecordVo.class,
            HashSet.class, LinkedList.class);
    return memberInfoReceiptRecordVo;
  }

  @Override
  @Transactional
  public MemberInfoReceiptRecordVo create(MemberInfoReceiptRecordVo memberInfoReceiptRecordVo) {
    this.createValidation(memberInfoReceiptRecordVo);
    MemberInfoReceiptRecord memberInfoReceiptRecord = this.nebulaToolkitService
        .copyObjectByWhiteList(memberInfoReceiptRecordVo, MemberInfoReceiptRecord.class,
            HashSet.class, LinkedList.class);
    memberInfoReceiptRecord.setTenantCode(TenantUtils.getTenantCode());
    memberInfoReceiptRecord.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    memberInfoReceiptRecord.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.memberInfoReceiptRecordRepository.saveOrUpdate(memberInfoReceiptRecord);
    // 处理明细
    if(CollectionUtils.isNotEmpty(memberInfoReceiptRecordVo.getList())){
      for (MemberInfoReceiptRecordDetailVo memberInfoReceiptRecordDetailVo : memberInfoReceiptRecordVo
          .getList()) {
        memberInfoReceiptRecordDetailVo.setReceiptCode(memberInfoReceiptRecord.getReceiptCode());
      }
      memberInfoReceiptRecordDetailService.batchSave(memberInfoReceiptRecordVo.getList());
    }
    return memberInfoReceiptRecordVo;
  }

  @Override
  @Transactional
  public MemberInfoReceiptRecordVo update(MemberInfoReceiptRecordVo memberInfoReceiptRecordVo) {
    this.updateValidation(memberInfoReceiptRecordVo);
    String currentId = memberInfoReceiptRecordVo.getId();
    MemberInfoReceiptRecord current = memberInfoReceiptRecordRepository.findById(currentId);
    current = Validate.notNull(current, "修改信息不存在");
    current = this.nebulaToolkitService
        .copyObjectByWhiteList(memberInfoReceiptRecordVo, MemberInfoReceiptRecord.class,
            HashSet.class, LinkedList.class);
    this.memberInfoReceiptRecordRepository.saveOrUpdate(current);
    return memberInfoReceiptRecordVo;
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.memberInfoReceiptRecordRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.memberInfoReceiptRecordRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.memberInfoReceiptRecordRepository.updateDelFlagByIds(ids);
  }

  /**
   * 根据单据编码获取签收产品明细
   * @param receiptCode
   * @return
   */
  @Override
  public List<MemberInfoReceiptRecordDetailVo> findDetailByReceiptCode(String receiptCode) {
    LambdaQueryWrapper<MemberInfoReceiptRecordDetail> queryDetail = new LambdaQueryWrapper<>();
    queryDetail.eq(MemberInfoReceiptRecordDetail::getReceiptCode, receiptCode)
        .eq(MemberInfoReceiptRecordDetail::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .orderByDesc(MemberInfoReceiptRecordDetail::getCreateTime);
    List<MemberInfoReceiptRecordDetail> details = this.memberInfoReceiptRecordDetailRepository
        .list(queryDetail);
    Collection<MemberInfoReceiptRecordDetailVo> result = this.nebulaToolkitService
        .copyCollectionByWhiteList(details,
            MemberInfoReceiptRecordDetail.class,
            MemberInfoReceiptRecordDetailVo.class,
            HashSet.class,
            LinkedList.class);
    return (List<MemberInfoReceiptRecordDetailVo>) result;
  }

  private void createValidation(MemberInfoReceiptRecordVo memberInfoReceiptRecordVo) {
    this.validation(memberInfoReceiptRecordVo);
  }

  private void updateValidation(MemberInfoReceiptRecordVo memberInfoReceiptRecordVo) {
    this.validation(memberInfoReceiptRecordVo);
  }

  private void validation(MemberInfoReceiptRecordVo memberInfoReceiptRecordVo) {
    Validate.notBlank(memberInfoReceiptRecordVo.getReceiptCode(), "单据流水号不能为空");
    Validate.notBlank(memberInfoReceiptRecordVo.getCustomerCode(), "发货单位编码不能为空");
    Validate.notBlank(memberInfoReceiptRecordVo.getCustomerName(), "发货单位不能为空");
    Validate.notBlank(memberInfoReceiptRecordVo.getTerminalCode(), "收货单位编码不能为空");
    Validate.notBlank(memberInfoReceiptRecordVo.getTerminalName(), "收货单位不能为空");
    Validate.notBlank(memberInfoReceiptRecordVo.getOrgCode(), "组织编码不能为空");
    Validate.notBlank(memberInfoReceiptRecordVo.getOrgName(), "组织名称不能为空");
    Validate.notNull(memberInfoReceiptRecordVo.getDocumentTime(), "单据时间不能为空");
    Validate.notNull(memberInfoReceiptRecordVo.getReceiptTime(), "签收时间不能为空");
    Validate.notBlank(memberInfoReceiptRecordVo.getReceiptAddr(), "签收地点不能为空");
    Validate.notNull(memberInfoReceiptRecordVo.getGetGrow(), "获得成长值不能为空");
  }
}
