package com.biz.crm.member.business.member.local.service.internal;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.member.business.member.local.helper.UserSearchHelper;
import com.biz.crm.member.business.member.local.service.MemberInfoIntegralRecordService;
import com.biz.crm.member.business.member.local.service.SignInRuleService;
import com.biz.crm.member.business.member.sdk.enums.IntegralSourceEnum;
import com.biz.crm.member.business.member.sdk.vo.MemberInfoIntegralRecordVo;
import com.biz.crm.member.business.member.sdk.vo.MemberInfoSignVo;
import com.biz.crm.member.business.member.local.entity.MemberInfoSign;
import com.biz.crm.member.business.member.local.repository.MemberInfoSignRepository;
import com.biz.crm.member.business.member.local.service.MemberInfoSignService;
import com.biz.crm.member.business.member.sdk.dto.MemberInfoSignPaginationDto;
import com.biz.crm.member.business.member.sdk.event.MemberInfoSignEventListener;
import com.biz.crm.member.business.member.sdk.vo.MemberSignCountVo;
import com.biz.crm.member.business.member.sdk.vo.MemberSignRecordVo;
import com.biz.crm.member.business.member.sdk.vo.SignInRuleDetailVo;
import com.biz.crm.member.business.member.sdk.vo.SignInRuleVo;
import com.biz.crm.member.business.member.sdk.vo.SignInfoVo;
import com.biz.crm.member.business.member.sdk.vo.SignResultVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 会员签到表(MemberInfoSign)表服务实现类
 *
 * @author zouhs
 * @date 2023-06-27 13:44:50
 */
@Slf4j
@Service("memberInfoSignService")
public class MemberInfoSignServiceImpl implements MemberInfoSignService {
  
  @Autowired(required = false)
  private MemberInfoSignRepository memberInfoSignRepository;

  @Autowired(required = false)
  private List<MemberInfoSignEventListener> eventListeners;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Autowired
  private UserSearchHelper userSearchHelper;

  @Autowired
  private SignInRuleService signInRuleService;

  @Autowired
  private MemberInfoIntegralRecordService memberInfoIntegralRecordService;

  @Override
  public Page<MemberInfoSignVo> findByConditions(Pageable pageable, MemberInfoSignPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new MemberInfoSignPaginationDto());
    Page<MemberInfoSignVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.memberInfoSignRepository.findByConditions(page, dto);
  }

  @Override
  public MemberInfoSignVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    MemberInfoSign memberInfoSign = this.memberInfoSignRepository.findById(id);
    MemberInfoSignVo memberInfoSignVo = this.nebulaToolkitService.copyObjectByWhiteList(memberInfoSign, MemberInfoSignVo.class, HashSet.class, LinkedList.class);
    return memberInfoSignVo;
  }
  
  @Override
  @Transactional
  public MemberInfoSignVo create(MemberInfoSignVo memberInfoSignVo) {
    this.createValidation(memberInfoSignVo);
    MemberInfoSign memberInfoSign = this.nebulaToolkitService.copyObjectByWhiteList(memberInfoSignVo, MemberInfoSign.class, HashSet.class, LinkedList.class);
    memberInfoSign.setTenantCode(TenantUtils.getTenantCode());
    memberInfoSign.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    memberInfoSign.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.memberInfoSignRepository.saveOrUpdate(memberInfoSign);
    // 发送通知
    // TODO 发送通知
    return memberInfoSignVo;
  }

  @Override
  @Transactional
  public MemberInfoSignVo update(MemberInfoSignVo memberInfoSignVo) {
    this.updateValidation(memberInfoSignVo);
    String currentId = memberInfoSignVo.getId();
    MemberInfoSign current = memberInfoSignRepository.findById(currentId);
    current = Validate.notNull(current, "修改信息不存在");
    current = this.nebulaToolkitService.copyObjectByWhiteList(memberInfoSignVo, MemberInfoSign.class, HashSet.class, LinkedList.class);
    this.memberInfoSignRepository.saveOrUpdate(current);
    // 发送修改通知
    // TODO 发送通知
    return memberInfoSignVo;
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.memberInfoSignRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.memberInfoSignRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
  }
  
  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.memberInfoSignRepository.updateDelFlagByIds(ids);
  }

  @Override
  public MemberSignCountVo signCountNum() {
    return this.memberInfoSignRepository.signCountNum();
  }

  @Override
  public MemberSignRecordVo querySignCountNum(
      MemberInfoSignPaginationDto memberInfoSignPaginationDto) {
    return this.memberInfoSignRepository.querySignCountNum(memberInfoSignPaginationDto);
  }

  /**
   * 小程序，获取当前用户本月的签到记录
   * @return
   */
  @Override
  public List<MemberInfoSignVo> listCurrentUserMemberInfoSign(String dateTime) {
    Validate.notBlank(dateTime, "时间不能为空");
    String memberCode = this.userSearchHelper.getMemberLogin().getMemberCode();
    // 根据签到时间升序获取当前用户的指定月份的签到信息
    List<MemberInfoSign> memberInfoSignList = this.memberInfoSignRepository
        .listMemberInfoSign(dateTime, memberCode, true);
    Collection<MemberInfoSignVo> result = this.nebulaToolkitService
        .copyCollectionByWhiteList(memberInfoSignList, MemberInfoSign.class, MemberInfoSignVo.class,
            HashSet.class, LinkedList.class);
    return (List<MemberInfoSignVo>) result;
  }

  /**
   * 小程序，获取当前登录用户本月的签到信息
   * @return
   */
  @Override
  public SignInfoVo getSignDetail(String dateTime) {
    Validate.notBlank(dateTime, "时间不能为空");
    String memberCode = this.userSearchHelper.getMemberLogin().getMemberCode();
    // 根据签到时间降序获取当前用户的指定月份的签到信息
    List<MemberInfoSign> memberInfoSignList = this.memberInfoSignRepository
        .listMemberInfoSign(dateTime, memberCode, false);
    SignInfoVo result = new SignInfoVo();
    // 计算连续签到天数、今日是否已签到
    this.generateContinueDay(result, memberInfoSignList);
    // 计算已获取到的连续签到奖励，以及下一个奖励剩余签到数、最多额外获得
    this.generateSignRuleInfo(result, memberInfoSignList);
    return result;
  }

  /**
   * 小程序签到
   * @return
   */
  @Transactional
  @Override
  public SignResultVo appSign() {
    // 签到校验
    this.signValidate();
    /**
     * 1、获取签到规则
     * 2、判断是否已达年上限
     * 3、判断本月连续签到奖励是否已领取完毕
     * 4、判断是否触发连续签到奖励
     *  4.1 没有触发连续签到奖励
     * 5、触发了连续签到奖励
     * 6、判断本月连续签到奖励是否已领取完毕
     */

    // 第1步======================
    // 获取签到规则
    SignInRuleVo signRule = this.signInRuleService.findDetail();
    // 每次签到获得积分
    Integer everyIntegral = signRule.getEveryIntegral();
    List<SignInRuleDetailVo> ruleDetailList = signRule.getList();
    Integer yearIntegralLimit = signRule.getYearIntegralLimit();
    // 获取用户本年获得的签到积分
    String memberCode = userSearchHelper.getMemberLogin().getMemberCode();
    Integer userYearIntegral = this.memberInfoSignRepository.countUserYearIntegral(memberCode);
    // 年限剩余积分
    int remainIntegral = yearIntegralLimit - userYearIntegral;
    SignResultVo result = new SignResultVo();
    MemberInfoSignVo memberInfoSign = new MemberInfoSignVo();
    memberInfoSign.setMemberCode(memberCode);
    memberInfoSign.setSignTime(new Date(System.currentTimeMillis()));
    // 第2步=======================
    // 本年是否已达上限
    Boolean yearLimit = userYearIntegral >= yearIntegralLimit;
    if (yearLimit){
      memberInfoSign.setIntegral(0);
      this.create(memberInfoSign);
      result.setYearLimit(true);
      return result;
    }

    // 第3步=========================
    // 获取本月已获得额外积分的签到信息
    SimpleDateFormat format = new SimpleDateFormat("yyyy-MM");
    String monthTime = format.format(new Date(System.currentTimeMillis()));
    List<MemberInfoSign> memberInfoSigns = this.memberInfoSignRepository
        .listExtendIntegralSign(monthTime, memberCode);
    SignInRuleDetailVo ruleDetailVo = this
        .computedDetailRuleArrived(ruleDetailList, memberInfoSigns);

    MemberInfoIntegralRecordVo memberIntegralRecord = new MemberInfoIntegralRecordVo();
    memberIntegralRecord.setMemberCode(memberCode);
    memberIntegralRecord.setType(IntegralSourceEnum.DAY_SIGN.getType());
    memberIntegralRecord.setSource(IntegralSourceEnum.DAY_SIGN.getValue());
    // 已领取月额外奖励
    if (ObjectUtils.isEmpty(ruleDetailVo)) {
      int realIntegral = remainIntegral >= everyIntegral ? everyIntegral : remainIntegral;
      memberInfoSign.setIntegral(realIntegral);
      this.create(memberInfoSign);
      result.setObtainIntegral(realIntegral);
      result.setMonthExtend(true);
      memberIntegralRecord.setIntegral(realIntegral);
      this.memberInfoIntegralRecordService.commonInternal(memberIntegralRecord);
      return result;
    }
    // 第4步===================
    List<MemberInfoSign> monthSignInfo = this.memberInfoSignRepository
        .listMemberInfoSign(monthTime, memberCode, false);
    SignInfoVo continueDayCount = new SignInfoVo();
    this.generateContinueDay(continueDayCount, monthSignInfo);
    Integer continueSignDay = continueDayCount.getContinueSignDay();
    Integer linkDayNum = ruleDetailVo.getLinkDayNum();
    Integer linkIntegralNum = ruleDetailVo.getLinkIntegralNum();
    // 没有触发额外奖励
    if ((continueSignDay+1) != linkDayNum){
      // 第4.1步==================
      int realIntegral = remainIntegral >= everyIntegral ? everyIntegral : remainIntegral;
      memberInfoSign.setIntegral(realIntegral);
      this.create(memberInfoSign);
      result.setObtainIntegral(realIntegral);
      result.setNextContinueDay(linkDayNum - (continueSignDay+1));
      result.setNextObtainIntegral(linkIntegralNum);
      memberIntegralRecord.setIntegral(realIntegral);
      this.memberInfoIntegralRecordService.commonInternal(memberIntegralRecord);
      return result;
    }
    // 第5步====================
    // 触发了额外奖励
    int currentIntegral = linkIntegralNum + everyIntegral;
    int realIntegral = remainIntegral >= currentIntegral ? currentIntegral : remainIntegral;
    memberInfoSign.setIntegral(realIntegral);
    this.create(memberInfoSign);
    result.setObtainIntegral(realIntegral);
    ruleDetailVo.setArrived(true);
    SignInRuleDetailVo nextRuleDetail = this.getNextRuleDetail(ruleDetailList);
    // 第6步===================
    // 达到了月额外领取上限
    if (ObjectUtils.isEmpty(nextRuleDetail)){
      result.setMonthExtend(true);
      memberIntegralRecord.setIntegral(realIntegral);
      this.memberInfoIntegralRecordService.commonInternal(memberIntegralRecord);
      return result;
    }
    result.setNextContinueDay(nextRuleDetail.getLinkDayNum());
    result.setNextObtainIntegral(nextRuleDetail.getLinkIntegralNum());
    memberIntegralRecord.setIntegral(realIntegral);
    this.memberInfoIntegralRecordService.commonInternal(memberIntegralRecord);
    return result;
  }

  /**
   * 签到校验
   */
  private void signValidate() {
    SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
    String dateTime = format.format(new Date(System.currentTimeMillis()));
    String memberCode = this.userSearchHelper.getMemberLogin().getMemberCode();
    LambdaQueryWrapper<MemberInfoSign> queryWrapper = new LambdaQueryWrapper<>();
    queryWrapper.likeRight(MemberInfoSign::getSignTime, dateTime)
        .eq(MemberInfoSign::getMemberCode, memberCode)
        .eq(MemberInfoSign::getDelFlag, DelFlagStatusEnum.NORMAL.getCode())
        .last("limit 1");
    MemberInfoSign currentSignInfo = this.memberInfoSignRepository.getOne(queryWrapper);
    Validate.isTrue(ObjectUtils.isEmpty(currentSignInfo), "一天只能签到一次");
  }

  /**
   * 获取下一次月额外奖励的详细规则
   * @param ruleDetailList
   * @return
   */
  private SignInRuleDetailVo getNextRuleDetail(List<SignInRuleDetailVo> ruleDetailList) {
    SignInRuleDetailVo result = ruleDetailList
        .stream()
        .filter(SignInRuleDetailVo::getArrived)
        .findFirst().get();
    return result;
  }

  /**
   * 计算已获取到的额外奖励
   * @param ruleDetailList
   * @param extendIntegralSignList
   */
  private SignInRuleDetailVo computedDetailRuleArrived(List<SignInRuleDetailVo> ruleDetailList,
      List<MemberInfoSign> extendIntegralSignList) {
    ruleDetailList.sort((obj1, obj2)->
        obj1.getLinkDayNum().compareTo(obj2.getLinkDayNum())
    );
    Map<Integer, Integer> extendMap = extendIntegralSignList.stream().collect(
        Collectors.toMap(MemberInfoSign::getContinueDay, MemberInfoSign::getExtendObtain));
    for (SignInRuleDetailVo ruleDetail : ruleDetailList) {
      Integer linkDayNum = ruleDetail.getLinkDayNum();
      Integer linkIntegralNum = ruleDetail.getLinkIntegralNum();
      if (extendMap.containsKey(linkDayNum)){
        Integer integer = extendMap.get(linkDayNum);
        ruleDetail.setArrived(linkIntegralNum == integer);
      }else {
        return ruleDetail;
      }
    }
    return null;
  }

  /**
   * 计算已获取到的连续签到奖励，以及下一个奖励剩余签到数、最多额外获得
   * @param result
   * @param memberInfoSignList
   */
  private void generateSignRuleInfo(SignInfoVo result, List<MemberInfoSign> memberInfoSignList) {
    /**
     * 1、获取已经获得额外奖励的签到信息
     * 2、获取签到规则的配置信息
     * 3、将签到规则明细已连续签到天数升序排序，循环遍历签到规则明细
     *  3.1 判断已获得奖励中是否已经包含了该明细规则
     *  3.2 计算下一次奖励的信息
     * 4、计算额外可获得积分的总合
     */
    // 第1步===================
    List<MemberInfoSign> extendObtainSignList = memberInfoSignList
        .stream()
        .filter(memberInfoSign ->
            ObjectUtils.isNotEmpty(memberInfoSign.getContinueDay()) &&
                ObjectUtils.isNotEmpty(memberInfoSign.getExtendObtain())
        ).collect(Collectors.toList());
    Map<Integer, Integer> arrivedSignMap = extendObtainSignList.stream()
        .collect(Collectors.toMap(MemberInfoSign::getContinueDay, MemberInfoSign::getExtendObtain));
    // 第2步==================
    SignInRuleVo signRule = this.signInRuleService.findDetail();
    result.setSignInRuleVo(signRule);
    List<SignInRuleDetailVo> ruleDetailsList = signRule.getList();
    // 第3步=================
    ruleDetailsList.sort((obj1, obj2)->
        obj1.getLinkDayNum().compareTo(obj2.getLinkDayNum())
    );
    for (SignInRuleDetailVo ruleDetail : ruleDetailsList) {
      Integer linkDayNum = ruleDetail.getLinkDayNum();
      Integer linkIntegralNum = ruleDetail.getLinkIntegralNum();
      if (arrivedSignMap.containsKey(linkDayNum)) {
        // 3.1 =================
        ruleDetail.setArrived(arrivedSignMap.get(linkDayNum) == linkIntegralNum);
      }else {
        // 3.2 =================
        Integer continueSignDay = result.getContinueSignDay();
        // 下一次获取额外奖励的剩余天数
        result.setNextRemainSignDay(linkDayNum - continueSignDay);
        // 下一次获取额外奖励的积分
        result.setLastObtain(BigDecimal.valueOf(linkIntegralNum));
        break;
      }
    }
    // 第4步===================
    ruleDetailsList.stream().map(SignInRuleDetailVo::getLinkIntegralNum).reduce(Integer::sum).ifPresent(extendObtain->{
      result.setMaxExtendObtain(BigDecimal.valueOf(extendObtain));
    });
    result.setSignInRuleDetailVos(ruleDetailsList);
  }

  /**
   * 计算连续签到天数、今日是否已签到
   * @param result
   * @param signList
   */
  private void generateContinueDay(SignInfoVo result, List<MemberInfoSign> signList){
    /**
     * 1、若本月的签到信息为空，则连续签到天数为0
     * 2、若最近的签到时间小于昨天，则连续签到天数为0
     * 3、遍历签到信息，若签到时间加上下标==最近的签到时间，则连续签到数+1，否则直接跳出循环
     * 注意：此方法的签到信息，一定是根据签到时间倒叙排序的
     */
    // 第1步===============
    if (CollectionUtils.isEmpty(signList)){
      result.setContinueSignDay(0);
      return;
    }
    Date lastSignTime = signList.get(0).getSignTime();
    int lastSignDay  = DateTime.of(lastSignTime).dayOfMonth();
    int currentDay = DateTime.of(System.currentTimeMillis()).dayOfMonth();
    // 填充今日是否已签到
    result.setTodaySign(lastSignDay == currentDay);
    // 第2步===============
    if (lastSignDay< currentDay - 1){
      result.setContinueSignDay(0);
      return;
    }
    // 第3步===============
    int continueDay = 0 ;
    for (int i = 0; i < signList.size(); i++) {
      MemberInfoSign memberInfoSign = signList.get(i);
      Date signTime = memberInfoSign.getSignTime();
      int signDay = DateTime.of(signTime).dayOfMonth();
      if ((signDay+i) == lastSignDay){
        continueDay++;
      }else {
        break;
      }
    }
    result.setContinueSignDay(continueDay);
  }

  private void createValidation(MemberInfoSignVo memberInfoSignVo) {
    // TODO 具体实现
    this.validation(memberInfoSignVo);
  }

  private void updateValidation(MemberInfoSignVo memberInfoSignVo) {
    // TODO 具体实现
    this.validation(memberInfoSignVo);
  }

  private void validation(MemberInfoSignVo memberInfoSignVo) {
    // TODO 具体实现
  }
}
