package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.local.entity.UuidEntity;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.GenerateCodeService;
import com.biz.crm.member.business.member.local.entity.MemberInfoTag;
import com.biz.crm.member.business.member.local.repository.MemberInfoTagRepository;
import com.biz.crm.member.business.member.local.service.MemberInfoTagService;
import com.biz.crm.member.business.member.sdk.dto.MemberInfoTagPaginationDto;
import com.biz.crm.member.business.member.sdk.event.MemberInfoTagEventListener;
import com.biz.crm.member.business.member.sdk.vo.MemberInfoTagVo;
import com.biz.crm.member.business.member.sdk.vo.MemberLabelUpdVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 会员标签(MemberInfoTag)表服务实现类
 *
 * @author zouhs
 * @date 2023-06-19 14:35:01
 */
@Slf4j
@Service("memberInfoTagService")
public class MemberInfoTagServiceImpl implements MemberInfoTagService {

  @Autowired(required = false)
  private MemberInfoTagRepository memberInfoTagRepository;

  @Autowired(required = false)
  private List<MemberInfoTagEventListener> eventListeners;

  @Autowired(required = false)
  private GenerateCodeService generateCodeService;

  @Autowired
  @Qualifier("nebulaToolkitService")
  private NebulaToolkitService nebulaToolkitService;

  @Override
  public Page<MemberInfoTagVo> findByConditions(Pageable pageable, MemberInfoTagPaginationDto dto) {
    pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
    dto = Optional.ofNullable(dto).orElse(new MemberInfoTagPaginationDto());
    Page<MemberInfoTagVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
    return this.memberInfoTagRepository.findByConditions(page, dto);
  }

  @Override
  public MemberInfoTagVo findDetailById(String id) {
    if (StringUtils.isBlank(id)) {
      return null;
    }
    MemberInfoTag memberInfoTag = this.memberInfoTagRepository.findById(id);
    MemberInfoTagVo memberInfoTagVo = this.nebulaToolkitService
        .copyObjectByWhiteList(memberInfoTag, MemberInfoTagVo.class, HashSet.class,
            LinkedList.class);
    return memberInfoTagVo;
  }

  @Override
  @Transactional
  public MemberInfoTagVo create(MemberInfoTagVo memberInfoTagVo) {
    this.createValidation(memberInfoTagVo);
    MemberInfoTag memberInfoTag = this.nebulaToolkitService
        .copyObjectByWhiteList(memberInfoTagVo, MemberInfoTag.class, HashSet.class,
            LinkedList.class);
    memberInfoTag.setTenantCode(TenantUtils.getTenantCode());
    memberInfoTag.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
    memberInfoTag.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
    this.memberInfoTagRepository.saveOrUpdate(memberInfoTag);
    // 发送通知
    // TODO 发送通知
    return memberInfoTagVo;
  }

  @Override
  @Transactional
  public MemberInfoTagVo update(MemberInfoTagVo memberInfoTagVo) {
    this.updateValidation(memberInfoTagVo);
    String currentId = memberInfoTagVo.getId();
    MemberInfoTag current = memberInfoTagRepository.findById(currentId);
    current = Validate.notNull(current, "修改信息不存在");
    current = this.nebulaToolkitService
        .copyObjectByWhiteList(memberInfoTagVo, MemberInfoTag.class, HashSet.class,
            LinkedList.class);
    this.memberInfoTagRepository.saveOrUpdate(current);
    // 发送修改通知
    // TODO 发送通知
    return memberInfoTagVo;
  }

  @Override
  @Transactional
  public void enableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.memberInfoTagRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
  }

  @Override
  @Transactional
  public void disableBatch(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.memberInfoTagRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
  }

  @Override
  @Transactional
  public void updateDelFlagByIds(List<String> ids) {
    Validate.isTrue(CollectionUtils.isNotEmpty(ids), "id集合不能为空");
    this.memberInfoTagRepository.updateDelFlagByIds(ids);
  }

  @Override
  public void memberLabelUpd(MemberLabelUpdVo memberLabelUpdVo) {
    Validate.notNull(memberLabelUpdVo, "入参不能为空");
    Validate.notBlank(memberLabelUpdVo.getMemberCode(), "会员编码不能为空");

    // 查询老的标签删除
    List<MemberInfoTag> list = this.memberInfoTagRepository.lambdaQuery()
        .eq(MemberInfoTag::getMemberCode, memberLabelUpdVo.getMemberCode()).list();
    if(CollectionUtils.isNotEmpty(list)){
      Set<String> ids = list.stream().map(UuidEntity::getId).collect(Collectors.toSet());
      this.memberInfoTagRepository.removeByIds(ids);
    }
    // 新增标签
    if(CollectionUtils.isNotEmpty(memberLabelUpdVo.getMemberInfoTagVo())){
      for (MemberInfoTagVo memberInfoTagVo : memberLabelUpdVo.getMemberInfoTagVo()) {
        memberInfoTagVo.setMemberCode(memberLabelUpdVo.getMemberCode());
        this.create(memberInfoTagVo);
      }
    }
  }

  @Override
  public List<MemberInfoTagVo> queryMemberLabel(String memberCode) {
    Validate.notBlank(memberCode, "会员编码不能为空");
    List<MemberInfoTag> list = this.memberInfoTagRepository.lambdaQuery()
        .eq(MemberInfoTag::getMemberCode, memberCode).list();
    if (CollectionUtils.isNotEmpty(list)) {
      return (List<MemberInfoTagVo>) this.nebulaToolkitService
          .copyCollectionByWhiteList(list, MemberInfoTag.class, MemberInfoTagVo.class,
              HashSet.class,
              ArrayList.class);
    }
    return Lists.newArrayList();
  }

  private void createValidation(MemberInfoTagVo memberInfoTagVo) {
    this.validation(memberInfoTagVo);
  }

  private void updateValidation(MemberInfoTagVo memberInfoTagVo) {
    this.validation(memberInfoTagVo);
  }

  private void validation(MemberInfoTagVo memberInfoTagVo) {
    Validate.notBlank(memberInfoTagVo.getMemberCode(),"会员编码不能为空");
    Validate.notBlank(memberInfoTagVo.getTagCode(),"标签编码不能为空");
    Validate.notBlank(memberInfoTagVo.getTagName(),"标签名称不能为空");
    Validate.notBlank(memberInfoTagVo.getTagType(),"标签类型不能为空");
  }
}
