package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.member.business.member.local.entity.*;
import com.biz.crm.member.business.member.local.repository.MessageManageRepository;
import com.biz.crm.member.business.member.local.repository.MessageNotifyRepository;
import com.biz.crm.member.business.member.local.repository.PopupNotifyRepository;
import com.biz.crm.member.business.member.local.service.MessageManageService;
import com.biz.crm.member.business.member.local.utils.AssertUtils;
import com.biz.crm.member.business.member.sdk.dto.*;
import com.biz.crm.member.business.member.sdk.enums.TriggerConditionEnum;
import com.biz.crm.member.business.member.sdk.enums.TriggerTypeEnum;
import com.biz.crm.member.business.member.sdk.vo.*;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import io.swagger.annotations.ApiModelProperty;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.text.SimpleDateFormat;
import java.util.*;

/**
 * 消息/弹窗实现类
 *
 * @Description
 * @Author monroe
 * @Date 2023/6/15 15:01
 **/
@Service
public class MessageManageServiceImpl implements MessageManageService {

    @Autowired
    private MessageManageRepository messageManageRepository;
    @Autowired
    private MessageNotifyRepository messageNotifyRepository;
    @Autowired
    private PopupNotifyRepository popupNotifyRepository;

    @Autowired
    private NebulaToolkitService nebulaToolkitService;
    @Autowired
    private RedisMutexService redisMutexService;

    /**
     * 消息管理编码redis key
     * 参数：租户
     */
    public static final String MMS_MESSAGE_MANAGE_CODE_REDIS_KEY = "mms:message:manage:code:index:%s";

    /**
     * 消息管理编码前缀
     */
    public static final String MMS_MESSAGE_MANAGE_CODE_PREFIX = "MM";

    /**
     * 生成编码
     */
    private String generateCode(String tenantCode) {
        String redisKey = String.format(MMS_MESSAGE_MANAGE_CODE_REDIS_KEY, tenantCode);
        String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        return StringUtils.join(MMS_MESSAGE_MANAGE_CODE_PREFIX, format.format(new Date()), index);
    }

    @Override
    public Page<MessageManagePageVo> findByConditions(Pageable pageable, MessageManagePageDto messageManageDto) {
        if (Objects.isNull(messageManageDto)) {
            messageManageDto = new MessageManagePageDto();
        }
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        Page<MessageManageVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.messageManageRepository.findByConditions(page, messageManageDto);
    }

    @Override
    public MessageManageVo findDetailById(String id) {
        if(StringUtils.isBlank(id)) {
            return null;
        }
        MessageManage messageManage = this.messageManageRepository.getById(id);
        Validate.notNull(messageManage,"消息配置不存在！");
        MessageManageVo messageManageVo = this.nebulaToolkitService.copyObjectByWhiteList(messageManage, MessageManageVo.class, HashSet.class, LinkedList.class);
        //消息通知
        MessageNotify messageNotify = this.messageNotifyRepository.lambdaQuery().eq(MessageNotify::getMessageCode, messageManage.getCode()).one();
        if(Objects.nonNull(messageNotify)) {
            MessageNotifyVo messageNotifyVo = this.nebulaToolkitService.copyObjectByWhiteList(messageNotify, MessageNotifyVo.class, HashSet.class, LinkedList.class);
            messageManageVo.setMessage(messageNotifyVo);
        }
        //弹窗
        PopupNotify popupNotify = this.popupNotifyRepository.lambdaQuery().eq(PopupNotify::getMessageCode, messageManage.getCode()).one();
        if(Objects.nonNull(popupNotify)) {
            PopupNotifyVo popupNotifyVo = this.nebulaToolkitService.copyObjectByWhiteList(popupNotify, PopupNotifyVo.class, HashSet.class, LinkedList.class);
            messageManageVo.setPopup(popupNotifyVo);
        }

        return messageManageVo;
    }

    @Transactional
    @Override
    public MessageManageVo create(MessageManageDto messageManageDto) {
        this.createValidate(messageManageDto);
        String tenantCode = TenantUtils.getTenantCode();
        messageManageDto.setCode(this.generateCode(tenantCode));
        messageManageDto.setTenantCode(tenantCode);
        messageManageDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        messageManageDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());

        MessageManage messageManage = this.nebulaToolkitService.copyObjectByWhiteList(messageManageDto, MessageManage.class, HashSet.class, LinkedList.class);
        this.messageManageRepository.saveOrUpdate(messageManage);

        MessageNotifyDto message = messageManageDto.getMessage();
        if(Objects.nonNull(message) && messageManage.getMessageNotify()) {
            message.setMessageCode(messageManage.getCode());
            message.setTenantCode(TenantUtils.getTenantCode());
            message.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            message.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            MessageNotify messageNotify = this.nebulaToolkitService.copyObjectByWhiteList(message, MessageNotify.class, HashSet.class, LinkedList.class);
            this.messageNotifyRepository.saveOrUpdate(messageNotify);
        }
        PopupNotifyDto popup = messageManageDto.getPopup();
        if(Objects.nonNull(popup) && messageManage.getPopupNotify()) {
            popup.setMessageCode(messageManage.getCode());
            popup.setTenantCode(TenantUtils.getTenantCode());
            popup.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            popup.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            PopupNotify popupNotify = this.nebulaToolkitService.copyObjectByWhiteList(popup, PopupNotify.class, HashSet.class, LinkedList.class);
            this.popupNotifyRepository.saveOrUpdate(popupNotify);
        }

        return this.nebulaToolkitService.copyObjectByWhiteList(messageManage,MessageManageVo.class,HashSet.class,LinkedList.class);
    }

    private void createValidate(MessageManageDto messageManageDto) {

        Validate.notNull(messageManageDto,"新增实体不能为空");
        Validate.notBlank(messageManageDto.getTriggerType(),"触发类型不能为空！");
//        Validate.notBlank(messageManageDto.getTriggerTypeName(),"触发类型名称不能为空！");
        Validate.notBlank(messageManageDto.getTriggerCondition(),"触发条件不能为空！");
//        Validate.notBlank(messageManageDto.getTriggerConditionName(),"触发条件名称不能为空！");
        Validate.notBlank(messageManageDto.getTriggerWeight(),"权重不能为空！");
        Validate.notNull(messageManageDto.getMessageNotify(),"是否消息通知不能为空！");
        Validate.notNull(messageManageDto.getPopupNotify(),"是否弹窗不能为空！");

        switch (TriggerConditionEnum.codeToEnum(messageManageDto.getTriggerCondition())) {
            //会员审批通过
            case MEMBER_APPROVAL_PASS:
                Validate.isTrue(!messageManageDto.getMessageNotify(),"会员审批通过不能开启消息通知");
                Validate.isTrue(!messageManageDto.getPopupNotify(),"会员审批通过不能开启弹窗");
                break;
                //未领取权益到期前30天
            case BENEFIT_NO_CLAIM_30:
            case BENEFIT_EXCHANGE_SUCCESS:
            case ORDER_EXPIRED_30:
            case ACTIVITY_APPROVAL_PASS:
            case ACTIVITY_REJECT:
            case ACTIVITY_SIGN_3:
                AssertUtils.isTrue(!messageManageDto.getPopupNotify(),"不能开启开屏弹窗");
                break;
        }

        //开启消息通知，校验相关参数
        if(messageManageDto.getMessageNotify()) {

            MessageNotifyDto message = messageManageDto.getMessage();
            Validate.notNull(message,"开启消息通知时，消息通知信息不能为空！");
            Validate.notBlank(message.getTitle(),"标题不能为空");
            Validate.notBlank(message.getContent(),"内容不能为空");

        }

        //开启弹窗，校验相关参数
        if(messageManageDto.getPopupNotify()) {

            PopupNotifyDto popup = messageManageDto.getPopup();
            Validate.notNull(popup,"开启弹窗时，弹窗信息不能为空！");
            Validate.notBlank(popup.getFrequency(),"频次不能为空");
            Validate.notBlank(popup.getSkipType(),"跳转不能为空");
//            Validate.notBlank(popup.getSkipTypeName(),"跳转名称不能为空");
//            Validate.notBlank(popup.getSkipUrl(),"跳转url不能为空");
//            Validate.notBlank(popup.getAppId(),"appid不能为空");
            Validate.notNull(popup.getStartDate(),"开始时间不能为空");
            Validate.notNull(popup.getEndDate(),"结束不能为空");
        }

    }

    @Transactional
    @Override
    public MessageManageVo update(MessageManageDto messageManageDto) {
        this.updateValidate(messageManageDto);

        MessageManage oldMessageManage = this.messageManageRepository.getById(messageManageDto.getId());
        Validate.notNull(oldMessageManage,"消息配置不存在");
        MessageManage messageManage = this.nebulaToolkitService.copyObjectByWhiteList(messageManageDto, MessageManage.class, HashSet.class, LinkedList.class);
        messageManage.setTenantCode(oldMessageManage.getTenantCode());
        messageManage.setEnableStatus(oldMessageManage.getEnableStatus());
        messageManage.setDelFlag(oldMessageManage.getDelFlag());
        this.messageManageRepository.saveOrUpdate(messageManage);


        QueryWrapper<MessageNotify> messageNotifyQueryWrapper = new QueryWrapper<>();
        messageNotifyQueryWrapper.eq("message_code",messageManage.getCode());
        messageNotifyQueryWrapper.eq("tenant_code",messageManage.getTenantCode());
        this.messageNotifyRepository.remove(messageNotifyQueryWrapper);

        QueryWrapper<PopupNotify> popupNotifyQueryWrapper = new QueryWrapper<>();
        popupNotifyQueryWrapper.eq("message_code",messageManage.getCode());
        popupNotifyQueryWrapper.eq("tenant_code",messageManage.getTenantCode());
        this.popupNotifyRepository.remove(popupNotifyQueryWrapper);

        MessageNotifyDto message = messageManageDto.getMessage();
        if(Objects.nonNull(message) && messageManageDto.getMessageNotify()) {
            message.setMessageCode(messageManage.getCode());
            message.setTenantCode(TenantUtils.getTenantCode());
            message.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            message.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            MessageNotify messageNotify = this.nebulaToolkitService.copyObjectByWhiteList(message, MessageNotify.class, HashSet.class, LinkedList.class);
            this.messageNotifyRepository.saveOrUpdate(messageNotify);
        }

        PopupNotifyDto popup = messageManageDto.getPopup();
        if(Objects.nonNull(popup) && messageManageDto.getPopupNotify()) {
            popup.setMessageCode(messageManage.getCode());
            popup.setTenantCode(TenantUtils.getTenantCode());
            popup.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
            popup.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
            PopupNotify popupNotify = this.nebulaToolkitService.copyObjectByWhiteList(popup, PopupNotify.class, HashSet.class, LinkedList.class);
            this.popupNotifyRepository.saveOrUpdate(popupNotify);
        }

        return this.nebulaToolkitService.copyObjectByWhiteList(messageManage,MessageManageVo.class,HashSet.class,LinkedList.class);
    }

    private void updateValidate(MessageManageDto messageManageDto) {

        Validate.notNull(messageManageDto,"编辑实体不能为空");
        Validate.notBlank(messageManageDto.getTriggerType(),"触发类型不能为空！");
//        Validate.notBlank(messageManageDto.getTriggerTypeName(),"触发类型名称不能为空！");
        Validate.notBlank(messageManageDto.getTriggerCondition(),"触发条件不能为空！");
//        Validate.notBlank(messageManageDto.getTriggerConditionName(),"触发条件名称不能为空！");
        Validate.notBlank(messageManageDto.getTriggerWeight(),"权重不能为空！");
        Validate.notNull(messageManageDto.getMessageNotify(),"是否消息通知不能为空！");
        Validate.notNull(messageManageDto.getPopupNotify(),"是否弹窗不能为空！");
        Validate.notBlank(messageManageDto.getId(),"编辑时，主键id不能为空");
        Validate.notBlank(messageManageDto.getCode(),"编辑时，编码不能为空");

        //开启消息通知，校验相关参数
        if(messageManageDto.getMessageNotify()) {

            MessageNotifyDto message = messageManageDto.getMessage();
            Validate.notNull(message,"开启消息通知时，消息通知信息不能为空！");
            Validate.notBlank(message.getTitle(),"标题不能为空");
            Validate.notBlank(message.getContent(),"内容不能为空");

        }

        //开启弹窗，校验相关参数
        if(messageManageDto.getPopupNotify()) {

            PopupNotifyDto popup = messageManageDto.getPopup();
            Validate.notNull(popup,"开启弹窗时，弹窗信息不能为空！");
            Validate.notBlank(popup.getFrequency(),"频次不能为空");
            Validate.notBlank(popup.getSkipType(),"跳转不能为空");
//            Validate.notBlank(popup.getSkipTypeName(),"跳转名称不能为空");
//            Validate.notBlank(popup.getSkipUrl(),"跳转url不能为空");
//            Validate.notBlank(popup.getAppId(),"appid不能为空");
            Validate.notNull(popup.getStartDate(),"开始时间不能为空");
            Validate.notNull(popup.getEndDate(),"结束不能为空");
        }

    }

    @Override
    public void enableBatch(List<String> ids) {

        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.messageManageRepository.lambdaUpdate().set(MessageManage::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                .in(MessageManage::getId,ids).update();

    }

    @Override
    public void disableBatch(List<String> ids) {

        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.messageManageRepository.lambdaUpdate().set(MessageManage::getEnableStatus, EnableStatusEnum.DISABLE.getCode())
                .in(MessageManage::getId,ids).update();

    }

    @Override
    public void deleteBatch(List<String> ids) {

        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.messageManageRepository.lambdaUpdate().set(MessageManage::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .in(MessageManage::getId,ids).update();

    }
}
