package com.biz.crm.member.business.member.local.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.member.business.member.local.entity.PrizeEntity;
import com.biz.crm.member.business.member.local.entity.ViewpagerEntity;
import com.biz.crm.member.business.member.local.observer.RafflePrizeObserver;
import com.biz.crm.member.business.member.local.repository.PrizeRepository;
import com.biz.crm.member.business.member.local.service.PrizeService;
import com.biz.crm.member.business.member.sdk.constants.PrizeRedisKeys;
import com.biz.crm.member.business.member.sdk.dto.PrizeDto;
import com.biz.crm.member.business.member.sdk.dto.ViewpagerDto;
import com.biz.crm.member.business.member.sdk.enums.PrizeInventoryEnum;
import com.biz.crm.member.business.member.sdk.enums.PrizeTypeEnum;
import com.biz.crm.member.business.member.sdk.vo.PrizeVo;
import com.biz.crm.member.business.member.sdk.vo.ViewpagerVo;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * TODO
 *
 * @Description
 * @Author monroe
 * @Date 2023/6/19 14:41
 **/
@Service
public class PrizeServiceImpl implements PrizeService {

    @Autowired
    private PrizeRepository prizeRepository;
    @Autowired
    private NebulaToolkitService nebulaToolkitService;
    @Autowired
    private RedisMutexService redisMutexService;
    @Autowired
    private RafflePrizeObserver rafflePrizeObserver;

    /**
     * 轮播图编码redis key
     * 参数：租户
     */
    public static final String MMS_PRIZE_CODE_REDIS_KEY = "mms:prize:code:index:%s";

    /**
     * 轮播图编码前缀
     */
    public static final String MMS_PRIZE_CODE_PREFIX = "JP";

    /**
     * 生成编码
     */
    private String generateCode(String tenantCode) {
        String redisKey = String.format(MMS_PRIZE_CODE_REDIS_KEY, tenantCode);
        String index = redisMutexService.getAndIncrement(redisKey, 1, 6);
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        return StringUtils.join(MMS_PRIZE_CODE_PREFIX, format.format(new Date()), index);
    }

    @Override
    public Page<PrizeVo> findByConditions(Pageable pageable, PrizeDto prizeDto) {
        if (Objects.isNull(prizeDto)) {
            prizeDto = new PrizeDto();
        }
        pageable = Optional.ofNullable(pageable).orElse(PageRequest.of(0, 50));
        Page<PrizeVo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        return this.prizeRepository.findByConditions(page, prizeDto);
    }

    @Override
    public PrizeVo findDetailById(String id) {

        if(StringUtils.isBlank(id)) {
            return null;
        }
        PrizeVo prizeVo = this.prizeRepository.findDetailById(id);

        //投放数量 = 总数量 - 库存数量
        if(PrizeInventoryEnum.EXITS.getCode().equals(prizeVo.getInventory())) {
            prizeVo.setPutNum(prizeVo.getTotalNum().subtract(prizeVo.getInventoryNum()));
        }
        return prizeVo;
    }

    @Override
    public PrizeVo create(PrizeDto prizeDto) {
        this.createValidate(prizeDto);
        prizeDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        prizeDto.setCode(this.generateCode(TenantUtils.getTenantCode()));
        prizeDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        prizeDto.setTenantCode(TenantUtils.getTenantCode());
        PrizeEntity prizeEntity = this.nebulaToolkitService.copyObjectByWhiteList(prizeDto, PrizeEntity.class, HashSet.class, LinkedList.class);
        this.prizeRepository.saveOrUpdate(prizeEntity);
        return this.nebulaToolkitService.copyObjectByWhiteList(prizeEntity,PrizeVo.class,HashSet.class,LinkedList.class);
    }

    private void createValidate(PrizeDto prizeDto) {
        Validate.notNull(prizeDto," 新增请求实体不能为空");
        Validate.notBlank(prizeDto.getPrizeType(),"奖品类型不能为空");
        Validate.notBlank(prizeDto.getPrizeName(),"奖品名称不能为空");
//        Validate.notBlank(prizeDto.getUnitCode(),"单位不能为空");
        //谢谢参与不校验库存
        if(!PrizeTypeEnum.THANKS.getCode().equals(prizeDto.getPrizeType())) {
            Validate.notNull(prizeDto.getInventory(), "是否库存不能为空");
            if(PrizeInventoryEnum.EXITS.getCode().equals(prizeDto.getInventory())) {
                Validate.notNull(prizeDto.getInventoryNum(),"库存数量不能为空");
                prizeDto.setTotalNum(prizeDto.getInventoryNum());
                prizeDto.setPutNumHit(BigDecimal.ZERO);
            }
        }


    }

    @Override
    public PrizeVo update(PrizeDto prizeDto) {
        this.updateValidate(prizeDto);
        PrizeEntity oldPrize = this.prizeRepository.getById(prizeDto.getId());
        if(PrizeInventoryEnum.WITH_OUT.getCode().equals(oldPrize.getInventory())
                && PrizeInventoryEnum.EXITS.getCode().equals(prizeDto.getInventory())) {
            prizeDto.setTotalNum(prizeDto.getInventoryNum());
        }
        prizeDto.setEnableStatus(EnableStatusEnum.ENABLE.getCode());
        prizeDto.setDelFlag(DelFlagStatusEnum.NORMAL.getCode());
        prizeDto.setTenantCode(TenantUtils.getTenantCode());
        PrizeEntity prizeEntity = this.nebulaToolkitService.copyObjectByWhiteList(prizeDto, PrizeEntity.class, HashSet.class, LinkedList.class);
        this.prizeRepository.saveOrUpdate(prizeEntity);
        return this.nebulaToolkitService.copyObjectByWhiteList(prizeEntity,PrizeVo.class,HashSet.class,LinkedList.class);
    }

    private void updateValidate(PrizeDto prizeDto) {
        Validate.notNull(prizeDto," 新增请求实体不能为空");
        Validate.notBlank(prizeDto.getPrizeType(),"奖品类型不能为空");
        Validate.notBlank(prizeDto.getPrizeName(),"奖品名称不能为空");
//        Validate.notBlank(prizeDto.getUnitCode(),"单位不能为空");
        if(!PrizeTypeEnum.THANKS.getCode().equals(prizeDto.getPrizeType())) {
            Validate.notNull(prizeDto.getInventory(), "是否库存不能为空");
            if(PrizeInventoryEnum.EXITS.getCode().equals(prizeDto.getInventory())) {
                Validate.notNull(prizeDto.getInventoryNum(),"库存数量不能为空");
            }
        }
        Validate.notBlank(prizeDto.getId(),"编辑时主键id不能为空");
        Validate.notBlank(prizeDto.getCode(),"编辑时,编码不能为空");

    }

    @Override
    public void enableBatch(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.prizeRepository.lambdaUpdate().set(PrizeEntity::getEnableStatus, EnableStatusEnum.ENABLE.getCode())
                .in(PrizeEntity::getId,ids).update();
    }

    @Override
    public void disableBatch(List<String> ids) {
        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.prizeRepository.lambdaUpdate().set(PrizeEntity::getEnableStatus, EnableStatusEnum.DISABLE.getCode())
                .in(PrizeEntity::getId,ids).update();
    }

    @Override
    public void deleteBatch(List<String> ids) {

        if(CollectionUtils.isEmpty(ids)) {
            return;
        }
        this.prizeRepository.lambdaUpdate().set(PrizeEntity::getDelFlag, DelFlagStatusEnum.DELETE.getCode())
                .in(PrizeEntity::getId,ids).update();

    }

    @Override
    public List<PrizeEntity> findByCodes(List<String> codes) {

        if(CollectionUtils.isEmpty(codes)) {
            return null;
        }
        return this.prizeRepository.lambdaQuery().eq(PrizeEntity::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
                .in(PrizeEntity::getCode,codes).list();
    }

    @Override
    public PrizeEntity findByCode(String code) {

        if(StringUtils.isBlank(code)) {
            return null;
        }
        return this.prizeRepository.lambdaQuery().eq(PrizeEntity::getDelFlag,DelFlagStatusEnum.NORMAL.getCode())
                .eq(PrizeEntity::getCode,code).one();
    }

    @Override
    public void adjust(String id, BigDecimal number) {

        if(StringUtils.isBlank(id) || Objects.isNull(number)) {
            return;
        }

        PrizeEntity prizeEntity = this.prizeRepository.getById(id);
        Validate.notNull(prizeEntity,"奖品不存在");
        //TODO:枷锁
        String redisKey = String.format(PrizeRedisKeys.TENANT_PRIZE_LOCK_PREFIX, TenantUtils.getTenantCode()+ prizeEntity.getCode());
        boolean locked = false;
        try {
            locked = redisMutexService.tryLock(redisKey, TimeUnit.SECONDS, 5);
            PrizeEntity entity = this.prizeRepository.getById(id);
            BigDecimal inventoryNum = entity.getInventoryNum();
            Validate.isTrue(inventoryNum.compareTo(number.negate()) > 0,"调整数量超过库存数量");
            entity.setInventoryNum(inventoryNum.add(number));
            this.prizeRepository.saveOrUpdate(entity);
        }finally {
            if (locked) {
                redisMutexService.unlock(redisKey);
            }
        }
  }

    /**
     * key 奖品编码
     * value 数量
     * @param putMap
     */
    @Override
    public void updateInventory(Map<String, BigDecimal> putMap) {
        this.prizeRepository.updateInventory(putMap);
    }

    @Override
    public void save(PrizeEntity prizeEntity) {
        this.prizeRepository.saveOrUpdate(prizeEntity);
    }
}
